<?php
/**
 * Script para verificar el funcionamiento del actualizador de stock
 * 
 * Este script revisa los logs de actualización y muestra un resumen
 * del funcionamiento del sistema de sincronización con Softland
 */

// Configuración
$logs_directory = __DIR__ . '/logs';
$dias_a_revisar = 3; // Número de días de logs a revisar

// Verificar si el directorio de logs existe
if (!is_dir($logs_directory)) {
    echo "Error: El directorio de logs no existe. El script podría no estar ejecutándose.\n";
    exit(1);
}

// Obtener archivos de log recientes
$archivos_log = [];
for ($i = 0; $i < $dias_a_revisar; $i++) {
    $fecha = date('Y-m-d', strtotime("-{$i} days"));
    $archivo = $logs_directory . '/actualizacion_stock_' . $fecha . '.log';
    if (file_exists($archivo)) {
        $archivos_log[$fecha] = $archivo;
    }
}

// Verificar si hay archivos de log
if (empty($archivos_log)) {
    echo "No se encontraron archivos de log para los últimos {$dias_a_revisar} días.\n";
    echo "El script podría no estar ejecutándose o podría estar configurado con otra ruta de logs.\n";
    exit(1);
}

echo "=== RESUMEN DE FUNCIONAMIENTO DEL ACTUALIZADOR DE STOCK ===\n\n";

// Analizar cada archivo de log
foreach ($archivos_log as $fecha => $archivo) {
    $contenido = file_get_contents($archivo);
    $lineas = explode(PHP_EOL, $contenido);
    
    // Calcular métricas
    $total_ejecuciones = 0;
    $total_actualizados = 0;
    $total_errores = 0;
    $ultima_ejecucion = "";
    
    $inicio_pattern = '/Iniciando actualización de stock desde Softland/';
    $resumen_pattern = '/Resumen de actualización: (\d+) actualizados, (\d+) sin cambios, (\d+) errores/';
    
    foreach ($lineas as $linea) {
        // Detectar inicios de ejecución
        if (preg_match($inicio_pattern, $linea)) {
            $total_ejecuciones++;
            
            // Extraer la fecha/hora de la ejecución
            preg_match('/\[(.*?)\]/', $linea, $matches);
            if (isset($matches[1])) {
                $ultima_ejecucion = $matches[1];
            }
        }
        
        // Detectar resúmenes de actualización
        if (preg_match($resumen_pattern, $linea, $matches)) {
            $total_actualizados += (int) $matches[1];
            $total_errores += (int) $matches[3];
        }
    }
    
    // Mostrar resultados para esta fecha
    echo "Fecha: {$fecha}\n";
    echo "  - Total de ejecuciones: {$total_ejecuciones}\n";
    echo "  - Repuestos actualizados: {$total_actualizados}\n";
    echo "  - Errores encontrados: {$total_errores}\n";
    
    if (!empty($ultima_ejecucion)) {
        echo "  - Última ejecución: {$ultima_ejecucion}\n";
    } else {
        echo "  - No se detectó la hora de ejecución (formato de log posiblemente incorrecto)\n";
    }
    
    echo "\n";
}

echo "=== VERIFICACIÓN DE ESTADO ===\n\n";

// Comprobar si se ha ejecutado hoy
$hoy = date('Y-m-d');
if (isset($archivos_log[$hoy])) {
    // Obtener la última ejecución
    $contenido = file_get_contents($archivos_log[$hoy]);
    $lineas = explode(PHP_EOL, $contenido);
    $ultima_hora = "";
    
    foreach ($lineas as $linea) {
        if (preg_match($inicio_pattern, $linea)) {
            preg_match('/\[(.*?)\]/', $linea, $matches);
            if (isset($matches[1])) {
                $ultima_hora = $matches[1];
            }
        }
    }
    
    if (!empty($ultima_hora)) {
        $ultima_timestamp = strtotime($ultima_hora);
        $tiempo_transcurrido = time() - $ultima_timestamp;
        $horas_transcurridas = floor($tiempo_transcurrido / 3600);
        
        echo "La última ejecución fue hace {$horas_transcurridas} horas.\n";
        
        if ($horas_transcurridas > 2) {
            echo "ALERTA: Han pasado más de 2 horas desde la última ejecución. El CRON podría no estar funcionando correctamente.\n";
        } else {
            echo "El script parece estar funcionando correctamente según lo programado.\n";
        }
    }
} else {
    echo "ALERTA: No se encontró registro de ejecución para hoy ({$hoy}).\n";
    echo "El script podría no estar ejecutándose.\n";
}

// Sugerir prueba manual
echo "\nPara realizar una prueba manual, ejecute:\n";
echo "php /ruta/a/actualizar_stock.php\n\n";

echo "Para revisar los logs detallados, revise los archivos en:\n";
echo $logs_directory . "\n";
?>