<?php
/**
 * Gestión de Solicitudes de Repuestos
 * CRUD completo para solicitudes de repuestos con integración mejorada con series de máquinas
 * Versión: 2.0
 */

// Definir el título de la página
$page_title = 'Gestión de Solicitudes de Repuestos';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Determinar la acción a realizar
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Variable para control de aprobación automática
$auto_aprobar = false; // Por defecto, no se aprueban automáticamente

// Incluir el header antes de cualquier salida
require_once 'includes/header.php';

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Procesar formulario de creación o edición
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        // Datos generales de la solicitud
        $maquina_id = isset($_POST['maquina_id']) ? intval($_POST['maquina_id']) : 0;
        $serie_id = isset($_POST['serie_id']) ? intval($_POST['serie_id']) : null;
        $area_origen = $_POST['area_origen'] ?? 'servicio_tecnico';
        $motivo_solicitud = $_POST['motivo_solicitud'] ?? 'otro';
        $descripcion = trim($_POST['descripcion'] ?? '');
        $fecha = $_POST['fecha'] ?? date('Y-m-d');
        
        // Datos de repuestos
        $repuestos = $_POST['repuestos'] ?? [];
        $cantidades = $_POST['cantidades'] ?? [];
        $codigos_proveedor = $_POST['codigos_proveedor'] ?? [];
        
        // Validación básica
        $errores = [];
        
        if (empty($maquina_id)) {
            $errores[] = "Debe seleccionar una máquina";
        }
        
        if (empty($serie_id)) {
            $errores[] = "Debe seleccionar una serie de máquina";
        }
        
        if (empty($descripcion)) {
            $errores[] = "La descripción es obligatoria";
        }
        
        if (empty($repuestos)) {
            $errores[] = "Debe seleccionar al menos un repuesto";
        }

        // Validación adicional de datos
        if (!empty($repuestos)) {
            foreach ($repuestos as $key => $repuesto_id) {
                if (empty($repuesto_id)) continue;
                
                $cantidad = isset($cantidades[$key]) ? intval($cantidades[$key]) : 0;
                if ($cantidad <= 0) {
                    $errores[] = "La cantidad debe ser mayor a 0 para todos los repuestos";
                    break;
                }

                // Verificar stock disponible
                $stmt = $conn->prepare("SELECT stock, nombre FROM repuestos WHERE id = :id AND activo = 1");
                $stmt->bindParam(':id', $repuesto_id);
                $stmt->execute();
                $repuesto = $stmt->fetch(PDO::FETCH_ASSOC);

                if (!$repuesto) {
                    $errores[] = "Uno de los repuestos seleccionados no está disponible o ha sido desactivado";
                    break;
                }

                if ($repuesto['stock'] < $cantidad) {
                    $errores[] = "Stock insuficiente para el repuesto: " . $repuesto['nombre'] . 
                                " (Disponible: " . $repuesto['stock'] . ", Solicitado: " . $cantidad . ")";
                }
            }
        }

        // Verificar que la máquina esté activa
        if ($maquina_id) {
            $stmt = $conn->prepare("SELECT activo FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $maquina_id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$maquina || !$maquina['activo']) {
                $errores[] = "La máquina seleccionada no está disponible o ha sido desactivada";
            }
        }

        // Verificar que la serie esté activa
        if ($serie_id) {
            $stmt = $conn->prepare("SELECT activo FROM series_maquinas WHERE id = :id");
            $stmt->bindParam(':id', $serie_id);
            $stmt->execute();
            $serie = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$serie || !$serie['activo']) {
                $errores[] = "La serie seleccionada no está disponible o ha sido desactivada";
            }
        }

        // Validar fecha
        if (!empty($fecha)) {
            $fecha_obj = DateTime::createFromFormat('Y-m-d', $fecha);
            if (!$fecha_obj || $fecha_obj->format('Y-m-d') !== $fecha) {
                $errores[] = "El formato de la fecha no es válido";
            }
        }
        
        // Si no hay errores, proceder a guardar
        if (empty($errores)) {
            $conn->beginTransaction();
            
            try {
                if ($action === 'edit') {
                    // Actualizar solicitud existente
                    $stmt = $conn->prepare("
                        UPDATE solicitud_repuestos 
                        SET maquina_id = :maquina_id, 
                            serie_id = :serie_id, 
                            descripcion = :descripcion, 
                            fecha = :fecha,
                            area_origen = :area_origen,
                            motivo_solicitud = :motivo_solicitud,
                            cliente_id = :cliente_id,
                            ultima_modificacion = NOW()
                        WHERE id = :id
                    ");
                    
                    $stmt->bindParam(':id', $id);
                    $stmt->bindParam(':maquina_id', $maquina_id);
                    $stmt->bindParam(':serie_id', $serie_id);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':area_origen', $area_origen);
                    $stmt->bindParam(':motivo_solicitud', $motivo_solicitud);
                    
                    // Procesar cliente_id
                    $cliente_id = null;
                    if (isset($_POST['asociar_cliente']) && isset($_POST['cliente_id']) && !empty($_POST['cliente_id'])) {
                        $cliente_id = intval($_POST['cliente_id']);
                    }
                    $stmt->bindParam(':cliente_id', $cliente_id, PDO::PARAM_INT);
                    
                    $stmt->execute();
                    
                    // Eliminar detalles anteriores
                    $stmt = $conn->prepare("DELETE FROM solicitud_repuesto_detalle WHERE solicitud_id = :solicitud_id");
                    $stmt->bindParam(':solicitud_id', $id);
                    $stmt->execute();
                    
                    // Registrar actividad con detalles
                    $detalles_log = [];
                    foreach ($repuestos as $key => $repuesto_id) {
                        if (empty($repuesto_id)) continue;
                        $cantidad = isset($cantidades[$key]) ? intval($cantidades[$key]) : 1;
                        $detalles_log[] = "Repuesto ID: $repuesto_id, Cantidad: $cantidad";
                    }
                    
                    registrarActividad(
                        $conn, 
                        $user_id, 
                        'editar', 
                        'Edición de solicitud de repuestos. Detalles: ' . implode('; ', $detalles_log), 
                        $id, 
                        'solicitud_repuesto'
                    );
                    
                    $mensaje = "Solicitud actualizada correctamente";
                    $solicitud_id = $id;
                } else {
                    // Crear nueva solicitud con número de seguimiento
                    $numero_seguimiento = generarNumeroSeguimiento($conn);
                    
                    $stmt = $conn->prepare("
                        INSERT INTO solicitud_repuestos 
                        (user_id, maquina_id, serie_id, descripcion, fecha, estado_id, 
                         fecha_creacion, area_origen, motivo_solicitud,
                         numero_seguimiento, cliente_id, ultima_modificacion) 
                        VALUES 
                        (:user_id, :maquina_id, :serie_id, :descripcion, :fecha, 1, 
                         NOW(), :area_origen, :motivo_solicitud,
                         :numero_seguimiento, :cliente_id, NOW())
                    ");
                    
                    $stmt->bindParam(':user_id', $user_id);
                    $stmt->bindParam(':maquina_id', $maquina_id);
                    $stmt->bindParam(':serie_id', $serie_id);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':area_origen', $area_origen);
                    $stmt->bindParam(':motivo_solicitud', $motivo_solicitud);
                    $stmt->bindParam(':numero_seguimiento', $numero_seguimiento);
                    
                    // Procesar cliente_id
                    $cliente_id = null;
                    if (isset($_POST['asociar_cliente']) && isset($_POST['cliente_id']) && !empty($_POST['cliente_id'])) {
                        $cliente_id = intval($_POST['cliente_id']);
                    }
                    $stmt->bindParam(':cliente_id', $cliente_id, PDO::PARAM_INT);
                    
                    $stmt->execute();
                    
                    $solicitud_id = $conn->lastInsertId();
                    
                    // Registrar actividad con detalles
                    $detalles_log = ["Número de seguimiento: $numero_seguimiento"];
                    foreach ($repuestos as $key => $repuesto_id) {
                        if (empty($repuesto_id)) continue;
                        $cantidad = isset($cantidades[$key]) ? intval($cantidades[$key]) : 1;
                        $detalles_log[] = "Repuesto ID: $repuesto_id, Cantidad: $cantidad";
                    }
                    
                    registrarActividad(
                        $conn, 
                        $user_id, 
                        'crear', 
                        'Creación de solicitud de repuestos. Detalles: ' . implode('; ', $detalles_log), 
                        $solicitud_id, 
                        'solicitud_repuesto'
                    );
                    
                    // Notificar al administrador
                    if (function_exists('notificarNuevaSolicitud')) {
                        notificarNuevaSolicitud($conn, $solicitud_id);
                    }
                    
                    $mensaje = "Solicitud creada correctamente. Número de seguimiento: $numero_seguimiento";
                }
                
                // Guardar detalles de repuestos con control de stock
                if (!empty($repuestos)) {
                    $stmt = $conn->prepare("
                        INSERT INTO solicitud_repuesto_detalle 
                        (solicitud_id, repuesto_id, cantidad, codigo_proveedor) 
                        VALUES (:solicitud_id, :repuesto_id, :cantidad, :codigo_proveedor)
                    ");
                    
                    $stmt_update_stock = $conn->prepare("
                        UPDATE repuestos 
                        SET stock = stock - :cantidad
                        WHERE id = :repuesto_id
                    ");
                    
                    foreach ($repuestos as $key => $repuesto_id) {
                        if (empty($repuesto_id)) continue;
                        
                        $cantidad = isset($cantidades[$key]) ? intval($cantidades[$key]) : 1;
                        $codigo_proveedor = isset($codigos_proveedor[$key]) ? trim($codigos_proveedor[$key]) : null;
                        
                        // Insertar detalle
                        $stmt->bindParam(':solicitud_id', $solicitud_id);
                        $stmt->bindParam(':repuesto_id', $repuesto_id);
                        $stmt->bindParam(':cantidad', $cantidad);
                        $stmt->bindParam(':codigo_proveedor', $codigo_proveedor);
                        $stmt->execute();
                        
                        // Actualizar stock si la solicitud es aprobada automáticamente
                        if ($auto_aprobar) {
                            $stmt_update_stock->bindParam(':cantidad', $cantidad);
                            $stmt_update_stock->bindParam(':repuesto_id', $repuesto_id);
                            $stmt_update_stock->execute();
                        }
                    }
                }
                
                $conn->commit();
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                
                // Redirigir después de procesar todo
                echo "<script>window.location.href = 'solicitud_repuestos.php';</script>";
                exit;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Aprobar o rechazar solicitud
    if ($action === 'approve' || $action === 'reject') {
        if (!$is_admin) {
            $_SESSION['flash_message'] = "No tiene permisos para realizar esta acción";
            $_SESSION['flash_type'] = 'danger';
            header('Location: solicitud_repuestos.php');
            exit;
        }
        
        $comentario = isset($_POST['comentario']) ? trim($_POST['comentario']) : '';
        $estado_id = ($action === 'approve') ? 3 : 4; // 3 = Aprobada, 4 = Rechazada
        
        $stmt = $conn->prepare("
            UPDATE solicitud_repuestos 
            SET estado_id = :estado_id, 
                revisor_id = :revisor_id,
                comentario_revision = :comentario,
                fecha_revision = NOW(),
                ultima_modificacion = NOW()
            WHERE id = :id
        ");
        
        $stmt->bindParam(':estado_id', $estado_id);
        $stmt->bindParam(':revisor_id', $user_id);
        $stmt->bindParam(':comentario', $comentario);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $accion = ($action === 'approve') ? 'aprobar' : 'rechazar';
        $mensaje = ($action === 'approve') ? 'Solicitud aprobada correctamente' : 'Solicitud rechazada correctamente';
        
        registrarActividad($conn, $user_id, $accion, ucfirst($accion) . ' solicitud de repuestos', $id, 'solicitud_repuesto');
        
        $_SESSION['flash_message'] = $mensaje;
        $_SESSION['flash_type'] = 'success';
        
        // Si es aprobada, redirigir para descargar el PDF
        if ($action === 'approve') {
            header('Location: generate_pdf_solicitud.php?id=' . $id);
        } else {
            header('Location: solicitud_repuestos.php');
        }
        exit;
    }
    
    // Eliminar solicitud
    if ($action === 'delete' && $id > 0) {
        try {
            $conn->beginTransaction();
            
            // Verificar que la solicitud exista
            $stmt = $conn->prepare("SELECT * FROM solicitud_repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $solicitud = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$solicitud) {
                throw new Exception("La solicitud no existe");
            }
            
            // Verificar si el usuario tiene permisos para eliminar
            if (!$is_admin && $solicitud['user_id'] != $user_id) {
                throw new Exception("No tiene permisos para eliminar esta solicitud");
            }
            
            // Verificar si la solicitud está en estado pendiente (1)
            if ($solicitud['estado_id'] != 1 && !$is_admin) {
                throw new Exception("Solo se pueden eliminar solicitudes en estado pendiente");
            }
            
            // Eliminar los detalles primero
            $stmt = $conn->prepare("DELETE FROM solicitud_repuesto_detalle WHERE solicitud_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Eliminar la solicitud
            $stmt = $conn->prepare("DELETE FROM solicitud_repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de solicitud de repuestos', $id, 'solicitud_repuesto');
            
            $conn->commit();
            
            $_SESSION['flash_message'] = "Solicitud eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: solicitud_repuestos.php');
            exit;
        } catch (Exception $e) {
            $conn->rollBack();
            $error_message = $e->getMessage();
        }
    }

} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}
?>

<style>
    /* Estilos de tabla unificados con ordenes_compra.php */
    .solicitudes-table td, .solicitudes-table th {
        padding: 8px 10px;
    }
    
    .solicitudes-table tr:hover {
        background-color: rgba(0,0,0,0.03);
    }
    
    /* Iconos de acción - Mismo estilo que en ordenes_compra.php */
    .action-icon {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 28px;
        height: 28px;
        border-radius: 50%;
        margin-right: 5px;
        transition: all 0.2s;
        color: #555 !important;
        background-color: #f0f0f0;
    }
    
    .action-icon:hover {
        background-color: #e0e0e0;
        transform: scale(1.1);
    }
    
    .action-icon.view-icon:hover {
        color: #1e88e5 !important;
    }
    
    .action-icon.pdf-icon:hover {
        color: #66bb6a !important;
    }
    
    .action-icon.edit-icon:hover {
        color: #ffa726 !important;
    }
    
    .action-icon.delete-icon:hover {
        color: #ef5350 !important;
    }
    
    .action-icon i {
        font-size: 18px;
    }
    
    /* Estilos para el formulario mejorados */
    .form-control:focus {
        border-color: #1e88e5;
        box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
    }
    
    .form-control:required {
        border-left: 3px solid #1e88e5;
    }
    
    /* Inputs más compactos y consistentes */
    .input-sm {
        padding: 8px 10px;
        height: auto;
    }
    
    /* Estilos para los repuestos seleccionados */
    .repuesto-item {
        display: flex;
        align-items: center;
        padding: 10px;
        border: 1px solid #ddd;
        border-radius: 4px;
        margin-bottom: 10px;
        background-color: #f9f9f9;
    }
    
    .repuesto-info {
        flex: 1;
    }
    
    .repuesto-actions {
        margin-left: 10px;
    }
    
    /* Estado de solicitudes - Estilos mejorados */
    .status-badge {
        display: inline-block;
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 0.8em;
        font-weight: bold;
        color: white;
    }
    
    /* Colores de estado basados en ordenes_compra.php */
    .badge-estado-1 {
        background-color: #FFC107; /* Pendiente/Enviada */
    }
    
    .badge-estado-2 {
        background-color: #3498DB; /* En revisión */
    }
    
    .badge-estado-3 {
        background-color: #28A745; /* Aprobada */
    }
    
    .badge-estado-4 {
        background-color: #DC3545; /* Rechazada */
    }
    
    .badge-estado-5 {
        background-color: #6C757D; /* Pagada/Cerrada */
    }
    
    /* Selector de series */
    .serie-selector {
        margin-top: 10px;
    }
    
    /* Info del cliente */
    .cliente-info {
        margin-top: 15px;
        padding: 15px;
        background-color: #f9f9f9;
        border-radius: 4px;
        border: 1px solid #ddd;
    }
    
    /* Tabla de repuestos seleccionables */
    .repuestos-table {
        width: 100%;
        margin-top: 15px;
    }
    
    .repuestos-table th, .repuestos-table td {
        padding: 8px 10px;
        text-align: left;
    }
    
    .repuestos-table thead {
        background-color: #f5f5f5;
    }
    
    .repuestos-table tbody tr:hover {
        background-color: #f0f0f0;
        cursor: pointer;
    }
    
    .selectable-row {
        cursor: pointer;
    }
    
    .selected-row {
        background-color: #e3f2fd !important;
    }
    
    /* Panel header - Estilo consistente con ordenes_compra.php */
    .panel-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 12px 15px;
        border-bottom: 1px solid #ddd;
        background-color: #f8f9fa;
    }
    
    .panel-title {
        margin: 0;
        font-size: 1.25rem;
        font-weight: 500;
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<?php
// Mostrar contenido según la acción
switch ($action) {
    case 'new':
    case 'edit':
        // Formulario para crear o editar solicitud
        $solicitud = null;
        $detalles = [];
        
        if ($action === 'edit' && $id > 0) {
            // Cargar datos de la solicitud para edición
            $stmt = $conn->prepare("SELECT * FROM solicitud_repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $solicitud = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$solicitud) {
                echo '<div class="alert alert-danger">Solicitud no encontrada</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="solicitud_repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Verificar permisos para editar
            if (!$is_admin && $solicitud['user_id'] != $user_id) {
                echo '<div class="alert alert-danger">No tiene permisos para editar esta solicitud</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="solicitud_repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Verificar estado para editar
            if ($solicitud['estado_id'] != 1 && !$is_admin) {
                echo '<div class="alert alert-danger">Solo se pueden editar solicitudes en estado pendiente</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="solicitud_repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Obtener detalles de repuestos
            $stmt = $conn->prepare("
                SELECT d.*, r.codigo, r.nombre
                FROM solicitud_repuesto_detalle d
                JOIN repuestos r ON d.repuesto_id = r.id
                WHERE d.solicitud_id = :solicitud_id
            ");
            $stmt->bindParam(':solicitud_id', $id);
            $stmt->execute();
            $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        // Obtener lista de máquinas activas
        $stmt = $conn->prepare("
            SELECT id, codigo, nombre, marca, modelo
            FROM maquinas 
            WHERE activo = 1
            ORDER BY codigo
        ");
        $stmt->execute();
        $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Obtener lista de series para la máquina seleccionada
        $series = [];
        if (isset($solicitud['maquina_id']) && $solicitud['maquina_id'] > 0) {
            $stmt = $conn->prepare("
                SELECT s.*, p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM series_maquinas s
                LEFT JOIN proveedores p ON s.proveedor_id = p.id
                WHERE s.maquina_id = :maquina_id AND s.activo = 1
                ORDER BY s.codigo_serie
            ");
            $stmt->bindParam(':maquina_id', $solicitud['maquina_id']);
            $stmt->execute();
            $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        // Obtener cliente asociado a la serie seleccionada
        $cliente = null;
        if (isset($solicitud['serie_id']) && $solicitud['serie_id'] > 0) {
            $stmt = $conn->prepare("
                SELECT p.id, p.nombre, p.rut
                FROM series_maquinas s
                JOIN proveedores p ON s.proveedor_id = p.id
                WHERE s.id = :serie_id
            ");
            $stmt->bindParam(':serie_id', $solicitud['serie_id']);
            $stmt->execute();
            $cliente = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        $titulo = ($action === 'new') ? 'Crear Nueva Solicitud de Repuestos' : 'Editar Solicitud de Repuestos';
        ?>
        
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><?php echo $titulo; ?></h2>
                <a href="solicitud_repuestos.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="solicitud_repuestos.php?action=<?php echo $action . ($action === 'edit' ? '&id=' . $id : ''); ?>">
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <h3>Información General</h3>
                        
                        <div class="form-group">
                            <label for="fecha" class="form-label">Fecha <span class="text-danger">*</span></label>
                            <input type="date" id="fecha" name="fecha" class="form-control input-sm"
                                  value="<?php echo isset($solicitud['fecha']) ? $solicitud['fecha'] : date('Y-m-d'); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="area_origen" class="form-label">Área de Origen <span class="text-danger">*</span></label>
                            <select id="area_origen" name="area_origen" class="form-control input-sm" required>
                                <option value="servicio_tecnico" <?php echo (isset($solicitud['area_origen']) && $solicitud['area_origen'] == 'servicio_tecnico') ? 'selected' : ''; ?>>Servicio Técnico</option>
                                <option value="bodega" <?php echo (isset($solicitud['area_origen']) && $solicitud['area_origen'] == 'bodega') ? 'selected' : ''; ?>>Bodega</option>
                                <option value="ventas" <?php echo (isset($solicitud['area_origen']) && $solicitud['area_origen'] == 'ventas') ? 'selected' : ''; ?>>Ventas</option>
                                <option value="otra" <?php echo (isset($solicitud['area_origen']) && $solicitud['area_origen'] == 'otra') ? 'selected' : ''; ?>>Otra</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="motivo_solicitud" class="form-label">Motivo de Solicitud <span class="text-danger">*</span></label>
                            <select id="motivo_solicitud" name="motivo_solicitud" class="form-control input-sm" required>
                                <option value="maquinas_comodato" <?php echo (isset($solicitud['motivo_solicitud']) && $solicitud['motivo_solicitud'] == 'maquinas_comodato') ? 'selected' : ''; ?>>Máquinas en Comodato</option>
                                <option value="presupuestos" <?php echo (isset($solicitud['motivo_solicitud']) && $solicitud['motivo_solicitud'] == 'presupuestos') ? 'selected' : ''; ?>>Presupuestos</option>
                                <option value="ordenes_compra" <?php echo (isset($solicitud['motivo_solicitud']) && $solicitud['motivo_solicitud'] == 'ordenes_compra') ? 'selected' : ''; ?>>Órdenes de Compra</option>
                                <option value="rebaja_inventario" <?php echo (isset($solicitud['motivo_solicitud']) && $solicitud['motivo_solicitud'] == 'rebaja_inventario') ? 'selected' : ''; ?>>Rebaja de Inventario</option>
                                <option value="otro" <?php echo (isset($solicitud['motivo_solicitud']) && $solicitud['motivo_solicitud'] == 'otro') ? 'selected' : ''; ?>>Otro</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="asociar_cliente" name="asociar_cliente" 
                                       <?php echo (isset($solicitud['cliente_id']) && $solicitud['cliente_id'] > 0) ? 'checked' : ''; ?>
                                       onchange="toggleClienteSection()">
                                <label class="custom-control-label" for="asociar_cliente">¿Desea asociar esta solicitud a un cliente?</label>
                            </div>
                        </div>

                        <div id="seccion_cliente" style="display: none;">
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h4>Información del Cliente</h4>
                                    <div class="form-group">
                                        <label for="cliente_id">Seleccione el Cliente:</label>
                                        <select name="cliente_id" id="cliente_id" class="form-control input-sm">
                                            <option value="">Seleccione un cliente</option>
                                            <?php
                                            // Obtener lista de clientes
                                            $stmt = $conn->prepare("SELECT id, nombre, rut FROM proveedores WHERE activo = 1 ORDER BY nombre");
                                            $stmt->execute();
                                            $clientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                            foreach ($clientes as $cliente): ?>
                                                <option value="<?php echo $cliente['id']; ?>" 
                                                        <?php echo (isset($solicitud['cliente_id']) && $solicitud['cliente_id'] == $cliente['id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($cliente['nombre'] . ' - ' . $cliente['rut']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div id="cliente_detalles" class="alert alert-info" style="display: none;">
                                        <!-- Los detalles del cliente se mostrarán aquí -->
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="descripcion" class="form-label">Descripción <span class="text-danger">*</span></label>
                            <textarea id="descripcion" name="descripcion" class="form-control" rows="4" required><?php echo isset($solicitud['descripcion']) ? htmlspecialchars($solicitud['descripcion']) : ''; ?></textarea>
                        </div>
                        
                        <h3>Máquina y Serie</h3>
                        
                        <div class="form-group">
                            <label for="maquina_id">Máquina:</label>
                            <select name="maquina_id" id="maquina_id" class="form-control input-sm" required>
                                <option value="">Seleccione una máquina</option>
                                <?php foreach ($maquinas as $maquina): ?>
                                    <option value="<?php echo $maquina['id']; ?>" <?php echo (isset($solicitud['maquina_id']) && $solicitud['maquina_id'] == $maquina['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($maquina['codigo'] . ' - ' . $maquina['nombre']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div id="repuestos_disponibles" class="mb-3">
                            <!-- Aquí se mostrarán los repuestos disponibles para la máquina seleccionada -->
                        </div>

                        <div class="form-group">
                            <label for="serie_id">Serie:</label>
                            <select id="serie_id" name="serie_id" class="form-control input-sm" required onchange="verificarCliente(this.value)">
                                <option value="">Seleccione una serie</option>
                                <?php foreach ($series as $serie): ?>
                                <option value="<?php echo $serie['id']; ?>" <?php echo (isset($solicitud['serie_id']) && $solicitud['serie_id'] == $serie['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($serie['codigo_serie']); ?>
                                    <?php if (!empty($serie['estado']) && $serie['estado'] != 'activo'): ?>
                                    (<?php echo ucfirst($serie['estado']); ?>)
                                    <?php endif; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div id="cliente_info" class="cliente-info" style="display: none;">
                            <?php if (isset($solicitud) && isset($cliente) && $cliente): ?>
                            <strong>Cliente asociado:</strong> <?php echo htmlspecialchars($cliente['nombre']); ?> (RUT: <?php echo htmlspecialchars($cliente['rut']); ?>)
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card">
                            <div class="panel-header">
                                <h3 class="panel-title">Repuestos</h3>
                                <button type="button" id="btn-agregar-producto" class="btn btn-success btn-sm">
                                    <i class="material-icons">add</i> Agregar Repuesto
                                </button>
                            </div>
                            
                            <div class="form-group" style="padding: 15px;">
                                <label for="buscar_repuesto" class="form-label">Buscar Repuesto</label>
                                <div class="input-group">
                                    <input type="text" id="buscar_repuesto" class="form-control input-sm" placeholder="Buscar por código o nombre">
                                    <div class="input-group-append">
                                        <button type="button" class="btn btn-primary" onclick="buscarRepuestos()">
                                            <i class="material-icons">search</i> Buscar
                                        </button>
                                    </div>
                                </div>
                            </div>
                        
                            <div id="resultados_repuestos" style="max-height: 300px; overflow-y: auto; margin: 0 15px;"></div>
                        
                            <hr>
                        
                            <h4 style="padding: 0 15px;">Repuestos Seleccionados</h4>
                            <div id="repuestos_seleccionados" style="padding: 0 15px 15px;">
                                <?php if (!empty($detalles)): ?>
                                    <?php foreach ($detalles as $index => $detalle): ?>
                                    <div class="repuesto-item" id="repuesto_item_<?php echo $index; ?>">
                                        <div class="repuesto-info">
                                            <input type="hidden" name="repuestos[]" value="<?php echo $detalle['repuesto_id']; ?>">
                                            <div><strong><?php echo htmlspecialchars($detalle['codigo']); ?></strong> - <?php echo htmlspecialchars($detalle['nombre']); ?></div>
                                            <div class="row" style="margin-top: 5px;">
                                                <div class="col-md-6">
                                                    <label>Cantidad:</label>
                                                    <input type="number" name="cantidades[]" class="form-control input-sm" value="<?php echo $detalle['cantidad']; ?>" min="1" required>
                                                </div>
                                                <div class="col-md-6">
                                                    <label>Código Proveedor:</label>
                                                    <input type="text" name="codigos_proveedor[]" class="form-control input-sm" value="<?php echo htmlspecialchars($detalle['codigo_proveedor'] ?? ''); ?>" placeholder="Opcional">
                                                </div>
                                            </div>
                                        </div>
                                        <div class="repuesto-actions">
                                            <button type="button" class="btn btn-danger btn-sm" onclick="quitarRepuesto(<?php echo $index; ?>)">
                                                <i class="material-icons">delete</i>
                                            </button>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> <?php echo ($action === 'new') ? 'Crear Solicitud' : 'Guardar Cambios'; ?>
                    </button>
                    <a href="solicitud_repuestos.php" class="btn btn-danger">
                        <i class="material-icons">cancel</i> Cancelar
                    </a>
                </div>
            </form>
        </div>
        
        <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
        <script>
            // Variable global para contar repuestos añadidos
            let repuestoCounter = <?php echo !empty($detalles) ? count($detalles) : 0; ?>;
            
            document.addEventListener('DOMContentLoaded', function() {
                if (typeof jQuery === 'undefined') {
                    console.error('jQuery no está cargado. Algunas funciones pueden no estar disponibles.');
                    return;
                }

                // Inicializar eventos
                inicializarEventos();
                
                // Cargar datos iniciales si hay una máquina seleccionada
                const maquinaId = $('#maquina_id').val();
                if (maquinaId) {
                    cargarSeries(maquinaId);
                    buscarRepuestos();
                }
                
                // Inicializar la sección de cliente
                toggleClienteSection();
                
                // Inicializar botón para agregar repuesto
                document.getElementById('btn-agregar-producto').addEventListener('click', function(e) {
                    e.preventDefault();
                    buscarRepuestos();
                });
            });

            // Función para cargar series de una máquina
            function cargarSeries(maquinaId) {
                if (!maquinaId) {
                    $('#serie_id').html('<option value="">Seleccione una serie</option>');
                    $('#cliente_info').hide();
                    return;
                }
                
                console.log('Cargando series para máquina:', maquinaId);
                
                // Mostrar indicador de carga
                $('#serie_id').html('<option value="">Cargando series...</option>');
                $('#cliente_info').hide();
                
                $.ajax({
                    url: 'get_series.php',
                    type: 'GET',
                    data: { maquina_id: maquinaId },
                    success: function(response) {
                        console.log('Respuesta de get_series.php:', response);
                        let seriesSelect = $('#serie_id');
                        seriesSelect.html('<option value="">Seleccione una serie</option>');
                        
                        if (response.success && response.series && response.series.length > 0) {
                            response.series.forEach(function(serie) {
                                let descripcionText = serie.descripcion ? ` - ${serie.descripcion}` : '';
                                seriesSelect.append(`<option value="${serie.id}" 
                                    data-cliente-nombre="${serie.cliente_nombre || ''}"
                                    data-cliente-rut="${serie.cliente_rut || ''}"
                                    data-tiene-cliente="${serie.cliente_id ? '1' : '0'}"
                                    data-descripcion="${serie.descripcion || ''}">${serie.codigo_serie}${descripcionText}</option>`);
                            });
                            console.log('Series cargadas correctamente');
                            
                            // Disparar evento change para verificar cliente si hay una serie seleccionada
                            const selectedSerieId = seriesSelect.val();
                            if (selectedSerieId) {
                                verificarCliente(selectedSerieId);
                            }
                        } else {
                            seriesSelect.html('<option value="">No hay series disponibles</option>');
                            console.log('No se encontraron series');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Error al cargar series:', error);
                        $('#serie_id').html('<option value="">Error al cargar series</option>');
                    }
                });
            }

            // Función para verificar cliente de una serie
            function verificarCliente(serieId) {
                // Limpiar información del cliente anterior y ocultarla
                $('#cliente_info').hide().html('');
                
                // Si no hay serie seleccionada, retornar
                if (!serieId) {
                    return;
                }

                const serieOption = $(`#serie_id option[value="${serieId}"]`);
                if (!serieOption.length) {
                    return;
                }

                // Si estamos en una nueva solicitud y no hay serie seleccionada, no mostrar nada
                if (window.location.href.includes('action=new') && !$('#serie_id').val()) {
                    return;
                }

                const clienteNombre = serieOption.data('cliente-nombre');
                const clienteRut = serieOption.data('cliente-rut');

                // Solo mostrar información si hay nombre o RUT del cliente
                if (clienteNombre || clienteRut) {
                    let clienteHtml = '<div class="alert alert-info mb-0">';
                    clienteHtml += '<strong>Cliente asociado a la serie:</strong><br>';
                    
                    if (clienteNombre) {
                        clienteHtml += `${clienteNombre}<br>`;
                    }
                    
                    if (clienteRut) {
                        clienteHtml += `<small>RUT: ${clienteRut}</small>`;
                    }
                    
                    clienteHtml += '</div>';
                    
                    $('#cliente_info').html(clienteHtml).show();
                }
            }

            // Función para manejar la visibilidad de la sección de cliente
            function toggleClienteSection() {
                const asociarCliente = $('#asociar_cliente').is(':checked');
                if (asociarCliente) {
                    $('#seccion_cliente').show();
                    $('#cliente_info').hide().html('');
                } else {
                    $('#seccion_cliente').hide();
                    const serieId = $('#serie_id').val();
                    if (serieId) {
                        verificarCliente(serieId);
                    }
                }
            }

            function inicializarEventos() {
                // Evento para cuando se selecciona una máquina
                $('#maquina_id').on('change', function() {
                    const maquinaId = $(this).val();
                    console.log('Máquina seleccionada:', maquinaId);
                    
                    // Limpiar búsqueda y resultados anteriores
                    $('#buscar_repuesto').val('');
                    $('#cliente_info').hide();
                    $('#repuestos_list').empty();
                    
                    if (maquinaId) {
                        cargarSeries(maquinaId);
                        buscarRepuestos();
                    } else {
                        $('#serie_id').html('<option value="">Seleccione una serie</option>');
                        $('#resultados_repuestos').html('<div class="alert alert-info">Seleccione una máquina para ver los repuestos disponibles</div>');
                    }
                });

                // Evento para cuando se selecciona una serie
                $('#serie_id').on('change', function() {
                    const serieId = $(this).val();
                    console.log('Serie seleccionada:', serieId);
                    verificarCliente(serieId);
                });

                // Evento para la búsqueda de repuestos
                $('#buscar_repuesto').on('input', function() {
                    buscarRepuestos();
                });
            }

            // Función para agregar un repuesto seleccionado
            function agregarRepuesto(id, codigo, nombre, stock) {
                console.log('Agregando repuesto:', { id, codigo, nombre, stock });
                
                // Verificar si el repuesto ya está seleccionado
                const repuestosSeleccionados = document.getElementsByName('repuestos[]');
                for (let i = 0; i < repuestosSeleccionados.length; i++) {
                    if (repuestosSeleccionados[i].value == id) {
                        alert('Este repuesto ya está seleccionado');
                        return;
                    }
                }
                
                const container = document.getElementById('repuestos_seleccionados');
                if (!container) {
                    console.error('No se encontró el contenedor de repuestos seleccionados');
                    return;
                }
                
                const html = `
                    <div class="repuesto-item" id="repuesto_item_${repuestoCounter}">
                        <div class="repuesto-info">
                            <input type="hidden" name="repuestos[]" value="${id}">
                            <div><strong>${codigo}</strong> - ${nombre}</div>
                            <div class="row" style="margin-top: 5px;">
                                <div class="col-md-6">
                                    <label>Cantidad:</label>
                                    <input type="number" name="cantidades[]" class="form-control input-sm" value="1" min="1" max="${stock}" required>
                                </div>
                                <div class="col-md-6">
                                    <label>Código Proveedor:</label>
                                    <input type="text" name="codigos_proveedor[]" class="form-control input-sm" placeholder="Opcional">
                                </div>
                            </div>
                        </div>
                        <div class="repuesto-actions">
                            <button type="button" class="btn btn-danger btn-sm" onclick="quitarRepuesto(${repuestoCounter})">
                                <i class="material-icons">delete</i>
                            </button>
                        </div>
                    </div>
                `;
                
                container.insertAdjacentHTML('beforeend', html);
                console.log('Repuesto agregado correctamente');
                repuestoCounter++;
            }
            
            // Función para quitar un repuesto seleccionado
            function quitarRepuesto(index) {
                const element = document.getElementById('repuesto_item_' + index);
                if (element) {
                    element.remove();
                }
            }

            // Función para buscar repuestos
            function buscarRepuestos() {
                const query = $('#buscar_repuesto').val() || '';
                const maquina_id = $('#maquina_id').val() || '';
                
                if (!maquina_id) {
                    $('#resultados_repuestos').html('<div class="alert alert-info">Seleccione una máquina para ver los repuestos disponibles</div>');
                    return;
                }
                
                // Mostrar indicador de carga
                $('#resultados_repuestos').html('<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Cargando repuestos...</div>');
                
                $.ajax({
                    url: 'buscar_repuestos.php',
                    type: 'GET',
                    data: {
                        query: query,
                        maquina_id: maquina_id
                    },
                    success: function(response) {
                        if (response.success) {
                            let html = '';
                            if (response.repuestos && response.repuestos.length > 0) {
                                html = '<div class="list-group">';
                                response.repuestos.forEach(function(repuesto) {
                                    const badgeClass = repuesto.stock > 0 ? 'badge-success' : 'badge-danger';
                                    const stockText = repuesto.stock > 0 ? `Stock: ${repuesto.stock}` : 'Sin stock';
                                    const ubicacion = repuesto.ubicacion ? `<br><small>Ubicación: ${repuesto.ubicacion}</small>` : '';
                                    
                                    html += `
                                        <div class="list-group-item">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong>${repuesto.codigo}</strong> - ${repuesto.nombre}
                                                    ${ubicacion}
                                                </div>
                                                <div>
                                                    <span class="badge ${badgeClass}">${stockText}</span>
                                                    ${repuesto.stock > 0 ? 
                                                        `<button type="button" class="btn btn-sm btn-primary ml-2" 
                                                         onclick="agregarRepuesto('${repuesto.id}', '${repuesto.codigo}', '${repuesto.nombre}', ${repuesto.stock})">
                                                            Agregar
                                                        </button>` : 
                                                        ''
                                                    }
                                                </div>
                                            </div>
                                        </div>`;
                                });
                                html += '</div>';
                            } else {
                                html = '<div class="alert alert-info">No se encontraron repuestos para esta máquina</div>';
                            }
                            $('#resultados_repuestos').html(html);
                        } else {
                            $('#resultados_repuestos').html(
                                '<div class="alert alert-danger">Error al buscar repuestos: ' + 
                                (response.message || 'Error desconocido') + '</div>'
                            );
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Error:', { xhr, status, error });
                        $('#resultados_repuestos').html(
                            '<div class="alert alert-danger">Error de conexión al buscar repuestos</div>'
                        );
                    }
                });
            }
        </script>
        <?php
        break;
        
    case 'view':
        // Ver detalles de solicitud
        try {
            $stmt = $conn->prepare("
                SELECT sr.*, 
                       m.codigo as maquina_codigo, m.nombre as maquina_nombre,
                       m.marca as maquina_marca, m.modelo as maquina_modelo,
                       sm.codigo_serie,
                       u.nombre as usuario_nombre, u.apellido as usuario_apellido,
                       e.nombre as estado_nombre, e.color as estado_color,
                       rev.nombre as revisor_nombre, rev.apellido as revisor_apellido,
                       p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM solicitud_repuestos sr
                JOIN maquinas m ON sr.maquina_id = m.id
                LEFT JOIN series_maquinas sm ON sr.serie_id = sm.id
                JOIN users u ON sr.user_id = u.id
                JOIN estados e ON sr.estado_id = e.id
                LEFT JOIN users rev ON sr.revisor_id = rev.id
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                WHERE sr.id = :id
            ");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $solicitud = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$solicitud) {
                echo '<div class="alert alert-danger">Solicitud no encontrada</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="solicitud_repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Obtener detalles de repuestos
            $stmt = $conn->prepare("
                SELECT d.*, r.codigo, r.nombre, r.numero_pieza
                FROM solicitud_repuesto_detalle d
                JOIN repuestos r ON d.repuesto_id = r.id
                WHERE d.solicitud_id = :solicitud_id
            ");
            $stmt->bindParam(':solicitud_id', $id);
            $stmt->execute();
            $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener colores para el estado
            $colores_estado = [
                1 => '#FFC107', // Enviada
                2 => '#3498DB', // En revisión
                3 => '#28A745', // Aprobada
                4 => '#DC3545', // Rechazada
                5 => '#6C757D'  // Pagada
            ];
            $color_estado = isset($colores_estado[$solicitud['estado_id']]) ? $colores_estado[$solicitud['estado_id']] : '#6C757D';
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">
                        Solicitud de Repuestos #<?php echo $id; ?>
                        <span class="status-badge badge-estado-<?php echo $solicitud['estado_id']; ?>">
                            <?php echo htmlspecialchars($solicitud['estado_nombre']); ?>
                        </span>
                    </h2>
                    <div>
                        <a href="solicitud_repuestos.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">arrow_back</i> Volver a la Lista
                        </a>
                        
                        <?php if ($solicitud['estado_id'] == 1 && ($is_admin || $solicitud['user_id'] == $user_id)): ?>
                        <a href="solicitud_repuestos.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                            <i class="material-icons">edit</i> Editar
                        </a>
                        <?php endif; ?>
                        
                        <?php if ($solicitud['estado_id'] == 3): ?>
                        <a href="generate_pdf_solicitud.php?id=<?php echo $id; ?>" class="btn btn-success btn-sm" target="_blank">
                            <i class="material-icons">picture_as_pdf</i> Ver PDF
                        </a>
                        <?php endif; ?>
                        
                        <?php if ($solicitud['estado_id'] == 1 && $is_admin): ?>
                        <button type="button" class="btn btn-success btn-sm" data-toggle="modal" data-target="#approveModal">
                            <i class="material-icons">check</i> Aprobar
                        </button>
                        
                        <button type="button" class="btn btn-danger btn-sm" data-toggle="modal" data-target="#rejectModal">
                            <i class="material-icons">close</i> Rechazar
                        </button>
                        <?php endif; ?>
                        
                        <?php if ($solicitud['estado_id'] == 1 && ($is_admin || $solicitud['user_id'] == $user_id)): ?>
                        <a href="solicitud_repuestos.php?action=delete&id=<?php echo $id; ?>" 
                           class="btn btn-danger btn-sm" 
                           onclick="return confirm('¿Está seguro de eliminar esta solicitud?')">
                            <i class="material-icons">delete</i> Eliminar
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card" style="height: 100%; margin-bottom: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Información General</h3>
                            </div>
                            <table>
                                <tr>
                                    <th style="width: 30%;">Fecha de Solicitud:</th>
                                    <td><?php echo date('d/m/Y', strtotime($solicitud['fecha'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Solicitante:</th>
                                    <td><?php echo htmlspecialchars($solicitud['usuario_nombre'] . ' ' . $solicitud['usuario_apellido']); ?></td>
                                </tr>
                                <tr>
                                    <th>Área de Origen:</th>
                                    <td><?php echo ucfirst(str_replace('_', ' ', $solicitud['area_origen'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Motivo de Solicitud:</th>
                                    <td><?php echo ucfirst(str_replace('_', ' ', $solicitud['motivo_solicitud'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Estado:</th>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $solicitud['estado_id']; ?>">
                                            <?php echo htmlspecialchars($solicitud['estado_nombre']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <?php if ($solicitud['estado_id'] != 1): ?>
                                <tr>
                                    <th>Revisado por:</th>
                                    <td>
                                        <?php 
                                        if ($solicitud['revisor_id']) {
                                            echo htmlspecialchars($solicitud['revisor_nombre'] . ' ' . $solicitud['revisor_apellido']);
                                        } else {
                                            echo 'Pendiente';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Fecha de Revisión:</th>
                                    <td>
                                        <?php 
                                        if ($solicitud['fecha_revision']) {
                                            echo date('d/m/Y H:i', strtotime($solicitud['fecha_revision']));
                                        } else {
                                            echo 'Pendiente';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <?php endif; ?>
                                <tr>
                                    <th>Fecha de Creación:</th>
                                    <td><?php echo date('d/m/Y H:i', strtotime($solicitud['fecha_creacion'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Última Modificación:</th>
                                    <td><?php echo date('d/m/Y H:i', strtotime($solicitud['ultima_modificacion'])); ?></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card" style="margin-bottom: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Máquina</h3>
                            </div>
                            <table>
                                <tr>
                                    <th style="width: 30%;">Código:</th>
                                    <td><?php echo htmlspecialchars($solicitud['maquina_codigo']); ?></td>
                                </tr>
                                <tr>
                                    <th>Nombre:</th>
                                    <td><?php echo htmlspecialchars($solicitud['maquina_nombre']); ?></td>
                                </tr>
                                <tr>
                                    <th>Marca:</th>
                                    <td><?php echo !empty($solicitud['maquina_marca']) ? htmlspecialchars($solicitud['maquina_marca']) : 'No especificada'; ?></td>
                                </tr>
                                <tr>
                                    <th>Modelo:</th>
                                    <td><?php echo !empty($solicitud['maquina_modelo']) ? htmlspecialchars($solicitud['maquina_modelo']) : 'No especificado'; ?></td>
                                </tr>
                                <tr>
                                    <th>Serie:</th>
                                    <td><?php echo !empty($solicitud['codigo_serie']) ? htmlspecialchars($solicitud['codigo_serie']) : 'No especificada'; ?></td>
                                </tr>
                                <?php if (!empty($solicitud['proveedor_nombre'])): ?>
                                <tr>
                                    <th>Cliente Asociado:</th>
                                    <td><?php echo htmlspecialchars($solicitud['proveedor_nombre']); ?></td>
                                </tr>
                                <tr>
                                    <th>RUT Cliente:</th>
                                    <td><?php echo htmlspecialchars($solicitud['proveedor_rut']); ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                        
                        <div class="card" style="margin-bottom: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Descripción</h3>
                            </div>
                            <div style="padding: 10px;">
                                <?php echo nl2br(htmlspecialchars($solicitud['descripcion'])); ?>
                            </div>
                        </div>
                        
                        <?php if (!empty($solicitud['comentario_revision'])): ?>
                        <div class="card" style="margin-bottom: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Comentarios de Revisión</h3>
                            </div>
                            <div style="padding: 10px;">
                                <?php echo nl2br(htmlspecialchars($solicitud['comentario_revision'])); ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Repuestos Solicitados</h3>
                    </div>
                    <div class="table-responsive">
                        <table class="solicitudes-table">
                            <thead>
                                <tr style="background-color: #f5f5f5;">
                                    <th style="width: 50px;">#</th>
                                    <th>Código</th>
                                    <th>Nombre</th>
                                    <th>Número de Pieza</th>
                                    <th>Cantidad</th>
                                    <th>Código Proveedor</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($detalles)): ?>
                                <tr>
                                    <td colspan="6" style="text-align: center;">No hay repuestos en esta solicitud</td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach ($detalles as $index => $detalle): ?>
                                    <tr>
                                        <td><?php echo $index + 1; ?></td>
                                        <td><?php echo htmlspecialchars($detalle['codigo']); ?></td>
                                        <td><?php echo htmlspecialchars($detalle['nombre']); ?></td>
                                        <td><?php echo !empty($detalle['numero_pieza']) ? htmlspecialchars($detalle['numero_pieza']) : 'N/A'; ?></td>
                                        <td><?php echo $detalle['cantidad']; ?></td>
                                        <td><?php echo !empty($detalle['codigo_proveedor']) ? htmlspecialchars($detalle['codigo_proveedor']) : 'N/A'; ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <!-- Modal para aprobar solicitud -->
            <div class="modal fade" id="approveModal" tabindex="-1" role="dialog" aria-labelledby="approveModalLabel" aria-hidden="true">
                <div class="modal-dialog" role="document">
                    <div class="modal-content">
                        <form method="POST" action="solicitud_repuestos.php?action=approve&id=<?php echo $id; ?>">
                            <div class="modal-header">
                                <h5 class="modal-title" id="approveModalLabel">Aprobar Solicitud</h5>
                                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">&times;</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <div class="form-group">
                                    <label for="comentario">Comentarios (opcional)</label>
                                    <textarea id="comentario" name="comentario" class="form-control" rows="3"></textarea>
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                                <button type="submit" class="btn btn-success">Aprobar</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <!-- Modal para rechazar solicitud -->
            <div class="modal fade" id="rejectModal" tabindex="-1" role="dialog" aria-labelledby="rejectModalLabel" aria-hidden="true">
                <div class="modal-dialog" role="document">
                    <div class="modal-content">
                        <form method="POST" action="solicitud_repuestos.php?action=reject&id=<?php echo $id; ?>">
                            <div class="modal-header">
                                <h5 class="modal-title" id="rejectModalLabel">Rechazar Solicitud</h5>
                                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">&times;</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <div class="form-group">
                                    <label for="comentario">Motivo del rechazo</label>
                                    <textarea id="comentario" name="comentario" class="form-control" rows="3" required></textarea>
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                                <button type="submit" class="btn btn-danger">Rechazar</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="solicitud_repuestos.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    default:
        // Lista de solicitudes
        try {
            // Filtros de búsqueda
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            $estado_id = isset($_GET['estado_id']) ? intval($_GET['estado_id']) : 0;
            $fecha_desde = isset($_GET['fecha_desde']) ? $_GET['fecha_desde'] : '';
            $fecha_hasta = isset($_GET['fecha_hasta']) ? $_GET['fecha_hasta'] : '';
            $maquina_id = isset($_GET['maquina_id']) ? intval($_GET['maquina_id']) : 0;
            
            // Construir la consulta
            $sql = "
                SELECT sr.*, 
                       m.codigo as maquina_codigo, m.nombre as maquina_nombre,
                       sm.codigo_serie,
                       u.nombre as usuario_nombre, u.apellido as usuario_apellido,
                       e.nombre as estado_nombre, e.color as estado_color,
                       p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM solicitud_repuestos sr
                JOIN maquinas m ON sr.maquina_id = m.id
                LEFT JOIN series_maquinas sm ON sr.serie_id = sm.id
                JOIN users u ON sr.user_id = u.id
                JOIN estados e ON sr.estado_id = e.id
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                WHERE 1=1
            ";
            
            $params = [];
            
            // Aplicar filtros
            if (!empty($busqueda)) {
                $sql .= " AND (m.codigo LIKE :busqueda OR m.nombre LIKE :busqueda OR sm.codigo_serie LIKE :busqueda OR p.nombre LIKE :busqueda OR p.rut LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            
            if ($estado_id > 0) {
                $sql .= " AND sr.estado_id = :estado_id";
                $params[':estado_id'] = $estado_id;
            }
            
            if (!empty($fecha_desde)) {
                $sql .= " AND sr.fecha >= :fecha_desde";
                $params[':fecha_desde'] = $fecha_desde;
            }
            
            if (!empty($fecha_hasta)) {
                $sql .= " AND sr.fecha <= :fecha_hasta";
                $params[':fecha_hasta'] = $fecha_hasta;
            }
            
            if ($maquina_id > 0) {
                $sql .= " AND sr.maquina_id = :maquina_id";
                $params[':maquina_id'] = $maquina_id;
            }
            
            // Si no es admin, solo ver sus propias solicitudes
            if (!$is_admin) {
                $sql .= " AND sr.user_id = :user_id";
                $params[':user_id'] = $user_id;
            }
            
            $sql .= " ORDER BY sr.fecha_creacion DESC";
            
            // Ejecutar la consulta
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            $stmt->execute();
            $solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener lista de estados para filtro
            $stmt = $conn->prepare("SELECT id, nombre FROM estados ORDER BY id");
            $stmt->execute();
            $estados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener lista de máquinas para filtro
            $stmt = $conn->prepare("
                SELECT id, codigo, nombre
                FROM maquinas 
                WHERE activo = 1
                ORDER BY codigo
            ");
            $stmt->execute();
            $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Gestión de Solicitudes de Repuestos</h2>
                    <a href="solicitud_repuestos.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nueva Solicitud
                    </a>
                </div>
                
                <!-- Filtros de búsqueda -->
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="solicitud_repuestos.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px; padding: 15px;">
                        <div style="flex: 1; min-width: 200px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control input-sm" 
                                   placeholder="Código máquina, número serie, cliente..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 150px;">
                            <label for="estado_id" class="form-label">Estado</label>
                            <select id="estado_id" name="estado_id" class="form-control input-sm">
                                <option value="0">Todos</option>
                                <?php foreach ($estados as $estado): ?>
                                <option value="<?php echo $estado['id']; ?>" <?php echo $estado_id == $estado['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($estado['nombre']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="flex: 1; min-width: 150px;">
                            <label for="fecha_desde" class="form-label">Fecha Desde</label>
                            <input type="date" id="fecha_desde" name="fecha_desde" class="form-control input-sm" value="<?php echo $fecha_desde; ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 150px;">
                            <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                            <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control input-sm" value="<?php echo $fecha_hasta; ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="maquina_id" class="form-label">Máquina</label>
                            <select id="maquina_id" name="maquina_id" class="form-control input-sm">
                                <option value="0">Todas</option>
                                <?php foreach ($maquinas as $maquina): ?>
                                <option value="<?php echo $maquina['id']; ?>" <?php echo $maquina_id == $maquina['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($maquina['codigo'] . ' - ' . $maquina['nombre']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="solicitud_repuestos.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <div class="table-responsive">
                    <table class="solicitudes-table">
                        <thead>
                            <tr style="background-color: #f5f5f5;">
                                <th>ID</th>
                                <th>Fecha</th>
                                <th>Máquina</th>
                                <th>Serie</th>
                                <th>Cliente</th>
                                <th>Solicitante</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($solicitudes)): ?>
                            <tr>
                                <td colspan="8" style="text-align: center;">No se encontraron solicitudes</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($solicitudes as $solicitud): ?>
                                <tr>
                                    <td><?php echo $solicitud['id']; ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($solicitud['fecha'])); ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($solicitud['maquina_codigo']); ?></strong><br>
                                        <small><?php echo htmlspecialchars($solicitud['maquina_nombre']); ?></small>
                                    </td>
                                    <td><?php echo !empty($solicitud['codigo_serie']) ? htmlspecialchars($solicitud['codigo_serie']) : 'N/A'; ?></td>
                                    <td>
                                        <?php if (!empty($solicitud['proveedor_nombre'])): ?>
                                            <?php echo htmlspecialchars($solicitud['proveedor_nombre']); ?><br>
                                            <small><?php echo htmlspecialchars($solicitud['proveedor_rut']); ?></small>
                                        <?php else: ?>
                                            No asociado
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($solicitud['usuario_nombre'] . ' ' . $solicitud['usuario_apellido']); ?></td>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $solicitud['estado_id']; ?>">
                                            <?php echo htmlspecialchars($solicitud['estado_nombre']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="solicitud_repuestos.php?action=view&id=<?php echo $solicitud['id']; ?>" 
                                           title="Ver detalle" class="action-icon view-icon">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                        
                                        <?php if ($solicitud['estado_id'] == 1 && ($is_admin || $solicitud['user_id'] == $user_id)): ?>
                                        <a href="solicitud_repuestos.php?action=edit&id=<?php echo $solicitud['id']; ?>" 
                                           title="Editar" class="action-icon edit-icon">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        
                                        <a href="solicitud_repuestos.php?action=delete&id=<?php echo $solicitud['id']; ?>" 
                                           title="Eliminar" class="action-icon delete-icon"
                                           onclick="return confirm('¿Está seguro de eliminar esta solicitud?')">
                                            <i class="material-icons">delete</i>
                                        </a>
                                        <?php endif; ?>
                                        
                                        <?php if ($solicitud['estado_id'] == 3): ?>
                                        <a href="generate_pdf_solicitud.php?id=<?php echo $solicitud['id']; ?>" 
                                           title="Descargar PDF" class="action-icon pdf-icon" target="_blank">
                                            <i class="material-icons">picture_as_pdf</i>
                                        </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar solicitudes: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>