<?php
/**
 * Gestión de máquinas
 * CRUD completo para máquinas con soporte para múltiples series
 * Integración con proveedores y estados
 * Versión: 2.1 (corregida para usar proveedores en lugar de clientes)
 */

$page_title = 'Gestión de Máquinas';
require_once 'includes/config.php';
require_once 'includes/functions.php';

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$serie_id = isset($_GET['serie_id']) ? intval($_GET['serie_id']) : 0;

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Procesar formulario de creación/edición de máquina
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $marca = trim($_POST['marca'] ?? '');
        $modelo = trim($_POST['modelo'] ?? '');
        $costo = !empty($_POST['costo']) ? floatval(str_replace(['$', '.', ','], '', $_POST['costo'])) : null;
        $activo = isset($_POST['activo']) ? 1 : 0;
        
        // Validación de datos
        $errores = [];
        if (empty($codigo)) $errores[] = "El código de la máquina es obligatorio";
        if (empty($nombre)) $errores[] = "El nombre de la máquina es obligatorio";
        
        // Verificar si el código ya existe
        if ($action === 'new' || ($action === 'edit' && $_POST['codigo_original'] !== $codigo)) {
            $stmt = $conn->prepare("SELECT id FROM maquinas WHERE codigo = :codigo");
            $stmt->bindParam(':codigo', $codigo);
            $stmt->execute();
            if ($stmt->rowCount() > 0) {
                $errores[] = "El código '{$codigo}' ya está en uso por otra máquina";
            }
        }
        
        if (empty($errores)) {
            try {
                $conn->beginTransaction();
                
                if ($action === 'edit' && $id > 0) {
                    $stmt = $conn->prepare("
                        UPDATE maquinas 
                        SET codigo = :codigo, nombre = :nombre, descripcion = :descripcion, 
                            marca = :marca, modelo = :modelo, costo = :costo, activo = :activo
                        WHERE id = :id
                    ");
                    $stmt->bindParam(':id', $id);
                    registrarActividad($conn, $user_id, 'editar', 'Edición de máquina', $id, 'maquina');
                    $mensaje = "Máquina actualizada correctamente";
                } else {
                    $stmt = $conn->prepare("
                        INSERT INTO maquinas 
                        (codigo, nombre, descripcion, marca, modelo, costo, fecha_registro, activo) 
                        VALUES (:codigo, :nombre, :descripcion, :marca, :modelo, :costo, NOW(), :activo)
                    ");
                    $mensaje = "Máquina creada correctamente";
                }
                
                $stmt->bindParam(':codigo', $codigo);
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':descripcion', $descripcion);
                $stmt->bindParam(':marca', $marca);
                $stmt->bindParam(':modelo', $modelo);
                $stmt->bindParam(':costo', $costo);
                $stmt->bindParam(':activo', $activo);
                $stmt->execute();
                
                if ($action === 'new') {
                    $id = $conn->lastInsertId();
                    registrarActividad($conn, $user_id, 'crear', 'Creación de máquina', $id, 'maquina');
                }
                
                // Procesar las series de la máquina
                $series = isset($_POST['series']) ? $_POST['series'] : [];
                $numeros_serie = $_POST['numero_serie'] ?? [];
                $estados_serie = $_POST['estado_serie'] ?? [];
                $proveedores_serie = $_POST['proveedor_id'] ?? [];
                $fechas_inicio = $_POST['fecha_inicio'] ?? [];
                $clientes_serie = $_POST['cliente_id'] ?? [];
                
                // Guardar nuevas series
                if (!empty($numeros_serie)) {
                    foreach ($numeros_serie as $key => $numero_serie) {
                        if (empty($numero_serie)) continue;
                        
                        $estado = $estados_serie[$key] ?? 'nueva';
                        $proveedor_id = !empty($proveedores_serie[$key]) ? $proveedores_serie[$key] : null;
                        $cliente_id = !empty($clientes_serie[$key]) ? $clientes_serie[$key] : null;
                        $fecha_inicio = !empty($fechas_inicio[$key]) ? $fechas_inicio[$key] : null;
                        $fecha_inicio_estado = date('Y-m-d H:i:s');
                        
                        // Verificar si la serie ya existe
                        $stmt = $conn->prepare("SELECT id FROM series_maquinas WHERE codigo_serie = :codigo_serie AND maquina_id != :maquina_id");
                        $stmt->bindParam(':codigo_serie', $numero_serie);
                        $stmt->bindParam(':maquina_id', $id);
                        $stmt->execute();
                        if ($stmt->rowCount() > 0) {
                            $errores[] = "El número de serie '{$numero_serie}' ya está asociado a otra máquina";
                            continue;
                        }
                        
                        // Preparar el historial inicial
                        $historial_inicial = json_encode([
                            [
                                'estado' => $estado,
                                'fecha' => $fecha_inicio_estado,
                                'usuario_id' => $user_id,
                                'cliente_id' => $cliente_id,
                                'proveedor_id' => $proveedor_id
                            ]
                        ]);
                        
                        $sql = "
                            INSERT INTO series_maquinas 
                            (maquina_id, codigo_serie, estado, proveedor_id, cliente_id, fecha_inicio, fecha_inicio_estado, historial_estados) 
                            VALUES (:maquina_id, :codigo_serie, :estado, :proveedor_id, :cliente_id, :fecha_inicio, :fecha_inicio_estado, :historial_estados)
                            ON DUPLICATE KEY UPDATE 
                            estado = VALUES(estado), 
                            proveedor_id = VALUES(proveedor_id),
                            cliente_id = VALUES(cliente_id),
                            fecha_inicio = VALUES(fecha_inicio),
                            fecha_inicio_estado = VALUES(fecha_inicio_estado),
                            historial_estados = VALUES(historial_estados)
                        ";
                        
                        $stmt = $conn->prepare($sql);
                        $stmt->bindParam(':maquina_id', $id, PDO::PARAM_INT);
                        $stmt->bindParam(':codigo_serie', $numero_serie);
                        $stmt->bindParam(':estado', $estado);
                        $stmt->bindParam(':proveedor_id', $proveedor_id, PDO::PARAM_INT);
                        $stmt->bindParam(':cliente_id', $cliente_id, PDO::PARAM_INT);
                        $stmt->bindParam(':fecha_inicio', $fecha_inicio);
                        $stmt->bindParam(':fecha_inicio_estado', $fecha_inicio_estado);
                        $stmt->bindParam(':historial_estados', $historial_inicial);
                        
                        try {
                            $stmt->execute();
                        } catch (PDOException $e) {
                            throw new Exception("Error en series_maquinas: " . $e->getMessage() . " - SQL: $sql");
                        }
                    }
                }
                
                // Eliminar series que se hayan quitado
                if ($action === 'edit' && !empty($series)) {
                    $series_a_mantener = array_filter($series, function($s) use ($numeros_serie) {
                        return in_array($s, $numeros_serie);
                    });
                    
                    if (!empty($series_a_mantener)) {
                        $placeholders = implode(',', array_fill(0, count($series_a_mantener), '?'));
                        $stmt = $conn->prepare("DELETE FROM series_maquinas WHERE maquina_id = ? AND codigo_serie NOT IN ($placeholders)");
                        $params = array_merge([$id], $series_a_mantener);
                        $stmt->execute($params);
                    }
                }
                
                // Subir imágenes
                if (isset($_FILES['imagenes']) && is_array($_FILES['imagenes']['name'])) {
                    $uploadDir = 'uploads/maquinas/';
                    if (!file_exists($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    for ($i = 0; $i < count($_FILES['imagenes']['name']); $i++) {
                        if ($_FILES['imagenes']['error'][$i] === 0) {
                            $tempFile = $_FILES['imagenes']['tmp_name'][$i];
                            $fileName = 'maq_' . uniqid() . '.jpg';
                            $targetFile = $uploadDir . $fileName;
                            
                            if (move_uploaded_file($tempFile, $targetFile)) {
                                $es_principal = (isset($_POST['imagen_principal']) && $_POST['imagen_principal'] == $i) ? 1 : 0;
                                
                                if ($es_principal) {
                                    $stmt = $conn->prepare("UPDATE maquina_imagenes SET es_principal = 0 WHERE maquina_id = :maquina_id");
                                    $stmt->bindParam(':maquina_id', $id);
                                    $stmt->execute();
                                }
                                
                                $stmt = $conn->prepare("
                                    INSERT INTO maquina_imagenes 
                                    (maquina_id, nombre_archivo, ruta_archivo, es_principal, fecha_subida) 
                                    VALUES (:maquina_id, :nombre_archivo, :ruta_archivo, :es_principal, NOW())
                                ");
                                
                                $stmt->bindParam(':maquina_id', $id);
                                $stmt->bindParam(':nombre_archivo', $fileName);
                                $stmt->bindParam(':ruta_archivo', $targetFile);
                                $stmt->bindParam(':es_principal', $es_principal);
                                $stmt->execute();
                            }
                        }
                    }
                }
                
                // Gestionar imagen principal existente
                if (isset($_POST['imagen_principal_existente']) && $_POST['imagen_principal_existente'] > 0) {
                    $imagen_id = $_POST['imagen_principal_existente'];
                    $stmt = $conn->prepare("UPDATE maquina_imagenes SET es_principal = 0 WHERE maquina_id = :maquina_id");
                    $stmt->bindParam(':maquina_id', $id);
                    $stmt->execute();
                    
                    $stmt = $conn->prepare("UPDATE maquina_imagenes SET es_principal = 1 WHERE id = :imagen_id AND maquina_id = :maquina_id");
                    $stmt->bindParam(':imagen_id', $imagen_id);
                    $stmt->bindParam(':maquina_id', $id);
                    $stmt->execute();
                }
                
                $conn->commit();
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: maquinas.php');
                exit;
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = "Error al guardar: " . $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Procesar formulario de gestión de series
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'manage_series') {
        try {
            $conn->beginTransaction();
            
            $serie_id = $_POST['serie_id'] ?? 0;
            $estado = $_POST['estado'] ?? 'nueva';
            $proveedor_id = !empty($_POST['proveedor_id']) ? $_POST['proveedor_id'] : null;
            $cliente_id = !empty($_POST['cliente_id']) ? $_POST['cliente_id'] : null;
            $fecha_inicio = !empty($_POST['fecha_inicio']) ? $_POST['fecha_inicio'] : null;
            $fecha_fin = !empty($_POST['fecha_fin']) ? $_POST['fecha_fin'] : null;
            $notas = $_POST['notas'] ?? '';
            
            // Obtener el historial actual
            $stmt = $conn->prepare("SELECT historial_estados FROM series_maquinas WHERE id = :serie_id");
            $stmt->bindParam(':serie_id', $serie_id);
            $stmt->execute();
            $historial_actual = $stmt->fetchColumn();
            
            // Preparar el nuevo registro de historial
            $nuevo_estado = [
                'estado' => $estado,
                'fecha' => date('Y-m-d H:i:s'),
                'usuario_id' => $user_id,
                'cliente_id' => $cliente_id,
                'proveedor_id' => $proveedor_id,
                'notas' => $notas
            ];
            
            // Actualizar el historial
            $historial = $historial_actual ? json_decode($historial_actual, true) : [];
            $historial[] = $nuevo_estado;
            
            $stmt = $conn->prepare("
                UPDATE series_maquinas 
                SET estado = :estado, 
                    proveedor_id = :proveedor_id,
                    cliente_id = :cliente_id,
                    fecha_inicio = :fecha_inicio,
                    fecha_fin = :fecha_fin,
                    fecha_inicio_estado = NOW(),
                    notas = :notas,
                    historial_estados = :historial_estados
                WHERE id = :serie_id
            ");
            
            $historial_json = json_encode($historial);
            
            $stmt->bindParam(':estado', $estado);
            $stmt->bindParam(':proveedor_id', $proveedor_id);
            $stmt->bindParam(':cliente_id', $cliente_id);
            $stmt->bindParam(':fecha_inicio', $fecha_inicio);
            $stmt->bindParam(':fecha_fin', $fecha_fin);
            $stmt->bindParam(':notas', $notas);
            $stmt->bindParam(':historial_estados', $historial_json);
            $stmt->bindParam(':serie_id', $serie_id);
            
            $stmt->execute();
            
            registrarActividad($conn, $user_id, 'editar', 'Actualización de estado de serie', $serie_id, 'serie_maquina');
            
            $conn->commit();
            
            $_SESSION['flash_message'] = "Serie actualizada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: maquinas.php');
            exit;
            
        } catch (Exception $e) {
            $conn->rollBack();
            $error_message = "Error al actualizar la serie: " . $e->getMessage();
        }
    }
    
    // Eliminar máquina
    if ($action === 'delete' && $id > 0) {
        try {
            $conn->beginTransaction();
            
            $stmt = $conn->prepare("SELECT * FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$maquina) {
                throw new Exception("La máquina no existe");
            }
            
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM solicitud_repuestos WHERE maquina_id = :maquina_id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            if ($stmt->fetch(PDO::FETCH_ASSOC)['count'] > 0) {
                throw new Exception("No se puede eliminar la máquina porque tiene solicitudes de repuestos asociadas");
            }
            
            $stmt = $conn->prepare("DELETE FROM maquina_repuesto WHERE maquina_id = :maquina_id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            
            $stmt = $conn->prepare("DELETE FROM series_maquinas WHERE maquina_id = :maquina_id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            
            $stmt = $conn->prepare("SELECT ruta_archivo FROM maquina_imagenes WHERE maquina_id = :maquina_id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            while ($imagen = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if (file_exists($imagen['ruta_archivo'])) {
                    unlink($imagen['ruta_archivo']);
                }
            }
            
            $stmt = $conn->prepare("DELETE FROM maquina_imagenes WHERE maquina_id = :maquina_id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            
            $stmt = $conn->prepare("DELETE FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de máquina', $id, 'maquina');
            
            $conn->commit();
            
            $_SESSION['flash_message'] = "Máquina eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: maquinas.php');
            exit;
        } catch (Exception $e) {
            $conn->rollBack();
            $error_message = "Error al eliminar: " . $e->getMessage();
        }
    }
    
    // Activar/Desactivar máquina
    if ($action === 'toggle' && $id > 0) {
        try {
            $stmt = $conn->prepare("SELECT activo FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$maquina) {
                throw new Exception("La máquina no existe");
            }
            
            $nuevo_estado = $maquina['activo'] ? 0 : 1;
            $stmt = $conn->prepare("UPDATE maquinas SET activo = :activo WHERE id = :id");
            $stmt->bindParam(':activo', $nuevo_estado);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            $accion = $nuevo_estado ? 'activar' : 'desactivar';
            registrarActividad($conn, $user_id, $accion, ucfirst($accion) . ' máquina', $id, 'maquina');
            
            $_SESSION['flash_message'] = "Máquina " . ($nuevo_estado ? "activada" : "desactivada") . " correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: maquinas.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al cambiar estado: " . $e->getMessage();
        }
    }
    
    // Eliminar imagen
    if ($action === 'delete_image' && isset($_GET['image_id'])) {
        try {
            $image_id = intval($_GET['image_id']);
            
            $stmt = $conn->prepare("SELECT * FROM maquina_imagenes WHERE id = :image_id");
            $stmt->bindParam(':image_id', $image_id);
            $stmt->execute();
            $imagen = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$imagen) {
                throw new Exception("La imagen no existe");
            }
            
            if (file_exists($imagen['ruta_archivo'])) {
                unlink($imagen['ruta_archivo']);
            }
            
            if ($imagen['es_principal']) {
                $stmt = $conn->prepare("
                    UPDATE maquina_imagenes 
                    SET es_principal = 1 
                    WHERE maquina_id = :maquina_id AND id != :image_id 
                    LIMIT 1
                ");
                $stmt->bindParam(':maquina_id', $imagen['maquina_id']);
                $stmt->bindParam(':image_id', $image_id);
                $stmt->execute();
            }
            
            $stmt = $conn->prepare("DELETE FROM maquina_imagenes WHERE id = :image_id");
            $stmt->bindParam(':image_id', $image_id);
            $stmt->execute();
            
            $_SESSION['flash_message'] = "Imagen eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header("Location: maquinas.php?action=edit&id=" . $imagen['maquina_id']);
            exit;
        } catch (Exception $e) {
            $error_message = "Error al eliminar imagen: " . $e->getMessage();
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

require_once 'includes/header.php';
?>

<style>
    .maquinas-table td, .maquinas-table th { padding: 8px 10px; }
    .maquinas-table tr:hover { background-color: rgba(0,0,0,0.03); }
    
    .action-icon {
        display: inline-flex; align-items: center; justify-content: center; width: 28px; height: 28px;
        border-radius: 50%; margin-right: 5px; transition: all 0.2s; color: #555 !important; background-color: #f0f0f0;
    }
    .action-icon:hover { background-color: #e0e0e0; transform: scale(1.1); }
    .action-icon.view-icon:hover { color: #1e88e5 !important; }
    .action-icon.edit-icon:hover { color: #ffa726 !important; }
    .action-icon.delete-icon:hover { color: #ef5350 !important; }
    .action-icon i { font-size: 18px; }
    
    .form-control:focus { border-color: #1e88e5; box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25); }
    .form-control:required { border-left: 3px solid #1e88e5; }
    
    .inactive-maquina { opacity: 0.7; font-style: italic; }
    
    .series-list {
        margin-top: 15px;
        max-height: 300px;
        overflow-y: auto;
    }
    
    .serie-item {
        display: flex;
        align-items: center;
        padding: 10px;
        border-bottom: 1px solid #eee;
        background-color: #f9f9f9;
        margin-bottom: 5px;
        border-radius: 4px;
    }
    
    .serie-principal {
        border-left: 3px solid #4caf50;
    }
    
    .serie-info {
        flex: 1;
    }
    
    .serie-actions {
        margin-left: 10px;
    }
    
    .serie-estado {
        display: inline-block;
        padding: 2px 6px;
        border-radius: 4px;
        font-size: 0.8em;
        font-weight: bold;
        margin-left: 5px;
    }
    
    .estado-nueva { background-color: #bbdefb; color: #1976d2; }
    .estado-usada { background-color: #c8e6c9; color: #388e3c; }
    .estado-servicio_tecnico { background-color: #ffccbc; color: #e64a19; }
    .estado-comodato { background-color: #d1c4e9; color: #512da8; }
    .estado-devuelta { background-color: #f5f5f5; color: #616161; }
    .estado-irreparable { background-color: #ffcdd2; color: #d32f2f; }
    
    .image-gallery {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-top: 20px;
    }
    
    .image-item {
        position: relative;
        width: 150px;
        height: 150px;
        border: 1px solid #ddd;
        border-radius: 4px;
        overflow: hidden;
    }
    
    .image-item img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    
    .image-actions {
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        background-color: rgba(0,0,0,0.5);
        display: flex;
        justify-content: space-around;
        padding: 5px;
    }
    
    .image-actions a {
        color: white;
        text-decoration: none;
    }
    
    .image-principal {
        position: absolute;
        top: 5px;
        right: 5px;
        background-color: #4caf50;
        color: white;
        padding: 2px 5px;
        border-radius: 3px;
        font-size: 0.7em;
    }
    
    .series-container {
        margin-top: 20px;
        border: 1px solid #ddd;
        border-radius: 5px;
        padding: 15px;
        background-color: #f9f9f9;
    }
    
    .series-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 15px;
    }
    
    .series-header h3 {
        margin: 0;
    }
    
    .series-list {
        max-height: 300px;
        overflow-y: auto;
        margin-bottom: 15px;
    }
    
    .serie-row {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 10px;
        padding: 10px;
        background-color: #fff;
        border-radius: 4px;
        border: 1px solid #eee;
    }
    
    .serie-field {
        flex: 1;
        min-width: 200px;
    }
    
    .serie-actions {
        display: flex;
        align-items: center;
    }
    
    .add-serie-btn {
        background-color: #4caf50;
        color: white;
        border: none;
        padding: 5px 10px;
        border-radius: 4px;
        cursor: pointer;
    }
    
    .remove-serie-btn {
        background-color: #f44336;
        color: white;
        border: none;
        padding: 5px 10px;
        border-radius: 4px;
        cursor: pointer;
    }
</style>

<?php if (isset($error_message)): ?>
<div class="alert alert-danger"><?php echo $error_message; ?></div>
<?php endif; ?>

<?php
switch ($action) {
    case 'new':
    case 'edit':
        $maquina = null;
        $series = [];
        $imagenes = [];
        
        if ($action === 'edit' && $id > 0) {
            $stmt = $conn->prepare("SELECT * FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$maquina) {
                echo '<div class="alert alert-danger">Máquina no encontrada</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="maquinas.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT sm.*, p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM series_maquinas sm
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                WHERE sm.maquina_id = :maquina_id
                ORDER BY sm.codigo_serie
            ");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $stmt = $conn->prepare("SELECT * FROM maquina_imagenes WHERE maquina_id = :maquina_id ORDER BY es_principal DESC, id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            $imagenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        // Obtener lista de proveedores (reemplaza clientes)
        $stmt = $conn->prepare("SELECT id, nombre, rut FROM proveedores WHERE activo = 1 ORDER BY nombre");
        $stmt->execute();
        $proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $title = ($action === 'new') ? 'Nueva Máquina' : 'Editar Máquina';
        ?>
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><?php echo $title; ?></h2>
                <a href="maquinas.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="maquinas.php?action=<?php echo $action . ($action === 'edit' ? '&id=' . $id : ''); ?>" enctype="multipart/form-data">
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <h3>Información Básica</h3>
                        
                        <div class="form-group">
                            <label for="codigo" class="form-label">Código <span class="text-danger">*</span></label>
                            <input type="text" id="codigo" name="codigo" class="form-control" 
                                   value="<?php echo isset($maquina['codigo']) ? htmlspecialchars($maquina['codigo']) : ''; ?>" 
                                   placeholder="Código único de la máquina" required>
                            <?php if ($action === 'edit'): ?>
                            <input type="hidden" name="codigo_original" value="<?php echo htmlspecialchars($maquina['codigo']); ?>">
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="nombre" class="form-label">Nombre <span class="text-danger">*</span></label>
                            <input type="text" id="nombre" name="nombre" class="form-control" 
                                   value="<?php echo isset($maquina['nombre']) ? htmlspecialchars($maquina['nombre']) : ''; ?>" 
                                   placeholder="Nombre de la máquina" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="marca" class="form-label">Marca</label>
                            <input type="text" id="marca" name="marca" class="form-control" 
                                   value="<?php echo isset($maquina['marca']) ? htmlspecialchars($maquina['marca']) : ''; ?>" 
                                   placeholder="Marca de la máquina">
                        </div>
                        
                        <div class="form-group">
                            <label for="modelo" class="form-label">Modelo</label>
                            <input type="text" id="modelo" name="modelo" class="form-control" 
                                   value="<?php echo isset($maquina['modelo']) ? htmlspecialchars($maquina['modelo']) : ''; ?>" 
                                   placeholder="Modelo de la máquina">
                        </div>
                        
                        <div class="form-group">
                            <label for="costo" class="form-label">Precio Costo</label>
                            <div class="input-group">
                                <span class="input-group-text">$</span>
                                <input type="text" id="costo" name="costo" class="form-control" 
                                       value="<?php echo isset($maquina['costo']) && $maquina['costo'] !== null ? number_format($maquina['costo'], 0, ',', '.') : ''; ?>" 
                                       placeholder="Costo de la máquina">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="descripcion" class="form-label">Descripción</label>
                            <textarea id="descripcion" name="descripcion" class="form-control" rows="4" 
                                      placeholder="Descripción detallada de la máquina"><?php echo isset($maquina['descripcion']) ? htmlspecialchars($maquina['descripcion']) : ''; ?></textarea>
                        </div>
                        
                        <div class="form-check" style="margin-top: 15px;">
                            <input type="checkbox" class="form-check-input" id="activo" name="activo" value="1" 
                                  <?php echo (!isset($maquina) || $maquina['activo'] == 1) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="activo">Activo</label>
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="series-container">
                            <div class="series-header">
                                <h3>Series de la Máquina</h3>
                                <button type="button" id="addSerieBtn" class="add-serie-btn">
                                    <i class="material-icons">add</i> Añadir Serie
                                </button>
                            </div>
                            
                            <div id="series-wrapper">
                                <?php if (!empty($series)): ?>
                                    <?php foreach ($series as $index => $serie): ?>
                                        <div class="serie-row">
                                            <input type="hidden" name="series[]" value="<?php echo htmlspecialchars($serie['codigo_serie']); ?>">
                                            
                                            <div class="serie-field">
                                                <label>Número de Serie <span class="text-danger">*</span></label>
                                                <input type="text" name="numero_serie[]" class="form-control" value="<?php echo htmlspecialchars($serie['codigo_serie']); ?>" required>
                                            </div>
                                            
                                            <div class="serie-field">
                                                <label>Estado</label>
                                                <select name="estado_serie[]" class="form-control estado-select">
                                                    <option value="nueva" <?php echo $serie['estado'] === 'nueva' ? 'selected' : ''; ?>>Nueva</option>
                                                    <option value="usada" <?php echo $serie['estado'] === 'usada' ? 'selected' : ''; ?>>Usada</option>
                                                    <option value="servicio_tecnico" <?php echo $serie['estado'] === 'servicio_tecnico' ? 'selected' : ''; ?>>En Servicio Técnico</option>
                                                    <option value="comodato" <?php echo $serie['estado'] === 'comodato' ? 'selected' : ''; ?>>En Comodato</option>
                                                    <option value="devuelta" <?php echo $serie['estado'] === 'devuelta' ? 'selected' : ''; ?>>Devuelta</option>
                                                    <option value="irreparable" <?php echo $serie['estado'] === 'irreparable' ? 'selected' : ''; ?>>Irreparable</option>
                                                </select>
                                            </div>
                                            
                                            <div class="serie-field proveedor-selector" style="<?php echo $serie['estado'] === 'comodato' ? '' : 'display: none;'; ?>">
                                                <label>Proveedor</label>
                                                <select name="proveedor_id[]" class="form-control">
                                                    <option value="">Seleccione Proveedor</option>
                                                    <?php foreach ($proveedores as $proveedor): ?>
                                                        <option value="<?php echo $proveedor['id']; ?>" <?php echo $serie['proveedor_id'] == $proveedor['id'] ? 'selected' : ''; ?>>
                                                            <?php echo htmlspecialchars($proveedor['nombre'] . ' (' . $proveedor['rut'] . ')'); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            
                                            <div class="serie-field fecha-selector" style="<?php echo $serie['estado'] === 'comodato' ? '' : 'display: none;'; ?>">
                                                <label>Fecha Inicio</label>
                                                <input type="date" name="fecha_inicio[]" class="form-control" value="<?php echo $serie['fecha_inicio'] ?? ''; ?>">
                                            </div>
                                            
                                            <div class="serie-actions">
                                                <button type="button" class="remove-serie-btn">
                                                    <i class="material-icons">remove</i>
                                                </button>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="serie-row">
                                        <div class="serie-field">
                                            <label>Número de Serie <span class="text-danger">*</span></label>
                                            <input type="text" name="numero_serie[]" class="form-control" required>
                                        </div>
                                        
                                        <div class="serie-field">
                                            <label>Estado</label>
                                            <select name="estado_serie[]" class="form-control estado-select">
                                                <option value="nueva">Nueva</option>
                                                <option value="usada">Usada</option>
                                                <option value="servicio_tecnico">En Servicio Técnico</option>
                                                <option value="comodato">En Comodato</option>
                                                <option value="devuelta">Devuelta</option>
                                                <option value="irreparable">Irreparable</option>
                                            </select>
                                        </div>
                                        
                                        <div class="serie-field proveedor-selector" style="display: none;">
                                            <label>Proveedor</label>
                                            <select name="proveedor_id[]" class="form-control">
                                                <option value="">Seleccione Proveedor</option>
                                                <?php foreach ($proveedores as $proveedor): ?>
                                                    <option value="<?php echo $proveedor['id']; ?>">
                                                        <?php echo htmlspecialchars($proveedor['nombre'] . ' (' . $proveedor['rut'] . ')'); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        
                                        <div class="serie-field fecha-selector" style="display: none;">
                                            <label>Fecha Inicio</label>
                                            <input type="date" name="fecha_inicio[]" class="form-control">
                                        </div>
                                        
                                        <div class="serie-actions">
                                            <button type="button" class="remove-serie-btn">
                                                <i class="material-icons">remove</i>
                                            </button>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div style="margin-top: 20px;">
                            <h3>Imágenes</h3>
                            
                            <?php if (!empty($imagenes)): ?>
                                <div class="image-gallery">
                                    <?php foreach ($imagenes as $imagen): ?>
                                        <div class="image-item">
                                            <img src="<?php echo htmlspecialchars($imagen['ruta_archivo']); ?>" alt="Imagen de la máquina">
                                            
                                            <?php if ($imagen['es_principal']): ?>
                                                <div class="image-principal">Principal</div>
                                            <?php endif; ?>
                                            
                                            <div class="image-actions">
                                                <?php if (!$imagen['es_principal']): ?>
                                                    <div class="form-check">
                                                        <input type="radio" name="imagen_principal_existente" id="imagen_<?php echo $imagen['id']; ?>" value="<?php echo $imagen['id']; ?>" class="form-check-input">
                                                        <label for="imagen_<?php echo $imagen['id']; ?>" class="form-check-label">Principal</label>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <a href="maquinas.php?action=delete_image&image_id=<?php echo $imagen['id']; ?>" 
                                                   onclick="return confirm('¿Está seguro de eliminar esta imagen?')">
                                                    <i class="material-icons">delete</i>
                                                </a>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div style="margin-top: 15px;">
                                <label>Añadir Imágenes</label>
                                <input type="file" name="imagenes[]" class="form-control" multiple accept="image/*">
                                
                                <div class="form-check" style="margin-top: 10px;">
                                    <input type="checkbox" class="form-check-input" id="imagen_principal" name="imagen_principal" value="0">
                                    <label class="form-check-label" for="imagen_principal">Establecer como imagen principal</label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Máquina
                    </button>
                    <a href="maquinas.php" class="btn btn-danger">
                        <i class="material-icons">cancel</i> Cancelar
                    </a>
                </div>
            </form>
        </div>
        
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const costoInput = document.getElementById('costo');
                if (costoInput) {
                    costoInput.addEventListener('input', function(e) {
                        let value = e.target.value.replace(/[^\d]/g, '');
                        if (value) {
                            value = parseInt(value).toLocaleString('es-CL');
                            e.target.value = value;
                        }
                    });
                }
                
                const addSerieBtn = document.getElementById('addSerieBtn');
                const seriesWrapper = document.getElementById('series-wrapper');
                
                if (addSerieBtn && seriesWrapper) {
                    addSerieBtn.addEventListener('click', function() {
                        const serieRow = document.createElement('div');
                        serieRow.className = 'serie-row';
                        
                        serieRow.innerHTML = `
                            <div class="serie-field">
                                <label>Número de Serie <span class="text-danger">*</span></label>
                                <input type="text" name="numero_serie[]" class="form-control" required>
                            </div>
                            
                            <div class="serie-field">
                                <label>Estado</label>
                                <select name="estado_serie[]" class="form-control estado-select">
                                    <option value="nueva">Nueva</option>
                                    <option value="usada">Usada</option>
                                    <option value="servicio_tecnico">En Servicio Técnico</option>
                                    <option value="comodato">En Comodato</option>
                                    <option value="devuelta">Devuelta</option>
                                    <option value="irreparable">Irreparable</option>
                                </select>
                            </div>
                            
                            <div class="serie-field proveedor-selector" style="display: none;">
                                <label>Proveedor</label>
                                <select name="proveedor_id[]" class="form-control">
                                    <option value="">Seleccione Proveedor</option>
                                    <?php foreach ($proveedores as $proveedor): ?>
                                        <option value="<?php echo $proveedor['id']; ?>">
                                            <?php echo htmlspecialchars($proveedor['nombre'] . ' (' . $proveedor['rut'] . ')'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="serie-field fecha-selector" style="display: none;">
                                <label>Fecha Inicio</label>
                                <input type="date" name="fecha_inicio[]" class="form-control">
                            </div>
                            
                            <div class="serie-actions">
                                <button type="button" class="remove-serie-btn">
                                    <i class="material-icons">remove</i>
                                </button>
                            </div>
                        `;
                        
                        seriesWrapper.appendChild(serieRow);
                        setupSerieListeners(serieRow);
                    });
                    
                    document.querySelectorAll('.serie-row').forEach(row => {
                        setupSerieListeners(row);
                    });
                    
                    function setupSerieListeners(row) {
                        const removeBtn = row.querySelector('.remove-serie-btn');
                        if (removeBtn) {
                            removeBtn.addEventListener('click', function() {
                                row.remove();
                            });
                        }
                        
                        const estadoSelect = row.querySelector('.estado-select');
                        if (estadoSelect) {
                            estadoSelect.addEventListener('change', function() {
                                const proveedorSelector = row.querySelector('.proveedor-selector');
                                const fechaSelector = row.querySelector('.fecha-selector');
                                
                                if (this.value === 'comodato') {
                                    proveedorSelector.style.display = '';
                                    fechaSelector.style.display = '';
                                } else {
                                    proveedorSelector.style.display = 'none';
                                    fechaSelector.style.display = 'none';
                                }
                            });
                            estadoSelect.dispatchEvent(new Event('change'));
                        }
                    }
                }
            });
        </script>
        <?php
        break;
        
    case 'view':
        try {
            $stmt = $conn->prepare("SELECT * FROM maquinas WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$maquina) {
                echo '<div class="alert alert-danger">Máquina no encontrada</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="maquinas.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            $stmt = $conn->prepare("
                SELECT sm.*, p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM series_maquinas sm
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                WHERE sm.maquina_id = :maquina_id
            ");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $stmt = $conn->prepare("SELECT * FROM maquina_imagenes WHERE maquina_id = :maquina_id ORDER BY es_principal DESC, id");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            $imagenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $stmt = $conn->prepare("
                SELECT r.id, r.codigo, r.nombre, r.stock, r.activo
                FROM repuestos r
                INNER JOIN maquina_repuesto mr ON r.id = mr.repuesto_id
                WHERE mr.maquina_id = :maquina_id
                ORDER BY r.codigo
            ");
            $stmt->bindParam(':maquina_id', $id);
            $stmt->execute();
            $repuestos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener lista de proveedores (reemplaza clientes)
            $stmt = $conn->prepare("SELECT id, nombre, rut FROM proveedores WHERE activo = 1 ORDER BY nombre");
            $stmt->execute();
            $proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Detalle de Máquina: <?php echo htmlspecialchars($maquina['codigo'] . ' - ' . $maquina['nombre']); ?></h2>
                    <div>
                        <a href="maquinas.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">arrow_back</i> Volver a la Lista
                        </a>
                        
                        <a href="maquinas.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                            <i class="material-icons">edit</i> Editar
                        </a>
                        
                        <?php if ($is_admin): ?>
                        <a href="maquinas.php?action=toggle&id=<?php echo $id; ?>" class="btn btn-secondary btn-sm">
                            <i class="material-icons"><?php echo $maquina['activo'] ? 'toggle_off' : 'toggle_on'; ?></i>
                            <?php echo $maquina['activo'] ? 'Desactivar' : 'Activar'; ?>
                        </a>
                        
                        <a href="maquinas.php?action=delete&id=<?php echo $id; ?>" class="btn btn-danger btn-sm" 
                           onclick="return confirm('¿Está seguro de eliminar esta máquina?')">
                            <i class="material-icons">delete</i> Eliminar
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card" style="height: 100%;">
                            <div class="panel-header">
                                <h3 class="panel-title">Información de la Máquina</h3>
                            </div>
                            <table>
                                <tr>
                                    <th style="width: 30%;">Código:</th>
                                    <td><?php echo htmlspecialchars($maquina['codigo']); ?></td>
                                </tr>
                                <tr>
                                    <th>Nombre:</th>
                                    <td><?php echo htmlspecialchars($maquina['nombre']); ?></td>
                                </tr>
                                <tr>
                                    <th>Marca:</th>
                                    <td><?php echo !empty($maquina['marca']) ? htmlspecialchars($maquina['marca']) : 'No especificada'; ?></td>
                                </tr>
                                <tr>
                                    <th>Modelo:</th>
                                    <td><?php echo !empty($maquina['modelo']) ? htmlspecialchars($maquina['modelo']) : 'No especificado'; ?></td>
                                </tr>
                                <?php if ($is_admin && $maquina['costo'] !== null): ?>
                                <tr>
                                    <th>Costo:</th>
                                    <td><?php echo formatoMoneda($maquina['costo']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <tr>
                                    <th>Estado:</th>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $maquina['activo'] ? '3' : '4'; ?>">
                                            <?php echo $maquina['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Fecha de Registro:</th>
                                    <td><?php echo date('d/m/Y', strtotime($maquina['fecha_registro'])); ?></td>
                                </tr>
                            </table>
                            
                            <?php if (!empty($maquina['descripcion'])): ?>
                            <div style="margin-top: 15px; padding: 0 10px;">
                                <h4>Descripción:</h4>
                                <p><?php echo nl2br(htmlspecialchars($maquina['descripcion'])); ?></p>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <?php if (!empty($repuestos)): ?>
                        <div class="card" style="margin-top: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Repuestos Compatibles</h3>
                                <a href="repuestos.php?maquina_id=<?php echo $id; ?>" class="btn btn-primary btn-sm">
                                    <i class="material-icons">build</i> Ver Todos
                                </a>
                            </div>
                            <div class="table-responsive">
                                <table class="maquinas-table">
                                    <thead>
                                        <tr>
                                            <th>Código</th>
                                            <th>Nombre</th>
                                            <th>Stock</th>
                                            <th>Estado</th>
                                            <th>Acciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($repuestos as $repuesto): ?>
                                        <tr class="<?php echo $repuesto['activo'] ? '' : 'inactive-maquina'; ?>">
                                            <td><?php echo htmlspecialchars($repuesto['codigo']); ?></td>
                                            <td><?php echo htmlspecialchars($repuesto['nombre']); ?></td>
                                            <td><?php echo $repuesto['stock']; ?></td>
                                            <td>
                                                <span class="status-badge badge-estado-<?php echo $repuesto['activo'] ? '3' : '4'; ?>">
                                                    <?php echo $repuesto['activo'] ? 'Activo' : 'Inactivo'; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <a href="repuestos.php?action=view&id=<?php echo $repuesto['id']; ?>" 
                                                   class="action-icon view-icon" title="Ver Repuesto">
                                                    <i class="material-icons">visibility</i>
                                                </a>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card">
                            <div class="panel-header">
                                <h3 class="panel-title">Series de la Máquina</h3>
                                <a href="maquinas.php?action=edit&id=<?php echo $id; ?>" class="btn btn-primary btn-sm">
                                    <i class="material-icons">add</i> Gestionar Series
                                </a>
                            </div>
                            
                            <?php if (empty($series)): ?>
                            <p class="text-muted" style="padding: 10px;">No hay series registradas para esta máquina</p>
                            <?php else: ?>
                            <div class="series-list">
                                <?php foreach ($series as $serie): ?>
                                <div class="serie-item">
                                    <div class="serie-info">
                                        <strong><?php echo htmlspecialchars($serie['codigo_serie']); ?></strong>
                                        <span class="serie-estado estado-<?php echo $serie['estado']; ?>">
                                            <?php 
                                            $estados = [
                                                'nueva' => 'Nueva',
                                                'usada' => 'Usada',
                                                'servicio_tecnico' => 'En Servicio Técnico',
                                                'comodato' => 'En Comodato',
                                                'devuelta' => 'Devuelta',
                                                'irreparable' => 'Irreparable'
                                            ];
                                            echo $estados[$serie['estado']] ?? $serie['estado']; 
                                            ?>
                                        </span>
                                        
                                        <?php if ($serie['proveedor_id'] && !empty($serie['proveedor_nombre'])): ?>
                                        <div style="margin-top: 5px;">
                                            <span class="text-muted">Proveedor:</span> 
                                            <?php echo htmlspecialchars($serie['proveedor_nombre'] . ' (' . $serie['proveedor_rut'] . ')'); ?>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($serie['fecha_inicio']): ?>
                                        <div style="margin-top: 3px;">
                                            <span class="text-muted">Inicio:</span> 
                                            <?php echo date('d/m/Y', strtotime($serie['fecha_inicio'])); ?>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($serie['fecha_fin']): ?>
                                        <div style="margin-top: 3px;">
                                            <span class="text-muted">Fin:</span> 
                                            <?php echo date('d/m/Y', strtotime($serie['fecha_fin'])); ?>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="serie-actions">
                                        <button type="button" onclick="openManageSerie(<?php echo $serie['id']; ?>)" class="btn btn-sm btn-primary">
                                            <i class="material-icons">edit</i>
                                        </button>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <?php if (!empty($imagenes)): ?>
                        <div class="card" style="margin-top: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Imágenes</h3>
                            </div>
                            
                            <div class="image-gallery" style="padding: 10px;">
                                <?php foreach ($imagenes as $imagen): ?>
                                <div class="image-item">
                                    <img src="<?php echo htmlspecialchars($imagen['ruta_archivo']); ?>" alt="Imagen de la máquina">
                                    
                                    <?php if ($imagen['es_principal']): ?>
                                    <div class="image-principal">Principal</div>
                                    <?php endif; ?>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <div class="modal fade" id="serieModal" tabindex="-1" role="dialog" aria-labelledby="serieModalLabel" aria-hidden="true">
                <div class="modal-dialog" role="document">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="serieModalLabel">Gestionar Serie</h5>
                            <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">&times;</span>
                            </button>
                        </div>
                        <form method="POST" action="maquinas.php?action=manage_series&id=<?php echo $id; ?>">
                            <div class="modal-body">
                                <input type="hidden" id="serie_id" name="serie_id" value="">
                                
                                <div class="form-group">
                                    <label for="codigo_serie_modal">Número de Serie</label>
                                    <input type="text" id="codigo_serie_modal" class="form-control" readonly>
                                </div>
                                
                                <div class="form-group">
                                    <label for="estado">Estado</label>
                                    <select id="estado" name="estado" class="form-control">
                                        <option value="nueva">Nueva</option>
                                        <option value="usada">Usada</option>
                                        <option value="servicio_tecnico">En Servicio Técnico</option>
                                        <option value="comodato">En Comodato</option>
                                        <option value="devuelta">Devuelta</option>
                                        <option value="irreparable">Irreparable</option>
                                    </select>
                                </div>
                                
                                <div id="proveedor_container" class="form-group">
                                    <label for="proveedor_id">Proveedor</label>
                                    <select id="proveedor_id" name="proveedor_id" class="form-control">
                                        <option value="">Seleccione Proveedor</option>
                                        <?php foreach ($proveedores as $proveedor): ?>
                                        <option value="<?php echo $proveedor['id']; ?>">
                                            <?php echo htmlspecialchars($proveedor['nombre'] . ' (' . $proveedor['rut'] . ')'); ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div id="fecha_container" class="form-group">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <label for="fecha_inicio">Fecha Inicio</label>
                                            <input type="date" id="fecha_inicio" name="fecha_inicio" class="form-control">
                                        </div>
                                        <div class="col-md-6">
                                            <label for="fecha_fin">Fecha Fin</label>
                                            <input type="date" id="fecha_fin" name="fecha_fin" class="form-control">
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="notas">Notas</label>
                                    <textarea id="notas" name="notas" class="form-control" rows="3"></textarea>
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                                <button type="submit" class="btn btn-primary">Guardar Cambios</button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <script>
                function openManageSerie(serieId) {
                    fetch('get_serie.php?id=' + serieId)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                const serie = data.serie;
                                
                                document.getElementById('serie_id').value = serie.id;
                                document.getElementById('codigo_serie_modal').value = serie.codigo_serie;
                                document.getElementById('estado').value = serie.estado;
                                document.getElementById('proveedor_id').value = serie.proveedor_id || '';
                                document.getElementById('fecha_inicio').value = serie.fecha_inicio || '';
                                document.getElementById('fecha_fin').value = serie.fecha_fin || '';
                                document.getElementById('notas').value = serie.notas || '';
                                
                                toggleProveedorFields();
                                
                                $('#serieModal').modal('show');
                            } else {
                                alert('Error: ' + data.message);
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            alert('Error al cargar los datos de la serie');
                        });
                }
                
                document.addEventListener('DOMContentLoaded', function() {
                    const estadoSelect = document.getElementById('estado');
                    if (estadoSelect) {
                        estadoSelect.addEventListener('change', toggleProveedorFields);
                    }
                    
                    function toggleProveedorFields() {
                        const estado = estadoSelect.value;
                        const proveedorContainer = document.getElementById('proveedor_container');
                        const fechaContainer = document.getElementById('fecha_container');
                        
                        if (estado === 'comodato') {
                            proveedorContainer.style.display = 'block';
                            fechaContainer.style.display = 'block';
                        } else {
                            proveedorContainer.style.display = 'none';
                            fechaContainer.style.display = 'none';
                        }
                    }
                });
            </script>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="maquinas.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    default:
        try {
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            $activo = isset($_GET['activo']) ? $_GET['activo'] : 'all';
            
            $sql = "SELECT * FROM maquinas WHERE 1=1";
            $params = [];
            
            if (!empty($busqueda)) {
                $sql .= " AND (codigo LIKE :busqueda OR nombre LIKE :busqueda OR marca LIKE :busqueda OR modelo LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            
            if ($activo !== 'all') {
                $activo_valor = ($activo === 'active') ? 1 : 0;
                $sql .= " AND activo = :activo";
                $params[':activo'] = $activo_valor;
            }
            
            $sql .= " ORDER BY codigo";
            
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            $stmt->execute();
            $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $series_por_maquina = [];
            if (!empty($maquinas)) {
                $ids = array_column($maquinas, 'id');
                if (!empty($ids)) {
                    $placeholders = implode(',', array_fill(0, count($ids), '?'));
                    $stmt = $conn->prepare("
                        SELECT maquina_id, COUNT(*) as total, 
                               SUM(CASE WHEN estado = 'comodato' THEN 1 ELSE 0 END) as en_comodato
                        FROM series_maquinas
                        WHERE maquina_id IN ($placeholders)
                        GROUP BY maquina_id
                    ");
                    $stmt->execute($ids);
                    $series_conteo = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    foreach ($series_conteo as $conteo) {
                        $series_por_maquina[$conteo['maquina_id']] = [
                            'total' => $conteo['total'],
                            'en_comodato' => $conteo['en_comodato']
                        ];
                    }
                }
            }
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Gestión de Máquinas</h2>
                    <a href="maquinas.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nueva Máquina
                    </a>
                </div>
                
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="maquinas.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px;">
                        <div style="flex: 2; min-width: 300px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" 
                                   placeholder="Código, Nombre, Marca, Modelo..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div style="flex: 1; min-width: 200px;">
                            <label for="activo" class="form-label">Estado</label>
                            <select id="activo" name="activo" class="form-control">
                                <option value="all" <?php echo $activo === 'all' ? 'selected' : ''; ?>>Todos</option>
                                <option value="active" <?php echo $activo === 'active' ? 'selected' : ''; ?>>Activos</option>
                                <option value="inactive" <?php echo $activo === 'inactive' ? 'selected' : ''; ?>>Inactivos</option>
                            </select>
                        </div>
                        
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="maquinas.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <div class="table-responsive">
                    <table class="maquinas-table">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Nombre</th>
                                <th>Marca</th>
                                <th>Modelo</th>
                                <th>Series</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($maquinas)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center;">No se encontraron máquinas</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($maquinas as $maquina): ?>
                                <tr class="<?php echo $maquina['activo'] ? '' : 'inactive-maquina'; ?>">
                                    <td><?php echo htmlspecialchars($maquina['codigo']); ?></td>
                                    <td><?php echo htmlspecialchars($maquina['nombre']); ?></td>
                                    <td><?php echo !empty($maquina['marca']) ? htmlspecialchars($maquina['marca']) : 'N/A'; ?></td>
                                    <td><?php echo !empty($maquina['modelo']) ? htmlspecialchars($maquina['modelo']) : 'N/A'; ?></td>
                                    <td>
                                        <?php 
                                        if (isset($series_por_maquina[$maquina['id']])) {
                                            $conteo = $series_por_maquina[$maquina['id']];
                                            echo $conteo['total'] . ' series';
                                            if ($conteo['en_comodato'] > 0) {
                                                echo ' <span class="badge badge-info">' . $conteo['en_comodato'] . ' en comodato</span>';
                                            }
                                        } else {
                                            echo 'Sin series';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $maquina['activo'] ? '3' : '4'; ?>">
                                            <?php echo $maquina['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="maquinas.php?action=view&id=<?php echo $maquina['id']; ?>" 
                                           title="Ver detalles" class="action-icon view-icon">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                        
                                        <a href="maquinas.php?action=edit&id=<?php echo $maquina['id']; ?>" 
                                           title="Editar" class="action-icon edit-icon">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        
                                        <?php if ($is_admin): ?>
                                        <a href="maquinas.php?action=toggle&id=<?php echo $maquina['id']; ?>" 
                                           title="<?php echo $maquina['activo'] ? 'Desactivar' : 'Activar'; ?>" 
                                           class="action-icon <?php echo $maquina['activo'] ? 'delete-icon' : 'view-icon'; ?>"
                                           onclick="return confirm('¿Está seguro de <?php echo $maquina['activo'] ? 'desactivar' : 'activar'; ?> esta máquina?')">
                                            <i class="material-icons"><?php echo $maquina['activo'] ? 'toggle_off' : 'toggle_on'; ?></i>
                                        </a>
                                        
                                        <a href="maquinas.php?action=delete&id=<?php echo $maquina['id']; ?>" 
                                           title="Eliminar" class="action-icon delete-icon"
                                           onclick="return confirm('¿Está seguro de eliminar esta máquina?')">
                                            <i class="material-icons">delete</i>
                                        </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar máquinas: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<?php require_once 'includes/footer.php'; ?>