<?php
/**
 * Módulo de historial de actividades
 * Permite visualizar el registro de todas las acciones realizadas en el sistema
 */

// Definir el título de la página
$page_title = 'Historial de Actividades';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Parámetros de paginación
$registros_por_pagina = 30;
$pagina_actual = isset($_GET['pagina']) ? intval($_GET['pagina']) : 1;
if ($pagina_actual < 1) $pagina_actual = 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// Filtros
$tipo = isset($_GET['tipo']) ? trim($_GET['tipo']) : '';
$documento_tipo = isset($_GET['documento_tipo']) ? trim($_GET['documento_tipo']) : '';
$fecha_desde = isset($_GET['fecha_desde']) ? $_GET['fecha_desde'] : '';
$fecha_hasta = isset($_GET['fecha_hasta']) ? $_GET['fecha_hasta'] : '';
$usuario_id = isset($_GET['usuario_id']) ? intval($_GET['usuario_id']) : 0;
$busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';

// Si no hay fechas específicas, usar el último mes como predeterminado
if (empty($fecha_desde)) {
    $fecha_desde = date('Y-m-d', strtotime('-30 days'));
}
if (empty($fecha_hasta)) {
    $fecha_hasta = date('Y-m-d');
}

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Obtener los tipos disponibles de actividades (para filtros)
    $stmt = $conn->prepare("SELECT DISTINCT tipo FROM historial_cambios ORDER BY tipo");
    $stmt->execute();
    $tipos_actividad = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Obtener los tipos de documentos disponibles (para filtros)
    $stmt = $conn->prepare("SELECT DISTINCT documento_tipo FROM historial_cambios WHERE documento_tipo IS NOT NULL ORDER BY documento_tipo");
    $stmt->execute();
    $tipos_documento = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Obtener usuarios para el filtro (solo si es admin)
    $usuarios = [];
    if ($is_admin) {
        $stmt = $conn->prepare("SELECT id, nombre, apellido FROM users ORDER BY nombre, apellido");
        $stmt->execute();
        $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Construir la consulta base
    $sql = "
        SELECT h.*, u.nombre, u.apellido 
        FROM historial_cambios h
        JOIN users u ON h.user_id = u.id
        WHERE 1=1
    ";
    
    $count_sql = "SELECT COUNT(*) FROM historial_cambios h WHERE 1=1";
    $params = [];
    
    // Si no es admin, solo mostrar actividades propias
    if (!$is_admin) {
        $sql .= " AND h.user_id = :user_id";
        $count_sql .= " AND h.user_id = :user_id";
        $params[':user_id'] = $user_id;
    } 
    // Si es admin y se seleccionó un usuario específico
    elseif ($usuario_id > 0) {
        $sql .= " AND h.user_id = :user_id";
        $count_sql .= " AND h.user_id = :user_id";
        $params[':user_id'] = $usuario_id;
    }
    
    // Filtro por tipo de actividad
    if (!empty($tipo)) {
        $sql .= " AND h.tipo = :tipo";
        $count_sql .= " AND h.tipo = :tipo";
        $params[':tipo'] = $tipo;
    }
    
    // Filtro por tipo de documento
    if (!empty($documento_tipo)) {
        $sql .= " AND h.documento_tipo = :documento_tipo";
        $count_sql .= " AND h.documento_tipo = :documento_tipo";
        $params[':documento_tipo'] = $documento_tipo;
    }
    
    // Filtro por fecha
    if (!empty($fecha_desde)) {
        $sql .= " AND DATE(h.fecha_hora) >= :fecha_desde";
        $count_sql .= " AND DATE(h.fecha_hora) >= :fecha_desde";
        $params[':fecha_desde'] = $fecha_desde;
    }
    
    if (!empty($fecha_hasta)) {
        $sql .= " AND DATE(h.fecha_hora) <= :fecha_hasta";
        $count_sql .= " AND DATE(h.fecha_hora) <= :fecha_hasta";
        $params[':fecha_hasta'] = $fecha_hasta;
    }
    
    // Filtro por búsqueda
    if (!empty($busqueda)) {
        $sql .= " AND (h.descripcion LIKE :busqueda OR u.nombre LIKE :busqueda OR u.apellido LIKE :busqueda)";
        $count_sql .= " AND (h.descripcion LIKE :busqueda OR u.nombre LIKE :busqueda OR u.apellido LIKE :busqueda)";
        $params[':busqueda'] = '%' . $busqueda . '%';
    }
    
    // Ordenar por fecha_hora descendente
    $sql .= " ORDER BY h.fecha_hora DESC";
    
    // Agregar límite para paginación
    $sql .= " LIMIT :offset, :limit";
    
    // Preparar y ejecutar consulta para el conteo total
    $stmt = $conn->prepare($count_sql);
    
    foreach ($params as $param => $value) {
        $stmt->bindValue($param, $value);
    }
    
    $stmt->execute();
    $total_registros = $stmt->fetchColumn();
    $total_paginas = ceil($total_registros / $registros_por_pagina);
    
    // Preparar y ejecutar consulta principal
    $stmt = $conn->prepare($sql);
    
    foreach ($params as $param => $value) {
        $stmt->bindValue($param, $value);
    }
    
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $registros_por_pagina, PDO::PARAM_INT);
    
    $stmt->execute();
    $actividades = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    /* Estilos personalizados para el historial */
    .historial-table td, .historial-table th {
        padding: 8px 10px; /* Reducimos el padding para filas menos gruesas */
    }
    
    .historial-table tr:hover {
        background-color: rgba(0,0,0,0.03);
    }
    
    .tipo-badge {
        display: inline-block;
        padding: 3px 8px;
        border-radius: 20px;
        font-size: 11px;
        font-weight: 500;
        text-align: center;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .tipo-login {
        background-color: rgba(30, 136, 229, 0.2);
        color: #1e88e5;
    }
    
    .tipo-logout {
        background-color: rgba(96, 125, 139, 0.2);
        color: #607d8b;
    }
    
    .tipo-crear {
        background-color: rgba(102, 187, 106, 0.2);
        color: #66bb6a;
    }
    
    .tipo-editar {
        background-color: rgba(255, 167, 38, 0.2);
        color: #ffa726;
    }
    
    .tipo-eliminar {
        background-color: rgba(239, 83, 80, 0.2);
        color: #ef5350;
    }
    
    .tipo-aprobar {
        background-color: rgba(123, 31, 162, 0.2);
        color: #7b1fa2;
    }
    
    .tipo-rechazar {
        background-color: rgba(216, 27, 96, 0.2);
        color: #d81b60;
    }
    
    .tipo-descargar {
        background-color: rgba(0, 150, 136, 0.2);
        color: #009688;
    }
    
    .doc-badge {
        display: inline-block;
        padding: 3px 8px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 500;
    }
    
    .doc-rendicion {
        background-color: rgba(30, 136, 229, 0.1);
        color: #1e88e5;
    }
    
    .doc-orden_compra {
        background-color: rgba(255, 167, 38, 0.1);
        color: #ffa726;
    }
    
    .doc-usuario {
        background-color: rgba(102, 187, 106, 0.1);
        color: #66bb6a;
    }
    
    .doc-sistema {
        background-color: rgba(96, 125, 139, 0.1);
        color: #607d8b;
    }
    
    .filtros-container {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 20px;
    }
    
    .filtro-item {
        flex: 1;
        min-width: 200px;
    }

    .paginacion {
        display: flex;
        justify-content: center;
        margin-top: 20px;
        gap: 5px;
    }
    
    .paginacion .page-item {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 36px;
        height: 36px;
        border-radius: 4px;
        background-color: white;
        color: #333;
        text-decoration: none;
        transition: all 0.2s;
    }
    
    .paginacion .page-item:hover {
        background-color: #f0f0f0;
    }
    
    .paginacion .page-item.active {
        background-color: #1e88e5;
        color: white;
    }
    
    .paginacion .page-item.disabled {
        color: #ccc;
        pointer-events: none;
    }
    
    /* Estilos responsivos */
    @media (max-width: 768px) {
        .filtro-item {
            min-width: 100%;
        }
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<div class="card">
    <div class="panel-header">
        <h2 class="panel-title">Historial de Actividades</h2>
        <span>
            Total: <?php echo $total_registros; ?> registros encontrados
        </span>
    </div>
    
    <!-- Filtros de búsqueda -->
    <div class="card" style="margin-bottom: 20px;">
        <div class="panel-header">
            <h3 class="panel-title">Filtros de Búsqueda</h3>
        </div>
        
        <form method="GET" action="historial.php" class="form-inline">
            <div class="filtros-container">
                <div class="filtro-item">
                    <label for="tipo" class="form-label">Tipo de Actividad</label>
                    <select id="tipo" name="tipo" class="form-control">
                        <option value="">Todos</option>
                        <?php foreach ($tipos_actividad as $tipo_act): ?>
                            <option value="<?php echo $tipo_act; ?>" <?php echo $tipo === $tipo_act ? 'selected' : ''; ?>>
                                <?php echo ucfirst($tipo_act); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filtro-item">
                    <label for="documento_tipo" class="form-label">Tipo de Documento</label>
                    <select id="documento_tipo" name="documento_tipo" class="form-control">
                        <option value="">Todos</option>
                        <?php foreach ($tipos_documento as $tipo_doc): ?>
                            <option value="<?php echo $tipo_doc; ?>" <?php echo $documento_tipo === $tipo_doc ? 'selected' : ''; ?>>
                                <?php 
                                    // Formatear nombre de tipo de documento
                                    $doc_name = str_replace('_', ' ', $tipo_doc);
                                    echo ucfirst($doc_name);
                                ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filtro-item">
                    <label for="fecha_desde" class="form-label">Fecha Desde</label>
                    <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" 
                           value="<?php echo $fecha_desde; ?>">
                </div>
                
                <div class="filtro-item">
                    <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                    <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" 
                           value="<?php echo $fecha_hasta; ?>">
                </div>
                
                <?php if ($is_admin): ?>
                <div class="filtro-item">
                    <label for="usuario_id" class="form-label">Usuario</label>
                    <select id="usuario_id" name="usuario_id" class="form-control">
                        <option value="0">Todos</option>
                        <?php foreach ($usuarios as $usuario): ?>
                            <option value="<?php echo $usuario['id']; ?>" <?php echo $usuario_id == $usuario['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <div class="filtro-item">
                    <label for="busqueda" class="form-label">Búsqueda</label>
                    <input type="text" id="busqueda" name="busqueda" class="form-control" 
                           placeholder="Descripción, usuario..." value="<?php echo htmlspecialchars($busqueda); ?>">
                </div>
                
                <div style="display: flex; align-items: flex-end; gap: 10px;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">search</i> Buscar
                    </button>
                    
                    <a href="historial.php" class="btn btn-danger">
                        <i class="material-icons">clear</i> Limpiar
                    </a>
                </div>
            </div>
        </form>
    </div>
    
    <!-- Tabla de resultados -->
    <div class="table-responsive">
        <table class="historial-table">
            <thead>
                <tr>
                    <th>Fecha y Hora</th>
                    <th>Usuario</th>
                    <th>Tipo</th>
                    <th>Descripción</th>
                    <th>Documento</th>
                    <th>IP</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($actividades)): ?>
                <tr>
                    <td colspan="6" style="text-align: center;">No se encontraron registros</td>
                </tr>
                <?php else: ?>
                    <?php foreach ($actividades as $actividad): ?>
                    <tr>
                        <td><?php echo date('d/m/Y H:i:s', strtotime($actividad['fecha_hora'])); ?></td>
                        <td>
                            <?php echo htmlspecialchars($actividad['nombre'] . ' ' . $actividad['apellido']); ?>
                        </td>
                        <td>
                            <span class="tipo-badge tipo-<?php echo $actividad['tipo']; ?>">
                                <?php echo ucfirst($actividad['tipo']); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($actividad['descripcion']); ?></td>
                        <td>
                            <?php if ($actividad['documento_tipo'] && $actividad['documento_id']): ?>
                                <?php 
                                    $doc_link = '';
                                    switch ($actividad['documento_tipo']) {
                                        case 'rendicion':
                                            $doc_link = 'rendiciones.php?action=view&id=' . $actividad['documento_id'];
                                            break;
                                        case 'orden_compra':
                                            $doc_link = 'ordenes_compra.php?action=view&id=' . $actividad['documento_id'];
                                            break;
                                        case 'usuario':
                                            $doc_link = 'usuarios.php?action=edit&id=' . $actividad['documento_id'];
                                            break;
                                    }
                                    
                                    $doc_class = 'doc-' . ($actividad['documento_tipo'] ?: 'sistema');
                                    $doc_text = ucfirst(str_replace('_', ' ', $actividad['documento_tipo'])) . 
                                               ' #' . $actividad['documento_id'];
                                ?>
                                
                                <?php if ($doc_link): ?>
                                <a href="<?php echo $doc_link; ?>" class="doc-badge <?php echo $doc_class; ?>">
                                    <?php echo $doc_text; ?>
                                </a>
                                <?php else: ?>
                                <span class="doc-badge <?php echo $doc_class; ?>">
                                    <?php echo $doc_text; ?>
                                </span>
                                <?php endif; ?>
                            <?php else: ?>
                                <span class="doc-badge doc-sistema">Sistema</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo htmlspecialchars($actividad['ip_address']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Paginación -->
    <?php if ($total_paginas > 1): ?>
    <div class="paginacion">
        <?php
        // Construir la URL base para paginación (mantener filtros)
        $params = $_GET;
        unset($params['pagina']); // Quitar parámetro de página para reconstruirlo
        $url_base = 'historial.php?' . http_build_query($params) . '&pagina=';
        
        // Botón Anterior
        if ($pagina_actual > 1): ?>
            <a href="<?php echo $url_base . ($pagina_actual - 1); ?>" class="page-item">
                <i class="material-icons" style="font-size: 18px;">chevron_left</i>
            </a>
        <?php else: ?>
            <span class="page-item disabled">
                <i class="material-icons" style="font-size: 18px;">chevron_left</i>
            </span>
        <?php endif;
        
        // Páginas
        $start_page = max(1, $pagina_actual - 2);
        $end_page = min($total_paginas, $pagina_actual + 2);
        
        // Mostrar primera página si no está en el rango
        if ($start_page > 1): ?>
            <a href="<?php echo $url_base . '1'; ?>" class="page-item">1</a>
            <?php if ($start_page > 2): ?>
                <span class="page-item disabled">...</span>
            <?php endif;
        endif;
        
        // Páginas centrales
        for ($i = $start_page; $i <= $end_page; $i++): ?>
            <a href="<?php echo $url_base . $i; ?>" class="page-item <?php echo $i == $pagina_actual ? 'active' : ''; ?>">
                <?php echo $i; ?>
            </a>
        <?php endfor;
        
        // Mostrar última página si no está en el rango
        if ($end_page < $total_paginas): 
            if ($end_page < $total_paginas - 1): ?>
                <span class="page-item disabled">...</span>
            <?php endif; ?>
            <a href="<?php echo $url_base . $total_paginas; ?>" class="page-item"><?php echo $total_paginas; ?></a>
        <?php endif;
        
        // Botón Siguiente
        if ($pagina_actual < $total_paginas): ?>
            <a href="<?php echo $url_base . ($pagina_actual + 1); ?>" class="page-item">
                <i class="material-icons" style="font-size: 18px;">chevron_right</i>
            </a>
        <?php else: ?>
            <span class="page-item disabled">
                <i class="material-icons" style="font-size: 18px;">chevron_right</i>
            </span>
        <?php endif; ?>
    </div>
    <?php endif; ?>
</div>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>