<?php
/**
 * Generador de PDFs para órdenes de compra
 * Usando tFPDF para soporte nativo de UTF-8
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$log_file = 'pdf_debug.log';
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - PDF de Orden de Compra solicitado: id=' . ($_GET['id'] ?? 'no_definido') . "\n", FILE_APPEND);

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'tfpdf/tfpdf.php';

file_put_contents($log_file, date('Y-m-d H:i:s') . ' - tFPDF cargado correctamente' . "\n", FILE_APPEND);

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Sesión verificada: user_id=' . $user_id . ', is_admin=' . ($is_admin ? 'true' : 'false') . "\n", FILE_APPEND);

$id = $_GET['id'] ?? 0;
$save = isset($_GET['save']) ? true : false;
$redirect = isset($_GET['redirect']) ? $_GET['redirect'] : '';

if (empty($id)) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: ID no especificado' . "\n", FILE_APPEND);
    die('ID no especificado');
}

if (!file_exists('temp')) {
    mkdir('temp', 0777, true);
}

$company_name = 'COMERCIAL FIGARO LIMITADA';
$company_address = 'Puerta del Sol 180 Of.02';
$company_rut = '76.080.987-K';
$company_phone = '+562 32212511';
$company_email = 'CONTACTO@FIGAROCAFE.CL';
$company_business = 'Comercializadora de Café y otros Productos Gourmet';
$company_logo = 'assets/img/logo.png';

define('COMPANY_NAME', $company_name);
define('COMPANY_ADDRESS', $company_address);
define('COMPANY_RUT', $company_rut);
define('COMPANY_PHONE', $company_phone);
define('COMPANY_EMAIL', $company_email);
define('COMPANY_BUSINESS', $company_business);
define('COMPANY_LOGO', $company_logo);

class OrderCompPDF extends tFPDF {
    private $logo = null;
    public $titulo;
    public $codigo;
    public $footer_text;
    public $documento;

    function __construct($documento = [], $titulo = '', $codigo = '', $orientation = 'P', $unit = 'mm', $size = 'A4') {
        setlocale(LC_ALL, 'es_ES.UTF-8');
        parent::__construct($orientation, $unit, $size);
        $this->AddFont('DejaVu', '', 'DejaVuSans.ttf', true);
        $this->AddFont('DejaVu', 'B', 'DejaVuSans-Bold.ttf', true);
        $this->SetFont('DejaVu', '', 8);
        if (file_exists(COMPANY_LOGO)) {
            $this->logo = COMPANY_LOGO;
        }
        $this->footer_text = 'Documento generado por ' . APP_NAME . ' - ' . date('d/m/Y H:i:s');
        $this->SetMargins(10, 10, 10);
        $this->documento = $documento;
        $this->titulo = $titulo;
        $this->codigo = $codigo;
        $this->AddPage();
    }

    function RoundedRect($x, $y, $w, $h, $r, $corners = '1234', $style = '') {
        $k = $this->k;
        $hp = $this->h;
        if($style=='F')
            $op='f';
        elseif($style=='FD' || $style=='DF')
            $op='B';
        else
            $op='S';
        $MyArc = 4/3 * (sqrt(2) - 1);
        $this->_out(sprintf('%.2F %.2F m',($x+$r)*$k,($hp-$y)*$k ));

        $xc = $x+$w-$r;
        $yc = $y+$r;
        $this->_out(sprintf('%.2F %.2F l', $xc*$k,($hp-$y)*$k ));

        if (strpos($corners, '2')===false)
            $this->_out(sprintf('%.2F %.2F l', ($x+$w)*$k,($hp-$y)*$k ));
        else
            $this->_Arc($xc + $r*$MyArc, $yc - $r, $xc + $r, $yc - $r*$MyArc, $xc + $r, $yc);

        $xc = $x+$w-$r;
        $yc = $y+$h-$r;
        $this->_out(sprintf('%.2F %.2F l',($x+$w)*$k,($hp-($y+$h))*$k));

        if (strpos($corners, '3')===false)
            $this->_out(sprintf('%.2F %.2F l',($x+$w)*$k,($hp-($y+$h))*$k));
        else
            $this->_Arc($xc + $r, $yc + $r*$MyArc, $xc + $r*$MyArc, $yc + $r, $xc, $yc + $r);

        $xc = $x+$r;
        $yc = $y+$h-$r;
        $this->_out(sprintf('%.2F %.2F l',$xc*$k,($hp-($y+$h))*$k));

        if (strpos($corners, '4')===false)
            $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-($y+$h))*$k));
        else
            $this->_Arc($xc - $r*$MyArc, $yc + $r, $xc - $r, $yc + $r*$MyArc, $xc - $r, $yc);

        $xc = $x+$r ;
        $yc = $y+$r;
        $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-$yc)*$k ));

        if (strpos($corners, '1')===false)
        {
            $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-$y)*$k ));
            $this->_out(sprintf('%.2F %.2F l',($x+$r)*$k,($hp-$y)*$k ));
        }
        else
            $this->_Arc($xc - $r, $yc - $r*$MyArc, $xc - $r*$MyArc, $yc - $r, $xc, $yc - $r);
        $this->_out($op);
    }

    function _Arc($x1, $y1, $x2, $y2, $x3, $y3) {
        $h = $this->h;
        $this->_out(sprintf('%.2F %.2F %.2F %.2F %.2F %.2F c ', 
            $x1*$this->k, ($h-$y1)*$this->k,
            $x2*$this->k, ($h-$y2)*$this->k,
            $x3*$this->k, ($h-$y3)*$this->k));
    }

    function Header() {
        // Establecer colores corporativos
        $this->SetDrawColor(30, 136, 229);
        $this->SetFillColor(240, 248, 255);
        
        $y_initial = $this->GetY();
        if ($this->logo) {
            $this->Image($this->logo, (210 - 40) / 2, 10, 40); // Logo más pequeño
            $this->SetY($y_initial + 25); // Ajustar posición después del logo
        }

        // Información de la empresa con formato más compacto
        $this->SetFont('DejaVu', 'B', 10);
        $this->SetTextColor(30, 82, 140);
        $this->Cell(0, 5, COMPANY_NAME, 0, 1, 'C');

        // Información adicional de la empresa más compacta
        $this->SetFont('DejaVu', '', 7);
        $this->SetTextColor(80, 80, 80);
        $this->Cell(0, 3, 'RUT: ' . COMPANY_RUT . ' | ' . COMPANY_ADDRESS, 0, 1, 'C');
        $this->Cell(0, 3, 'Tel: ' . COMPANY_PHONE . ' | Email: ' . COMPANY_EMAIL, 0, 1, 'C');
        $this->Cell(0, 3, 'Giro: ' . COMPANY_BUSINESS, 0, 1, 'C');
        
        // Espacio en blanco después de la información de la empresa
        $this->Ln(5);
        
        // Título del documento
        $this->SetFont('DejaVu', 'B', 14);
        $this->SetTextColor(30, 82, 140);
        $this->Cell(0, 10, $this->titulo, 0, 1, 'C');
        
        // Código y estado de la orden
        $this->SetFont('DejaVu', 'B', 9);
        $this->SetFillColor(30, 136, 229);
        $this->SetTextColor(255, 255, 255);
        
        // Código en una celda con fondo azul
        $this->Cell(50, 6, 'CÓDIGO: ' . $this->codigo, 1, 0, 'C', true);
        
        // Estado en otra celda con fondo según el estado
        $estado_color = '#FFC107'; // Color por defecto (amarillo)
        $estado_nombre = 'Pendiente'; // Estado por defecto

        if (isset($this->documento['estado_nombre'])) {
            $estado_nombre = $this->documento['estado_nombre'];
        }
        
        if (isset($this->documento['estado_id'])) {
            switch($this->documento['estado_id']) {
                case 1: // Enviada
                    $estado_color = '#FFC107'; // Amarillo
                    break;
                case 2: // En revisión
                    $estado_color = '#3498DB'; // Azul
                    break;
                case 3: // Aprobada
                    $estado_color = '#28A745'; // Verde
                    break;
                case 4: // Rechazada
                    $estado_color = '#DC3545'; // Rojo
                    break;
                case 5: // Pagada
                    $estado_color = '#6C757D'; // Gris
                    break;
                default:
                    $estado_color = '#6C757D'; // Gris por defecto
            }
        }
        
        // Convertir color hexadecimal a RGB
        if (preg_match('/^#[a-fA-F0-9]{6}$/', $estado_color)) {
            $r = hexdec(substr($estado_color, 1, 2));
            $g = hexdec(substr($estado_color, 3, 2));
            $b = hexdec(substr($estado_color, 5, 2));
        } else {
            // Color por defecto si el formato es inválido
            $r = 108; // #6C
            $g = 117; // #75
            $b = 125; // #7D
        }
        
        $this->SetFillColor($r, $g, $b);
        $this->Cell(50, 6, 'ESTADO: ' . strtoupper($estado_nombre), 1, 0, 'C', true);
        
        // Fecha en otra celda
        $this->SetTextColor(0, 0, 0);
        $this->SetFillColor(245, 245, 245);
        $this->Cell(0, 6, 'FECHA: ' . date('d/m/Y', strtotime($this->documento['fecha'])), 1, 1, 'C', true);
        
        $this->Ln(5);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetDrawColor(30, 136, 229); // Azul para la línea
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        $this->Ln(1);
        $this->SetFont('DejaVu', '', 8);
        $this->SetTextColor(80, 80, 80); // Gris oscuro para el pie de página
        $this->Cell(0, 10, $this->footer_text, 0, 0, 'C');
        $this->Cell(0, 10, 'Página ' . $this->PageNo() . ' de {nb}', 0, 0, 'R');
    }

    function FormatoMoneda($monto) {
        return '$' . number_format($monto, 0, ',', '.');
    }

    function GenerateOrderContent($detalles) {
        // Configuración inicial
        $this->SetFont('DejaVu', '', 9);
        $left_x = $this->GetX();
        $page_width = $this->GetPageWidth();
        $col_width = ($page_width - 20) / 2 - 1;
        
        // Información del proveedor y de la orden en dos columnas
        $this->SetFillColor(30, 136, 229);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('DejaVu', 'B', 7);
        
        // Títulos de secciones
        $this->Cell($col_width, 6, 'INFORMACIÓN DEL PROVEEDOR', 1, 0, 'C', true);
        $this->Cell(2, 6, '', 0, 0); // Espacio entre columnas
        $this->Cell($col_width, 6, 'INFORMACIÓN DE LA ORDEN', 1, 1, 'C', true);
        
        // Restablecer colores para el contenido
        $this->SetTextColor(0, 0, 0);
        $this->SetFont('DejaVu', '', 8);
        
        // Datos del proveedor
        $this->SetFillColor(245, 245, 245);
        $y_start = $this->GetY();
        
        // Proveedor - Primera columna
        $this->Cell(30, 6, 'Razón Social:', 1, 0, 'L', true);
        $this->Cell($col_width - 30, 6, $this->documento['proveedor'], 1, 1, 'L');
        
        $this->Cell(30, 6, 'RUT:', 1, 0, 'L', true);
        $this->Cell($col_width - 30, 6, $this->documento['rut_proveedor'], 1, 1, 'L');
        
        if (!empty($this->documento['direccion'])) {
            $this->Cell(30, 6, 'Dirección:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['direccion'], 1, 1, 'L');
        }
        
        if (!empty($this->documento['telefono'])) {
            $this->Cell(30, 6, 'Teléfono:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['telefono'], 1, 1, 'L');
        }
        
        if (!empty($this->documento['email'])) {
            $this->Cell(30, 6, 'Email:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['email'], 1, 1, 'L');
        }
        
        if (!empty($this->documento['contacto'])) {
            $this->Cell(30, 6, 'Contacto:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['contacto'], 1, 1, 'L');
        }
        
        if (!empty($this->documento['giro'])) {
            $this->Cell(30, 6, 'Giro:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['giro'], 1, 1, 'L');
        }
        
        // Guardar altura final del bloque de proveedor
        $proveedor_height = $this->GetY() - $y_start;
        
        // Restablecer posición para datos de la orden
        $this->SetY($y_start);
        $this->SetX($left_x + $col_width + 2);
        
        // Datos de la orden - Segunda columna
        $this->Cell(30, 6, 'N° Documento:', 1, 0, 'L', true);
        $this->Cell($col_width - 30, 6, $this->documento['numero_documento'], 1, 1, 'L');
        $this->SetX($left_x + $col_width + 2);
        
        $this->Cell(30, 6, 'Solicitante:', 1, 0, 'L', true);
        $this->Cell($col_width - 30, 6, $this->documento['nombre'] . ' ' . $this->documento['apellido'], 1, 1, 'L');
        $this->SetX($left_x + $col_width + 2);
        
        if (!empty($this->documento['cargo'])) {
            $this->Cell(30, 6, 'Cargo:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['cargo'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        if (!empty($this->documento['area_solicitud'])) {
            $this->Cell(30, 6, 'Área:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['area_solicitud'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        if (!empty($this->documento['motivo_orden'])) {
            $this->Cell(30, 6, 'Motivo:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['motivo_orden'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        if (!empty($this->documento['direccion_despacho'])) {
            $this->Cell(30, 6, 'Despacho:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['direccion_despacho'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        if (!empty($this->documento['tiempo_entrega'])) {
            $this->Cell(30, 6, 'Tiempo Entrega:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['tiempo_entrega'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        if (!empty($this->documento['condiciones_pago'])) {
            $this->Cell(30, 6, 'Condiciones:', 1, 0, 'L', true);
            $this->Cell($col_width - 30, 6, $this->documento['condiciones_pago'], 1, 1, 'L');
            $this->SetX($left_x + $col_width + 2);
        }
        
        // Calcular cuál sección es más alta para ajustar
        $orden_height = $this->GetY() - $y_start;
        $max_height = max($proveedor_height, $orden_height);
        
        // Ajustar la posición Y al final del contenido más alto
        $this->SetY($y_start + $max_height + 5);
        
        // Título para la tabla de productos
        $this->SetFillColor(30, 136, 229);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('DejaVu', 'B', 7);
        $this->Cell(0, 6, 'DETALLE DE PRODUCTOS', 1, 1, 'C', true);
        
        // Encabezados de la tabla de productos
        $this->SetFont('DejaVu', '', 7);
        $w = array(10, 80, 15, 20, 15, 20, 30);
        $headers = array('N°', 'Producto/Servicio', 'Cant.', 'Precio Unit.', 'Desc. %', 'Precio Final', 'Subtotal');
        
        $this->SetFillColor(245, 245, 245);
        $this->SetTextColor(0, 0, 0);
        
        // Dibujar encabezados
        for($i = 0; $i < count($headers); $i++) {
            $this->Cell($w[$i], 6, $headers[$i], 1, 0, 'C', true);
        }
        $this->Ln();
        
        // Variables para el cálculo del total
        $subtotal_neto = 0;
        $total_descuentos = 0;
        $total_iva = 0;
        $total_bruto = 0;
        $iva_tasa = 0.19; // 19% IVA
        
        // Datos de productos
        $this->SetFont('DejaVu', '', 7);
        $fill = false;
        
        foreach($detalles as $i => $item) {
            $cantidad = $item['cantidad'];
            $precio = $item['precio_unitario'];
            $descuento = $item['descuento'] ?? 0;
            $precio_final = $precio * (1 - ($descuento / 100));
            $subtotal = $cantidad * $precio_final;
            
            // Acumular totales
            $subtotal_neto += $subtotal;
            $total_descuentos += ($precio - $precio_final) * $cantidad;
            
            // Formatear datos para mostrar
            $precio_str = $this->FormatoMoneda($precio);
            $precio_final_str = $this->FormatoMoneda($precio_final);
            $subtotal_str = $this->FormatoMoneda($subtotal);
            
            // Imprimir fila
            $this->Cell($w[0], 6, $i + 1, 1, 0, 'C', $fill);
            
            // Verificar si el texto es muy largo y usar MultiCell si es necesario
            $producto_texto = $item['producto'];
            $x = $this->GetX();
            $y = $this->GetY();
            
            if (strlen($producto_texto) > 50) {
                $this->Cell($w[1], 6, '', 1, 0, 'L', $fill);
                $this->SetXY($x, $y);
                $this->MultiCell($w[1], 3, $producto_texto, 0, 'L');
                $this->SetXY($x + $w[1], $y);
            } else {
                $this->Cell($w[1], 6, $producto_texto, 1, 0, 'L', $fill);
            }
            
            $this->Cell($w[2], 6, $cantidad, 1, 0, 'C', $fill);
            $this->Cell($w[3], 6, $precio_str, 1, 0, 'R', $fill);
            $this->Cell($w[4], 6, $descuento . '%', 1, 0, 'C', $fill);
            $this->Cell($w[5], 6, $precio_final_str, 1, 0, 'R', $fill);
            $this->Cell($w[6], 6, $subtotal_str, 1, 1, 'R', $fill);
            
            $fill = !$fill;
        }
        
        // Calcular IVA y total bruto
        $total_iva = $subtotal_neto * $iva_tasa;
        $total_bruto = $subtotal_neto + $total_iva;
        
        // Espacio para la sección de totales
        $this->Ln(5);
        
        // Sección de totales
        $this->SetFillColor(245, 245, 245);
        $this->SetFont('DejaVu', 'B', 8);
        
        // Ancho de la sección de totales
        $total_width = 80;
        $label_width = 50;
        $value_width = $total_width - $label_width;
        $x_pos = $this->GetPageWidth() - $total_width - 10;
        
        // Subtotal Neto
        $this->SetX($x_pos);
        $this->Cell($label_width, 6, 'Subtotal Neto:', 1, 0, 'R', true);
        $this->Cell($value_width, 6, $this->FormatoMoneda($subtotal_neto), 1, 1, 'R', true);
        
        // Total Descuentos (si hay)
        if ($total_descuentos > 0) {
            $this->SetX($x_pos);
            $this->Cell($label_width, 6, 'Total Descuentos:', 1, 0, 'R', true);
            $this->Cell($value_width, 6, $this->FormatoMoneda($total_descuentos), 1, 1, 'R', true);
        }
        
        // IVA
        $this->SetX($x_pos);
        $this->Cell($label_width, 6, 'IVA (19%):', 1, 0, 'R', true);
        $this->Cell($value_width, 6, $this->FormatoMoneda($total_iva), 1, 1, 'R', true);
        
        // Total Bruto
        $this->SetX($x_pos);
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell($label_width, 7, 'TOTAL:', 1, 0, 'R', true);
        $this->Cell($value_width, 7, $this->FormatoMoneda($total_bruto), 1, 1, 'R', true);
        
        // Observaciones (si hay)
        if (!empty($this->documento['descripcion'])) {
            $this->Ln(5);
            $this->SetFont('DejaVu', 'B', 8);
            $this->Cell(0, 6, 'OBSERVACIONES:', 0, 1);
            $this->SetFont('DejaVu', '', 8);
            $this->MultiCell(0, 5, $this->documento['descripcion']);
        }
        
        // Firmas
        $this->Ln(15);
        $this->SetFont('DejaVu', '', 8);
        
        $this->Cell(90, 5, '____________________________', 0, 0, 'C');
        $this->Cell(90, 5, '____________________________', 0, 1, 'C');
        
        $this->Cell(90, 5, $this->documento['nombre'] . ' ' . $this->documento['apellido'], 0, 0, 'C');
        
        if ($this->documento['estado_id'] == 3 && !empty($this->documento['revisor_nombre'])) {
            $this->Cell(90, 5, $this->documento['revisor_nombre'] . ' ' . $this->documento['revisor_apellido'], 0, 1, 'C');
        } else {
            $this->Cell(90, 5, 'Firma Autorizada', 0, 1, 'C');
        }
        
        $this->Cell(90, 5, 'Solicitante', 0, 0, 'C');
        $this->Cell(90, 5, 'Aprobador', 0, 1, 'C');
    }
}

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Conexión a la base de datos establecida' . "\n", FILE_APPEND);

    // Obtener datos de la orden de compra
    $stmt = $conn->prepare("
        SELECT oc.*, 
               u.nombre, u.apellido, u.rut, u.cargo,
               p.direccion, p.telefono, p.email, p.contacto, p.giro,
               e.nombre as estado_nombre,
               rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
        FROM ordenes_compra oc
        JOIN users u ON oc.user_id = u.id
        LEFT JOIN proveedores p ON oc.proveedor_id = p.id
        LEFT JOIN estados e ON oc.estado_id = e.id
        LEFT JOIN users rev ON oc.revisor_id = rev.id
        WHERE oc.id = :id
    ");
    
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    $orden = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$orden) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Orden de compra no encontrada para id=' . $id . "\n", FILE_APPEND);
        die('Orden de compra no encontrada');
    }
    
    // Verificar permisos
    if (!$is_admin && $orden['user_id'] != $user_id) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: No tiene permisos para ver esta orden. user_id=' . $user_id . ', orden_user_id=' . $orden['user_id'] . "\n", FILE_APPEND);
        die('No tiene permisos para ver esta orden de compra');
    }
    
    // Obtener detalles de la orden
    $stmt = $conn->prepare("SELECT * FROM detalle_orden_compra WHERE orden_compra_id = :id ORDER BY id");
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Generar PDF
    $codigo = generarCodigo('OC', $id);
    $titulo = 'ORDEN DE COMPRA';
    
    $pdf = new OrderCompPDF($orden, $titulo, $codigo);
    $pdf->AliasNbPages();
    $pdf->GenerateOrderContent($detalles);
    
    // Registrar actividad
    registrarActividad($conn, $user_id, 'descargar', 'Descarga de PDF: ' . $codigo, $id, 'orden_compra');
    
    // Generar archivo
    $file_name = $codigo . '.pdf';
    if ($save) {
        $file_path = 'temp/' . $file_name;
        $pdf->Output($file_path, 'F');
        if (!empty($redirect)) {
            header('Location: ' . $redirect);
            exit;
        }
        echo "PDF guardado exitosamente en: " . $file_path;
    } else {
        $pdf->Output($file_name, 'I');
    }
    
} catch (PDOException $e) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - ERROR PDO: ' . $e->getMessage() . "\n", FILE_APPEND);
    die('Error de conexión a la base de datos: ' . $e->getMessage());
} catch (Exception $e) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - ERROR General: ' . $e->getMessage() . "\n", FILE_APPEND);
    die('Error general: ' . $e->getMessage());
}