<?php
/**
 * Script actualizado para sincronizar el stock y descripciones desde Softland
 * 
 * Este script usa los métodos ObtenerStockPorBodega y ObtenerProductosPorCodigoV2
 * para obtener stock y descripciones de productos.
 */

// Mostrar todos los errores
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Aumentar el tiempo límite de ejecución
set_time_limit(300);

// Variables globales para estadísticas
global $STATS_total_repuestos, $STATS_actualizados_stock, $STATS_actualizados_ean, 
       $STATS_actualizados_ambos, $STATS_sin_cambios, $STATS_errores,
       $STATS_ean_encontrados, $STATS_ean_no_encontrados;

// Configuración de acceso a Softland
$softland_api_url_producto = 'http://201.236.190.215/ECOMMERCE/wsproducto.asmx';
$softland_username = 'STORE';
$softland_password = 'softland';
$softland_empresa = 'FIGARO';
$softland_codbode = '32';

// Configuración de acceso a la base de datos
$db_host = 'localhost';
$db_user = 'santavictoria_s3rv1c10';
$db_pass = 'd.=[YeZkc}]h';
$db_name = 'santavictoria_t0l1g3';

// Configuración de zona horaria
date_default_timezone_set('America/Santiago');

echo "<h2>Actualización de Stock y Descripciones desde Softland</h2>";
echo "<pre>";

// Crear directorio de logs si no existe
$logs_dir = __DIR__ . '/logs';
if (!is_dir($logs_dir)) {
    if (!mkdir($logs_dir, 0755, true)) {
        echo "ERROR: No se pudo crear el directorio de logs\n";
        exit(1);
    }
}

// Archivo de log
$log_file = $logs_dir . '/actualizacion_stock_desc_' . date('Y-m-d_H-i-s') . '.log';

// Función para registrar en log
function log_message($message) {
    global $log_file;
    $time = date('Y-m-d H:i:s');
    $entry = "[$time] $message" . PHP_EOL;
    file_put_contents($log_file, $entry, FILE_APPEND);
    echo $entry;
}

// Iniciar log
log_message("=== INICIANDO ACTUALIZACIÓN DE STOCK Y DESCRIPCIONES ===");
log_message("Archivo de log: $log_file");

// Conectar a la base de datos
try {
    log_message("Conectando a la base de datos MySQL...");
    $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
    
    if ($conn->connect_error) {
        throw new Exception("Error de conexión a la base de datos: " . $conn->connect_error);
    }
    
    log_message("Conexión a la base de datos establecida correctamente");
    
    // Verificar si existe la columna ubicacion
    $result = $conn->query("SHOW COLUMNS FROM repuestos LIKE 'ubicacion'");
    if ($result->num_rows == 0) {
        log_message("La columna 'ubicacion' no existe en la tabla repuestos. Creándola...");
        
        if ($conn->query("ALTER TABLE repuestos ADD COLUMN ubicacion VARCHAR(100) DEFAULT NULL")) {
            log_message("Columna 'ubicacion' creada correctamente");
        } else {
            log_message("ERROR: No se pudo crear la columna 'ubicacion': " . $conn->error);
        }
    }
    
    // Consultar repuestos activos
    log_message("Consultando repuestos activos...");
    $sql = "SELECT id, codigo, stock, ubicacion FROM repuestos WHERE activo = 1";
    $result = $conn->query($sql);
    
    if (!$result) {
        throw new Exception("Error al consultar repuestos: " . $conn->error);
    }
    
    $total_repuestos = $result->num_rows;
    $STATS_total_repuestos = $total_repuestos;
    log_message("Se encontraron $total_repuestos repuestos activos para actualizar");
    
    $actualizados_stock = 0;
    $actualizados_ean = 0;
    $actualizados_ambos = 0;
    $errores = 0;
    $sin_cambios = 0;
    $ean_encontrados = 0;
    $ean_no_encontrados = 0;
    
    // Procesar cada repuesto
    while ($repuesto = $result->fetch_assoc()) {
        $id = $repuesto['id'];
        $codigo = $repuesto['codigo'];
        $stock_actual = $repuesto['stock'];
        $ubicacion_actual = $repuesto['ubicacion'];
        
        log_message("\nProcesando repuesto ID: $id, Código: $codigo, Stock actual: $stock_actual, Descripción actual: " . ($ubicacion_actual ?: 'No definida'));
        
        try {
            // Obtener stock desde Softland
            $stock_softland = obtenerStockDesdeSoftland($codigo);
            
            // Obtener descripción desde Softland
            $descripcion_softland = obtenerDescripcionDesdeSoftland($codigo);
            
            $hay_cambios = false;
            $cambio_stock = false;
            $cambio_desc = false;
            
            // Verificar si hay cambios en el stock
            if ($stock_softland !== false && $stock_actual != $stock_softland) {
                log_message("Stock obtenido desde Softland: $stock_softland (diferente al actual: $stock_actual)");
                $hay_cambios = true;
                $cambio_stock = true;
            } else if ($stock_softland === false) {
                log_message("Error al obtener stock para el código: $codigo");
            } else {
                log_message("Stock obtenido desde Softland: $stock_softland (sin cambios)");
            }
            
            // Verificar si hay cambios en la descripción
            if ($descripcion_softland !== false) {
                $ean_encontrados++;
                $STATS_ean_encontrados = $ean_encontrados;
                
                if ($ubicacion_actual != $descripcion_softland) {
                    log_message("Descripción obtenida desde Softland: $descripcion_softland (diferente a la actual: " . ($ubicacion_actual ?: 'No definida') . ")");
                    $hay_cambios = true;
                    $cambio_desc = true;
                } else {
                    log_message("Descripción obtenida desde Softland: $descripcion_softland (sin cambios)");
                }
            } else {
                $ean_no_encontrados++;
                $STATS_ean_no_encontrados = $ean_no_encontrados;
                log_message("No se encontró descripción para el producto: $codigo");
            }
            
            // Actualizar solo si hay cambios
            if ($hay_cambios) {
                // Calcular disponibilidad basada en stock
                $disponible = $stock_softland > 0 ? 1 : 0;
                
                // Construir la consulta SQL según los campos a actualizar
                if ($cambio_stock && $cambio_desc) {
                    $stmt = $conn->prepare("UPDATE repuestos SET stock = ?, ubicacion = ?, disponible = ? WHERE id = ?");
                    $stmt->bind_param("isii", $stock_softland, $descripcion_softland, $disponible, $id);
                    $actualizados_ambos++;
                    $STATS_actualizados_ambos = $actualizados_ambos;
                    log_message("Actualizando stock y descripción");
                } else if ($cambio_stock) {
                    $stmt = $conn->prepare("UPDATE repuestos SET stock = ?, disponible = ? WHERE id = ?");
                    $stmt->bind_param("iii", $stock_softland, $disponible, $id);
                    $actualizados_stock++;
                    $STATS_actualizados_stock = $actualizados_stock;
                    log_message("Actualizando solo stock");
                } else if ($cambio_desc) {
                    $stmt = $conn->prepare("UPDATE repuestos SET ubicacion = ? WHERE id = ?");
                    $stmt->bind_param("si", $descripcion_softland, $id);
                    $actualizados_ean++;
                    $STATS_actualizados_ean = $actualizados_ean;
                    log_message("Actualizando solo descripción");
                }
                
                if (isset($stmt)) {
                    if ($stmt->execute()) {
                        log_message("Actualización realizada correctamente");
                    } else {
                        $errores++;
                        $STATS_errores = $errores;
                        log_message("ERROR al actualizar en la base de datos: " . $stmt->error);
                    }
                    
                    $stmt->close();
                }
            } else {
                $sin_cambios++;
                $STATS_sin_cambios = $sin_cambios;
                log_message("No hay cambios que realizar");
            }
        } catch (Exception $e) {
            $errores++;
            $STATS_errores = $errores;
            log_message("ERROR al procesar repuesto: " . $e->getMessage());
        }
    }
    
    // Registrar la actividad en el historial
    try {
        $user_id = 1; // Asumimos que es el administrador
        $fecha_actual = date('Y-m-d H:i:s');
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
        $descripcion = "Actualización automática desde Softland. Repuestos actualizados: " . 
                      ($actualizados_stock + $actualizados_ean + $actualizados_ambos);
        
        $sql = "INSERT INTO historial_cambios (user_id, tipo, descripcion, documento_tipo, fecha_hora, ip_address) 
                VALUES (?, 'sistema', ?, 'sistema', ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("isss", $user_id, $descripcion, $fecha_actual, $ip_address);
        $stmt->execute();
        $stmt->close();
    } catch (Exception $e) {
        log_message("ERROR al registrar actividad: " . $e->getMessage());
    }
    
    // Resumen final
    log_message("\n=== RESUMEN DE LA ACTUALIZACIÓN ===");
    log_message("Total de repuestos procesados: $total_repuestos");
    log_message("Repuestos con stock actualizado: $actualizados_stock");
    log_message("Repuestos con descripción actualizada: $actualizados_ean");
    log_message("Repuestos con ambos valores actualizados: $actualizados_ambos");
    log_message("Descripciones encontradas: $ean_encontrados");
    log_message("Descripciones no encontradas: $ean_no_encontrados");
    log_message("Repuestos sin cambios: $sin_cambios");
    log_message("Errores encontrados: $errores");
    
    $conn->close();
    log_message("Conexión a la base de datos cerrada");
    
} catch (Exception $e) {
    log_message("ERROR CRÍTICO: " . $e->getMessage());
    $STATS_errores++;
}

log_message("=== PROCESO FINALIZADO ===");
echo "</pre>";

/**
 * Obtiene el stock de un producto desde Softland usando ObtenerStockPorBodega
 * 
 * @param string $codigo Código del producto
 * @return int|bool Stock disponible o false si hay error
 */
function obtenerStockDesdeSoftland($codigo) {
    global $softland_api_url_producto, $softland_username, $softland_password, $softland_empresa, $softland_codbode;
    
    // Crear solicitud SOAP para ObtenerStockPorBodega
    $xml_request = '<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                   xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                   xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
      <soap:Header>
        <AuthHeader xmlns="http://softland.cl/">
          <Username>' . $softland_username . '</Username>
          <Password>' . $softland_password . '</Password>
        </AuthHeader>
      </soap:Header>
      <soap:Body>
        <ObtenerStockPorBodega xmlns="http://softland.cl/">
          <codprod>' . $codigo . '</codprod>
          <Empresa>' . $softland_empresa . '</Empresa>
          <Token></Token>
          <CodBode>' . $softland_codbode . '</CodBode>
          <Talla></Talla>
          <Color></Color>
        </ObtenerStockPorBodega>
      </soap:Body>
    </soap:Envelope>';
    
    // Configurar cURL para la solicitud SOAP
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $softland_api_url_producto);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $xml_request);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: text/xml; charset=utf-8',
        'SOAPAction: "http://softland.cl/ObtenerStockPorBodega"',
        'Content-Length: ' . strlen($xml_request)
    ));
    
    // Establecer timeout para evitar bloqueos
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Ejecutar la solicitud
    $response = curl_exec($ch);
    
    // Verificar errores
    if (curl_errno($ch)) {
        log_message("Error en cURL: " . curl_error($ch));
        curl_close($ch);
        return false;
    }
    
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($http_code != 200) {
        log_message("Error HTTP $http_code al conectar con Softland");
        curl_close($ch);
        return false;
    }
    
    curl_close($ch);
    
    // Procesar la respuesta XML
    libxml_use_internal_errors(true);
    $xml = simplexml_load_string($response);
    
    if ($xml === false) {
        $errors = libxml_get_errors();
        $error_msg = "Errores al parsear XML: ";
        foreach ($errors as $error) {
            $error_msg .= $error->message . " ";
        }
        libxml_clear_errors();
        log_message($error_msg);
        return false;
    }
    
    // Extraer el valor de stock disponible
    try {
        // Definir los namespaces
        $soap_ns = 'http://schemas.xmlsoap.org/soap/envelope/';
        $softland_ns = 'http://softland.cl/';
        
        // Extraer el nodo soap:Body
        $body = $xml->children($soap_ns)->Body;
        
        // Extraer la respuesta
        $response_node = $body->children($softland_ns)->ObtenerStockPorBodegaResponse;
        $result_node = $response_node->ObtenerStockPorBodegaResult;
        
        // Verificar si hay stock disponible
        if (isset($result_node->stock->stock->disponible)) {
            return (int) $result_node->stock->stock->disponible;
        } else {
            log_message("No se encontró información de stock para este producto");
            return 0; // Si no hay información, asumir stock 0
        }
    } catch (Exception $e) {
        log_message("Error al procesar la respuesta: " . $e->getMessage());
        return false;
    }
}

/**
 * Obtiene la descripción de un producto desde Softland usando ObtenerProductosPorCodigoV2
 * 
 * @param string $codigo Código del producto
 * @return string|bool Descripción o false si hay error
 */
function obtenerDescripcionDesdeSoftland($codigo) {
    global $softland_api_url_producto, $softland_username, $softland_password, $softland_empresa;
    
    // Crear solicitud SOAP para ObtenerProductosPorCodigoV2 (nuevo método)
    $xml_request = '<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                   xmlns:xsd="http://www.w3.org/2001/XMLSchema"
                   xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
      <soap:Header>
        <AuthHeader xmlns="http://softland.cl/">
          <Username>' . $softland_username . '</Username>
          <Password>' . $softland_password . '</Password>
        </AuthHeader>
      </soap:Header>
      <soap:Body>
        <ObtenerProductosPorCodigoV2 xmlns="http://softland.cl/">
          <codigo>' . $codigo . '</codigo>
          <Empresa>' . $softland_empresa . '</Empresa>
          <Token></Token>
          <SoloActivos>0</SoloActivos>
          <EsParaVenta>0</EsParaVenta>
        </ObtenerProductosPorCodigoV2>
      </soap:Body>
    </soap:Envelope>';
    
    // Configurar cURL para la solicitud SOAP
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $softland_api_url_producto);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $xml_request);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: text/xml; charset=utf-8',
        'SOAPAction: "http://softland.cl/ObtenerProductosPorCodigoV2"',
        'Content-Length: ' . strlen($xml_request)
    ));
    
    // Establecer timeout para evitar bloqueos
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Ejecutar la solicitud
    $response = curl_exec($ch);
    
    // Verificar errores
    if (curl_errno($ch)) {
        log_message("Error en cURL (Descripción): " . curl_error($ch));
        curl_close($ch);
        return false;
    }
    
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($http_code != 200) {
        log_message("Error HTTP $http_code al conectar con Softland (Descripción)");
        curl_close($ch);
        return false;
    }
    
    curl_close($ch);
    
    // Procesar la respuesta XML
    libxml_use_internal_errors(true);
    $xml = simplexml_load_string($response);
    
    if ($xml === false) {
        $errors = libxml_get_errors();
        $error_msg = "Errores al parsear XML (Descripción): ";
        foreach ($errors as $error) {
            $error_msg .= $error->message . " ";
        }
        libxml_clear_errors();
        log_message($error_msg);
        return false;
    }
    
    // Extraer la descripción
    try {
        // Definir los namespaces
        $soap_ns = 'http://schemas.xmlsoap.org/soap/envelope/';
        $softland_ns = 'http://softland.cl/';
        
        // Extraer el nodo soap:Body
        $body = $xml->children($soap_ns)->Body;
        
        // Extraer la respuesta
        $response_node = $body->children($softland_ns)->ObtenerProductosPorCodigoV2Response;
        $result_node = $response_node->ObtenerProductosPorCodigoV2Result;
        
        // Verificar si hay descripción (primero Descipcion, luego desprod como alternativa)
        if (isset($result_node->Descipcion) && !empty($result_node->Descipcion)) {
            return (string) $result_node->Descipcion;
        } else if (isset($result_node->desprod) && !empty($result_node->desprod)) {
            return (string) $result_node->desprod;
        } else {
            log_message("No se encontró descripción para este producto");
            return null; // Si no hay descripción, retornar null
        }
    } catch (Exception $e) {
        log_message("Error al procesar la respuesta (Descripción): " . $e->getMessage());
        return false;
    }
}
?>