<?php
/**
 * Gestión de repuestos
 * CRUD completo para repuestos del sistema
 * Integración mejorada con Softland
 * Versión: 2.0
 */

$page_title = 'Gestión de Repuestos';
require_once 'includes/config.php';
require_once 'includes/functions.php';

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Procesar formulario de creación/edición
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        $codigo = trim($_POST['codigo'] ?? '');
        $nombre = trim($_POST['nombre'] ?? '');
        $descripcion = trim($_POST['descripcion'] ?? '');
        $numero_pieza = trim($_POST['numero_pieza'] ?? '');
        $codigo_pieza = trim($_POST['codigo_pieza'] ?? '');
        $codigo_proveedor = trim($_POST['codigo_proveedor'] ?? '');
        $costo = !empty($_POST['costo']) ? floatval(str_replace(['$', '.', ','], '', $_POST['costo'])) : null;
        $stock = isset($_POST['stock']) ? intval($_POST['stock']) : 0;
        $ubicacion = trim($_POST['ubicacion'] ?? ''); // Campo para código de barras EAN
        
        // Determinar disponibilidad basado en stock
        $disponible = $stock > 0 ? 1 : 0;
        
        // Validación de datos
        $errores = [];
        if (empty($codigo)) $errores[] = "El código Softland del repuesto es obligatorio";
        if (empty($nombre)) $errores[] = "El nombre del repuesto es obligatorio";
        
        // Validación de stock
        if ($stock < 0) $errores[] = "El stock no puede ser negativo";
        
        // Validación de costo
        if (!empty($_POST['costo']) && !is_numeric(str_replace(['$', '.', ','], '', $_POST['costo']))) {
            $errores[] = "El formato del costo no es válido";
        }
        
        // Verificar si el código ya existe
        if ($action === 'new' || ($action === 'edit' && $_POST['codigo_original'] !== $codigo)) {
            $stmt = $conn->prepare("SELECT id FROM repuestos WHERE codigo = :codigo");
            $stmt->bindParam(':codigo', $codigo);
            $stmt->execute();
            if ($stmt->rowCount() > 0) {
                $errores[] = "El código '{$codigo}' ya está en uso por otro repuesto";
            }
        }
        
        if (empty($errores)) {
            try {
                $conn->beginTransaction();
                
                if ($action === 'edit' && $id > 0) {
                    $stmt = $conn->prepare("
                        UPDATE repuestos 
                        SET codigo = :codigo, nombre = :nombre, descripcion = :descripcion, 
                            numero_pieza = :numero_pieza, codigo_pieza = :codigo_pieza, 
                            costo = :costo, disponible = :disponible, stock = :stock, 
                            ubicacion = :ubicacion, codigo_proveedor = :codigo_proveedor
                        WHERE id = :id
                    ");
                    $stmt->bindParam(':id', $id);
                    
                    // Registrar en historial_cambios
                    $stmt_hist = $conn->prepare("
                        INSERT INTO historial_cambios (
                            fecha, usuario_id, tipo_cambio, tabla_afectada, 
                            registro_id, detalle_cambio
                        ) VALUES (
                            NOW(), :usuario_id, 'edicion', 'repuestos', 
                            :registro_id, 'Edición de repuesto'
                        )
                    ");
                    $stmt_hist->bindParam(':usuario_id', $user_id);
                    $stmt_hist->bindParam(':registro_id', $id);
                    $stmt_hist->execute();
                    
                    $mensaje = "Repuesto actualizado correctamente";
                } else {
                    $stmt = $conn->prepare("
                        INSERT INTO repuestos 
                        (codigo, nombre, descripcion, numero_pieza, codigo_pieza, 
                         costo, disponible, stock, fecha_registro, activo, 
                         ubicacion, codigo_proveedor) 
                        VALUES (:codigo, :nombre, :descripcion, :numero_pieza, :codigo_pieza, 
                               :costo, :disponible, :stock, NOW(), 1, 
                               :ubicacion, :codigo_proveedor)
                    ");
                    $mensaje = "Repuesto creado correctamente";
                }
                
                $stmt->bindParam(':codigo', $codigo);
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':descripcion', $descripcion);
                $stmt->bindParam(':numero_pieza', $numero_pieza);
                $stmt->bindParam(':codigo_pieza', $codigo_pieza);
                $stmt->bindParam(':codigo_proveedor', $codigo_proveedor);
                $stmt->bindParam(':costo', $costo);
                $stmt->bindParam(':disponible', $disponible);
                $stmt->bindParam(':stock', $stock);
                $stmt->bindParam(':ubicacion', $ubicacion);
                $stmt->execute();
                
                if ($action === 'new') {
                    $id = $conn->lastInsertId();
                    // Registrar en historial_cambios
                    $stmt_hist = $conn->prepare("
                        INSERT INTO historial_cambios (
                            fecha, usuario_id, tipo_cambio, tabla_afectada, 
                            registro_id, detalle_cambio
                        ) VALUES (
                            NOW(), :usuario_id, 'creacion', 'repuestos', 
                            :registro_id, 'Creación de repuesto'
                        )
                    ");
                    $stmt_hist->bindParam(':usuario_id', $user_id);
                    $stmt_hist->bindParam(':registro_id', $id);
                    $stmt_hist->execute();
                }
                
                // Procesar máquinas asociadas
                // Primero, eliminar todas las asociaciones actuales
                $stmt = $conn->prepare("DELETE FROM maquina_repuesto WHERE repuesto_id = :repuesto_id");
                $stmt->bindParam(':repuesto_id', $id);
                $stmt->execute();
                
                // Luego, crear las nuevas asociaciones
                if (isset($_POST['maquinas']) && is_array($_POST['maquinas']) && !empty($_POST['maquinas'])) {
                    $stmt = $conn->prepare("INSERT INTO maquina_repuesto (maquina_id, repuesto_id) VALUES (:maquina_id, :repuesto_id)");
                    
                    foreach ($_POST['maquinas'] as $maquina_id) {
                        $stmt->bindParam(':maquina_id', $maquina_id);
                        $stmt->bindParam(':repuesto_id', $id);
                        $stmt->execute();
                    }
                }
                
                $conn->commit();
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: repuestos.php');
                exit;
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = "Error al guardar: " . $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Ejecutar actualización de stock desde Softland
    if ($action === 'update_stock') {
        try {
            // Incluir el archivo con la integración de Softland
            require_once 'softland_integration.php';
            
            // Ejecutar la actualización
            $resultado = actualizarStockDesdeSoftland($conn);
            
            if ($resultado['success']) {
                // Registrar en historial_cambios
                $stmt_hist = $conn->prepare("
                    INSERT INTO historial_cambios (
                        fecha, usuario_id, tipo_cambio, tabla_afectada, 
                        registro_id, detalle_cambio
                    ) VALUES (
                        NOW(), :usuario_id, 'actualizacion_stock', 'repuestos', 
                        NULL, 'Actualización de stock desde Softland'
                    )
                ");
                $stmt_hist->bindParam(':usuario_id', $user_id);
                $stmt_hist->execute();
                
                $_SESSION['flash_message'] = "Stock actualizado correctamente. Total actualizado: " . 
                    ($resultado['estadisticas']['actualizados_stock'] + 
                     $resultado['estadisticas']['actualizados_ean'] + 
                     $resultado['estadisticas']['actualizados_ambos']) . 
                    ". Códigos EAN encontrados: " . $resultado['estadisticas']['ean_encontrados'];
                $_SESSION['flash_type'] = 'success';
            } else {
                $_SESSION['flash_message'] = "Error al actualizar stock: " . ($resultado['error'] ?? 'Error desconocido');
                $_SESSION['flash_type'] = 'danger';
            }
            
            header('Location: repuestos.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al ejecutar actualización: " . $e->getMessage();
        }
    }
    
    // Eliminar repuesto
    if ($action === 'delete' && $id > 0) {
        try {
            $conn->beginTransaction();
            
            $stmt = $conn->prepare("SELECT * FROM repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $repuesto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$repuesto) throw new Exception("El repuesto no existe");
            
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM solicitud_repuesto_detalle WHERE repuesto_id = :repuesto_id");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            if ($stmt->fetch(PDO::FETCH_ASSOC)['count'] > 0) {
                throw new Exception("No se puede eliminar el repuesto porque está siendo utilizado en solicitudes");
            }
            
            $stmt = $conn->prepare("DELETE FROM maquina_repuesto WHERE repuesto_id = :repuesto_id");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            
            $stmt = $conn->prepare("DELETE FROM repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Registrar en historial_cambios
            $stmt_hist = $conn->prepare("
                INSERT INTO historial_cambios (
                    fecha, usuario_id, tipo_cambio, tabla_afectada, 
                    registro_id, detalle_cambio
                ) VALUES (
                    NOW(), :usuario_id, 'eliminacion', 'repuestos', 
                    :registro_id, 'Eliminación de repuesto'
                )
            ");
            $stmt_hist->bindParam(':usuario_id', $user_id);
            $stmt_hist->bindParam(':registro_id', $id);
            $stmt_hist->execute();
            
            $conn->commit();
            
            $_SESSION['flash_message'] = "Repuesto eliminado correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: repuestos.php');
            exit;
        } catch (Exception $e) {
            $conn->rollBack();
            $error_message = "Error al eliminar: " . $e->getMessage();
        }
    }
    
    // Activar/Desactivar repuesto
    if ($action === 'toggle' && $id > 0) {
        try {
            $stmt = $conn->prepare("SELECT activo, stock FROM repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $repuesto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$repuesto) throw new Exception("El repuesto no existe");
            
            // Verificar si se está intentando activar un repuesto con stock en 0
            if (!$repuesto['activo'] && $repuesto['stock'] <= 0) {
                throw new Exception("No es posible activar un repuesto con stock en 0");
            }
            
            $nuevo_estado = $repuesto['activo'] ? 0 : 1;
            $stmt = $conn->prepare("UPDATE repuestos SET activo = :activo WHERE id = :id");
            $stmt->bindParam(':activo', $nuevo_estado);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            $accion = $nuevo_estado ? 'activar' : 'desactivar';
            // Registrar en historial_cambios
            $stmt_hist = $conn->prepare("
                INSERT INTO historial_cambios (
                    fecha, usuario_id, tipo_cambio, tabla_afectada, 
                    registro_id, detalle_cambio
                ) VALUES (
                    NOW(), :usuario_id, :tipo_cambio, 'repuestos', 
                    :registro_id, :detalle_cambio
                )
            ");
            $stmt_hist->bindParam(':usuario_id', $user_id);
            $stmt_hist->bindParam(':tipo_cambio', $accion);
            $stmt_hist->bindParam(':registro_id', $id);
            $detalle = ucfirst($accion) . ' repuesto';
            $stmt_hist->bindParam(':detalle_cambio', $detalle);
            $stmt_hist->execute();
            
            $_SESSION['flash_message'] = "Repuesto " . ($nuevo_estado ? "activado" : "desactivado") . " correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: repuestos.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al cambiar estado: " . $e->getMessage();
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

require_once 'includes/header.php';
?>

<style>
    .repuestos-table td, .repuestos-table th { padding: 8px 10px; }
    .repuestos-table tr:hover { background-color: rgba(0,0,0,0.03); }
    .action-icon {
        display: inline-flex; align-items: center; justify-content: center; width: 28px; height: 28px;
        border-radius: 50%; margin-right: 5px; transition: all 0.2s; color: #555 !important; background-color: #f0f0f0;
    }
    .action-icon:hover { background-color: #e0e0e0; transform: scale(1.1); }
    .action-icon.view-icon:hover { color: #1e88e5 !important; }
    .action-icon.pdf-icon:hover { color: #66bb6a !important; }
    .action-icon.edit-icon:hover { color: #ffa726 !important; }
    .action-icon.delete-icon:hover { color: #ef5350 !important; }
    .action-icon i { font-size: 18px; }
    
    .form-control:focus { border-color: #1e88e5; box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25); }
    .form-control:required { border-left: 3px solid #1e88e5; }
    
    /* Estilos para repuestos inactivos */
    .inactive-repuesto { opacity: 0.7; font-style: italic; }
    
    /* Estilos para máquinas asociadas */
    .maquinas-asociadas {
        margin-top: 15px;
        border-left: 3px solid #1e88e5;
        padding-left: 10px;
    }
    .maquina-item {
        display: flex;
        align-items: center;
        margin-bottom: 5px;
        padding: 5px;
        border-bottom: 1px solid #eee;
    }
    .maquina-item:last-child { border-bottom: none; margin-bottom: 0; }
    .maquina-icon { margin-right: 10px; color: #1e88e5; }

    /* Estilos para series de máquinas */
    .series-maquina {
        margin-left: 25px;
        font-size: 0.9em;
        color: #666;
    }
    .serie-item {
        padding: 3px 8px;
        margin: 2px 0;
        background-color: #f9f9f9;
        border-radius: 4px;
        display: inline-block;
    }
    .serie-separator {
        margin: 0 5px;
        color: #ccc;
    }
    
    /* Estilos para códigos de proveedor */
    .codigos-proveedor {
        margin-top: 15px;
        border-left: 3px solid #ffa726;
        padding-left: 10px;
    }
    .codigo-item {
        display: flex;
        align-items: center;
        margin-bottom: 5px;
        padding: 5px;
        border-bottom: 1px solid #eee;
    }
    .codigo-item:last-child { border-bottom: none; margin-bottom: 0; }
    .codigo-icon { margin-right: 10px; color: #ffa726; }

    /* Panel de actualización de stock */
    .stock-update-panel {
        background-color: #f5f5f5;
        border: 1px solid #ddd;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 20px;
    }
    .stock-update-panel h3 {
        margin-top: 0;
        color: #1e88e5;
        font-size: 18px;
    }
    .stock-update-panel p {
        margin-bottom: 15px;
        color: #555;
    }
    .stock-update-btn {
        background-color: #4caf50;
        color: white;
        border: none;
        padding: 10px 15px;
        border-radius: 4px;
        cursor: pointer;
        font-weight: bold;
        display: inline-flex;
        align-items: center;
        transition: background-color 0.3s;
    }
    .stock-update-btn:hover {
        background-color: #388e3c;
    }
    .stock-update-btn i {
        margin-right: 8px;
    }
    .stock-stats {
        display: flex;
        flex-wrap: wrap;
        gap: 15px;
        margin-top: 15px;
    }
    .stat-item {
        background-color: white;
        padding: 10px 15px;
        border-radius: 4px;
        border-left: 3px solid #1e88e5;
        flex: 1;
        min-width: 150px;
    }
    .stat-value {
        font-size: 18px;
        font-weight: bold;
        color: #1e88e5;
    }
    .stat-label {
        font-size: 13px;
        color: #666;
    }
    
    /* Estilos para selección de máquinas */
    .maquinas-selector {
        max-height: 300px;
        overflow-y: auto;
        border: 1px solid #ddd;
        border-radius: 4px;
        padding: 10px;
        margin-top: 10px;
    }
    .maquina-checkbox {
        display: flex;
        align-items: center;
        margin-bottom: 5px;
        padding: 5px;
        border-bottom: 1px solid #f0f0f0;
    }
    .maquina-checkbox:last-child {
        border-bottom: none;
        margin-bottom: 0;
    }
    .maquina-checkbox input {
        margin-right: 10px;
    }
    .maquina-checkbox label {
        margin-bottom: 0;
        cursor: pointer;
        flex: 1;
    }
    
    /* Mejora en etiquetas de campos */
    .field-label {
        font-weight: 500;
        color: #555;
        margin-bottom: 5px;
    }
    .field-label .required {
        color: #e53935;
        margin-left: 3px;
    }
    .field-note {
        font-size: 0.85em;
        color: #757575;
        margin-top: 3px;
    }
    
    /* Estilos para pestaña de búsqueda y filtros */
    .search-tab {
        margin-bottom: 20px;
    }
    .search-form {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 15px;
    }
    .search-field {
        flex: 1;
        min-width: 200px;
    }
    .search-buttons {
        display: flex;
        gap: 10px;
        align-items: flex-end;
    }
    
    /* Badges de estado */
    .badge-estado-3 {
        background-color: #e8f5e9;
        color: #2e7d32;
        padding: 3px 8px;
        border-radius: 3px;
        font-size: 0.85em;
    }
    
    .badge-estado-4 {
        background-color: #ffebee;
        color: #c62828;
        padding: 3px 8px;
        border-radius: 3px;
        font-size: 0.85em;
    }
</style>

<?php if (isset($error_message)): ?>
<div class="alert alert-danger"><?php echo $error_message; ?></div>
<?php endif; ?>

<?php
switch ($action) {
    case 'new':
    case 'edit':
        $repuesto = null;
        $maquinas_asociadas = [];
        
        if ($action === 'edit' && $id > 0) {
            $stmt = $conn->prepare("SELECT * FROM repuestos WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $repuesto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$repuesto) {
                echo '<div class="alert alert-danger">Repuesto no encontrado</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Obtener máquinas asociadas a este repuesto
            $stmt = $conn->prepare("
                SELECT maquina_id FROM maquina_repuesto WHERE repuesto_id = :repuesto_id
            ");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            $maquinas_asociadas = $stmt->fetchAll(PDO::FETCH_COLUMN);
        }
        
        // Obtener todas las máquinas activas para el selector
        $stmt = $conn->prepare("
            SELECT id, codigo, nombre, marca, modelo
            FROM maquinas 
            WHERE activo = 1
            ORDER BY codigo
        ");
        $stmt->execute();
        $maquinas_disponibles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $title = ($action === 'new') ? 'Nuevo Repuesto' : 'Editar Repuesto';
        ?>
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><?php echo $title; ?></h2>
                <a href="repuestos.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="repuestos.php?action=<?php echo $action . ($action === 'edit' ? '&id=' . $id : ''); ?>">
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                        <div class="form-group">
                            <label for="codigo" class="field-label">Código Softland <span class="required">*</span></label>
                            <input type="text" id="codigo" name="codigo" class="form-control" 
                                   value="<?php echo isset($repuesto['codigo']) ? htmlspecialchars($repuesto['codigo']) : ''; ?>" 
                                   placeholder="Código Softland del repuesto" required>
                            <?php if ($action === 'edit'): ?>
                            <input type="hidden" name="codigo_original" value="<?php echo htmlspecialchars($repuesto['codigo']); ?>">
                            <?php endif; ?>
                            <div class="field-note">Código único identificador en el sistema Softland</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="nombre" class="field-label">Nombre <span class="required">*</span></label>
                            <input type="text" id="nombre" name="nombre" class="form-control" 
                                   value="<?php echo isset($repuesto['nombre']) ? htmlspecialchars($repuesto['nombre']) : ''; ?>" 
                                   placeholder="Nombre descriptivo del repuesto" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="codigo_pieza" class="field-label">Código de Pieza</label>
                            <input type="text" id="codigo_pieza" name="codigo_pieza" class="form-control" 
                                   value="<?php echo isset($repuesto['codigo_pieza']) ? htmlspecialchars($repuesto['codigo_pieza']) : ''; ?>" 
                                   placeholder="Código de identificación de la pieza">
                            <div class="field-note">Código específico de la pieza según fabricante</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="numero_pieza" class="field-label">Número de Pieza</label>
                            <input type="text" id="numero_pieza" name="numero_pieza" class="form-control" 
                                   value="<?php echo isset($repuesto['numero_pieza']) ? htmlspecialchars($repuesto['numero_pieza']) : ''; ?>" 
                                   placeholder="Número de serie o referencia de la pieza">
                            <div class="field-note">Número o referencia alternativa de la pieza</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="descripcion" class="field-label">Descripción</label>
                            <textarea id="descripcion" name="descripcion" class="form-control" rows="4" 
                                     placeholder="Descripción detallada del repuesto"><?php echo isset($repuesto['descripcion']) ? htmlspecialchars($repuesto['descripcion']) : ''; ?></textarea>
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                        <div class="form-group">
                            <label for="ubicacion" class="field-label">Código de Barras (EAN)</label>
                            <input type="text" id="ubicacion" name="ubicacion" class="form-control" 
                                   value="<?php echo isset($repuesto['ubicacion']) ? htmlspecialchars($repuesto['ubicacion']) : ''; ?>" 
                                   placeholder="Código de barras EAN del repuesto">
                            <div class="field-note">Este código se actualiza automáticamente desde Softland</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="codigo_proveedor" class="field-label">Código de Proveedor / Ubicación</label>
                            <input type="text" id="codigo_proveedor" name="codigo_proveedor" class="form-control" 
                                   value="<?php echo isset($repuesto['codigo_proveedor']) ? htmlspecialchars($repuesto['codigo_proveedor']) : ''; ?>" 
                                   placeholder="Código usado por el proveedor o ubicación física">
                            <div class="field-note">Código del proveedor o ubicación física en bodega</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="costo" class="field-label">Costo</label>
                            <div class="input-group">
                                <span class="input-group-text">$</span>
                                <input type="text" id="costo" name="costo" class="form-control" 
                                       value="<?php echo isset($repuesto['costo']) && $repuesto['costo'] !== null ? number_format($repuesto['costo'], 0, ',', '.') : ''; ?>" 
                                       placeholder="Costo del repuesto">
                            </div>
                            <div class="field-note">Costo de adquisición del repuesto (sin IVA)</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="stock" class="field-label">Stock Disponible</label>
                            <input type="number" id="stock" name="stock" class="form-control" 
                                   value="<?php echo isset($repuesto['stock']) ? $repuesto['stock'] : '0'; ?>" 
                                   placeholder="Cantidad en stock" min="0">
                            <div class="field-note">Stock en 0 desactiva automáticamente la disponibilidad del repuesto</div>
                        </div>
                        
                        <?php if (isset($repuesto) && $repuesto['stock'] <= 0): ?>
                        <div class="alert alert-warning">
                            <i class="material-icons" style="vertical-align: middle; margin-right: 5px;">warning</i>
                            Este repuesto no está disponible porque su stock es 0 o negativo.
                        </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label class="field-label">Máquinas Compatibles</label>
                            <div class="field-note">Seleccione las máquinas compatibles con este repuesto</div>
                            <div class="maquinas-selector">
                                <?php if (empty($maquinas_disponibles)): ?>
                                <p class="text-muted">No hay máquinas disponibles para asociar.</p>
                                <?php else: ?>
                                    <?php foreach ($maquinas_disponibles as $maquina): ?>
                                    <div class="maquina-checkbox">
                                        <input type="checkbox" 
                                               id="maquina_<?php echo $maquina['id']; ?>" 
                                               name="maquinas[]" 
                                               value="<?php echo $maquina['id']; ?>"
                                               <?php echo in_array($maquina['id'], $maquinas_asociadas) ? 'checked' : ''; ?>>
                                        <label for="maquina_<?php echo $maquina['id']; ?>">
                                            <strong><?php echo htmlspecialchars($maquina['codigo']); ?></strong> - 
                                            <?php echo htmlspecialchars($maquina['nombre']); ?>
                                            <?php if (!empty($maquina['marca']) || !empty($maquina['modelo'])): ?>
                                            <small class="text-muted">
                                                (<?php 
                                                    $info = [];
                                                    if (!empty($maquina['marca'])) $info[] = $maquina['marca'];
                                                    if (!empty($maquina['modelo'])) $info[] = $maquina['modelo'];
                                                    echo htmlspecialchars(implode(' ', $info));
                                                ?>)
                                            </small>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Repuesto
                    </button>
                    <a href="repuestos.php" class="btn btn-danger">
                        <i class="material-icons">cancel</i> Cancelar
                    </a>
                </div>
            </form>
        </div>
        
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const costoInput = document.getElementById('costo');
                if (costoInput) {
                    costoInput.addEventListener('input', function(e) {
                        let value = e.target.value.replace(/[^\d]/g, '');
                        if (value) {
                            value = parseInt(value).toLocaleString('es-CL');
                            e.target.value = value;
                        }
                    });
                }
                
                // Control de stock y disponibilidad automática
                const stockInput = document.getElementById('stock');
                if (stockInput) {
                    stockInput.addEventListener('change', function() {
                        const stockValue = parseInt(this.value) || 0;
                        if (stockValue <= 0) {
                            alert('Advertencia: Con stock 0 o negativo el repuesto no estará disponible para su solicitud.');
                        }
                    });
                }
            });
        </script>
        <?php
        break;
        
    case 'view':
        try {
            $stmt = $conn->prepare("
                SELECT r.* 
                FROM repuestos r
                WHERE r.id = :id
            ");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $repuesto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$repuesto) {
                echo '<div class="alert alert-danger">Repuesto no encontrado</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="repuestos.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Obtener máquinas compatibles
            $stmt = $conn->prepare("
                SELECT m.id, m.codigo, m.nombre, m.marca, m.modelo, m.activo
                FROM maquinas m
                INNER JOIN maquina_repuesto mr ON m.id = mr.maquina_id
                WHERE mr.repuesto_id = :repuesto_id
                ORDER BY m.codigo
            ");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener series de máquinas
            $series_por_maquina = [];
            if (!empty($maquinas)) {
                $ids = array_column($maquinas, 'id');
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                
                $stmt = $conn->prepare("
                    SELECT maquina_id, codigo_serie, estado, proveedor_id
                    FROM series_maquinas
                    WHERE maquina_id IN ($placeholders)
                    ORDER BY maquina_id, codigo_serie
                ");
                
                $stmt->execute($ids);
                $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($series as $serie) {
                    if (!isset($series_por_maquina[$serie['maquina_id']])) {
                        $series_por_maquina[$serie['maquina_id']] = [];
                    }
                    $series_por_maquina[$serie['maquina_id']][] = $serie;
                }
            }
            
            // Obtener códigos de proveedor usados para este repuesto
            $stmt = $conn->prepare("
                SELECT DISTINCT srd.codigo_proveedor, sr.fecha
                FROM solicitud_repuesto_detalle srd
                JOIN solicitud_repuestos sr ON srd.solicitud_id = sr.id
                WHERE srd.repuesto_id = :repuesto_id AND srd.codigo_proveedor IS NOT NULL AND srd.codigo_proveedor != ''
                ORDER BY sr.fecha DESC
            ");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            $codigos_proveedor = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener solicitudes donde se ha usado este repuesto
            $stmt = $conn->prepare("
                SELECT sr.id, sr.fecha, sr.area_origen, sr.motivo_solicitud, srd.cantidad,
                       u.nombre as usuario_nombre, u.apellido as usuario_apellido,
                       m.codigo as maquina_codigo, m.nombre as maquina_nombre,
                       sm.codigo_serie, p.nombre as proveedor_nombre, p.rut as proveedor_rut
                FROM solicitud_repuesto_detalle srd
                JOIN solicitud_repuestos sr ON srd.solicitud_id = sr.id
                JOIN users u ON sr.user_id = u.id
                JOIN maquinas m ON sr.maquina_id = m.id
                LEFT JOIN series_maquinas sm ON sr.serie_id = sm.id
                LEFT JOIN proveedores p ON sm.proveedor_id = p.id
                WHERE srd.repuesto_id = :repuesto_id
                ORDER BY sr.fecha DESC
                LIMIT 10
            ");
            $stmt->bindParam(':repuesto_id', $id);
            $stmt->execute();
            $solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Detalle de Repuesto: <?php echo htmlspecialchars($repuesto['codigo']); ?></h2>
                    <div>
                        <a href="repuestos.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">arrow_back</i> Volver a la Lista
                        </a>
                        
                        <a href="repuestos.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                            <i class="material-icons">edit</i> Editar
                        </a>
                        
                        <?php if ($is_admin): ?>
                        <?php if ($repuesto['activo'] || (!$repuesto['activo'] && $repuesto['stock'] > 0)): ?>
                        <a href="repuestos.php?action=toggle&id=<?php echo $id; ?>" class="btn btn-secondary btn-sm">
                            <i class="material-icons"><?php echo $repuesto['activo'] ? 'toggle_off' : 'toggle_on'; ?></i>
                            <?php echo $repuesto['activo'] ? 'Desactivar' : 'Activar'; ?>
                        </a>
                        <?php endif; ?>
                        
                        <a href="repuestos.php?action=delete&id=<?php echo $id; ?>" class="btn btn-danger btn-sm" 
                           onclick="return confirm('¿Está seguro de eliminar este repuesto?')">
                            <i class="material-icons">delete</i> Eliminar
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card" style="height: 100%;">
                            <div class="panel-header">
                                <h3 class="panel-title">Información General</h3>
                            </div>
                            <table>
                                <tr>
                                    <th style="width: 30%;">Código Softland:</th>
                                    <td><?php echo htmlspecialchars($repuesto['codigo']); ?></td>
                                </tr>
                                <tr>
                                    <th>Nombre:</th>
                                    <td><?php echo htmlspecialchars($repuesto['nombre']); ?></td>
                                </tr>
                                <tr>
                                    <th>Código de Pieza:</th>
                                    <td><?php echo !empty($repuesto['codigo_pieza']) ? htmlspecialchars($repuesto['codigo_pieza']) : 'No especificado'; ?></td>
                                </tr>
                                <tr>
                                    <th>Número de Pieza:</th>
                                    <td><?php echo !empty($repuesto['numero_pieza']) ? htmlspecialchars($repuesto['numero_pieza']) : 'No especificado'; ?></td>
                                </tr>
                                <tr>
                                    <th>Código de Barras (EAN):</th>
                                    <td><?php echo !empty($repuesto['ubicacion']) ? htmlspecialchars($repuesto['ubicacion']) : 'No especificado'; ?></td>
                                </tr>
                                <tr>
                                    <th>Código Proveedor/Ubicación:</th>
                                    <td><?php echo !empty($repuesto['codigo_proveedor']) ? htmlspecialchars($repuesto['codigo_proveedor']) : 'No especificado'; ?></td>
                                </tr>
                                <tr>
                                    <th>Estado:</th>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $repuesto['activo'] ? '3' : '4'; ?>">
                                            <?php echo $repuesto['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Disponibilidad:</th>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $repuesto['disponible'] ? '3' : '4'; ?>">
                                            <?php echo $repuesto['disponible'] ? 'Disponible' : 'No disponible'; ?>
                                        </span>
                                        <?php if ($repuesto['stock'] <= 0): ?>
                                        <small class="text-danger">(Stock insuficiente)</small>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php if ($is_admin): ?>
                                <tr>
                                    <th>Costo:</th>
                                    <td><?php echo isset($repuesto['costo']) && $repuesto['costo'] !== null ? formatoMoneda($repuesto['costo']) : 'No definido'; ?></td>
                                </tr>
                                <?php endif; ?>
                                <tr>
                                    <th>Stock:</th>
                                    <td><?php echo isset($repuesto['stock']) ? $repuesto['stock'] : '0'; ?> unidades</td>
                                </tr>
                                <tr>
                                    <th>Ubicación:</th>
                                    <td><?php echo !empty($repuesto['ubicacion']) ? htmlspecialchars($repuesto['ubicacion']) : 'No especificada'; ?></td>
                                </tr>
                                <tr>
                                    <th>Fecha de Registro:</th>
                                    <td><?php echo date('d/m/Y H:i', strtotime($repuesto['fecha_registro'])); ?></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 300px;">
                        <div class="card" style="height: 100%;">
                            <div class="panel-header">
                                <h3 class="panel-title">Descripción</h3>
                            </div>
                            <div style="padding: 10px;">
                                <?php if (!empty($repuesto['descripcion'])): ?>
                                <p><?php echo nl2br(htmlspecialchars($repuesto['descripcion'])); ?></p>
                                <?php else: ?>
                                <p class="text-muted">No hay descripción disponible</p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if (!empty($codigos_proveedor)): ?>
                        <div class="card" style="margin-top: 20px;">
                            <div class="panel-header">
                                <h3 class="panel-title">Códigos de Proveedor Históricos</h3>
                            </div>
                            <div style="padding: 10px;">
                                <div class="codigos-proveedor">
                                    <?php foreach ($codigos_proveedor as $codigo): ?>
                                    <div class="codigo-item">
                                        <i class="material-icons codigo-icon">label</i>
                                        <div>
                                            <strong><?php echo htmlspecialchars($codigo['codigo_proveedor']); ?></strong>
                                            <?php if ($codigo['fecha']): ?>
                                            <small class="text-muted">(Usado en solicitud del <?php echo formatoFecha($codigo['fecha']); ?>)</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <?php if (!empty($maquinas)): ?>
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Máquinas Compatibles</h3>
                        <a href="maquinas.php" class="btn btn-primary btn-sm">
                            <i class="material-icons">precision_manufacturing</i> Gestionar Máquinas
                        </a>
                    </div>
                    <div class="table-responsive">
                        <table class="repuestos-table">
                            <thead>
                                <tr>
                                    <th>Código</th>
                                    <th>Nombre</th>
                                    <th>Marca</th>
                                    <th>Modelo</th>
                                    <th>Series</th>
                                    <th>Estado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($maquinas as $maquina): ?>
                                <tr class="<?php echo $maquina['activo'] ? '' : 'inactive-repuesto'; ?>">
                                    <td><?php echo htmlspecialchars($maquina['codigo']); ?></td>
                                    <td><?php echo htmlspecialchars($maquina['nombre']); ?></td>
                                    <td><?php echo !empty($maquina['marca']) ? htmlspecialchars($maquina['marca']) : 'N/A'; ?></td>
                                    <td><?php echo !empty($maquina['modelo']) ? htmlspecialchars($maquina['modelo']) : 'N/A'; ?></td>
                                    <td>
                                        <?php if (isset($series_por_maquina[$maquina['id']])): ?>
                                            <?php
                                            $series = $series_por_maquina[$maquina['id']];
                                            $totalSeries = count($series);
                                            $seriesMostradas = min(3, $totalSeries);
                                            
                                            for ($i = 0; $i < $seriesMostradas; $i++) {
                                                echo '<span class="serie-item">' . htmlspecialchars($series[$i]['codigo_serie']) . '</span>';
                                                if ($i < $seriesMostradas - 1) {
                                                    echo '<span class="serie-separator">|</span>';
                                                }
                                            }
                                            
                                            if ($totalSeries > $seriesMostradas) {
                                                echo '<span class="text-muted"> (+' . ($totalSeries - $seriesMostradas) . ' más)</span>';
                                            }
                                            ?>
                                        <?php else: ?>
                                            <span class="text-muted">Sin series</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="status-badge badge-estado-<?php echo $maquina['activo'] ? '3' : '4'; ?>">
                                            <?php echo $maquina['activo'] ? 'Activo' : 'Inactivo'; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <a href="maquinas.php?action=view&id=<?php echo $maquina['id']; ?>" 
                                           class="action-icon view-icon" title="Ver Máquina">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                        <?php if ($is_admin): ?>
                                        <a href="maquinas.php?action=edit&id=<?php echo $maquina['id']; ?>" 
                                           class="action-icon edit-icon" title="Editar Máquina">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($solicitudes)): ?>
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Últimas Solicitudes</h3>
                        <a href="solicitud_repuestos.php?repuesto_id=<?php echo $id; ?>" class="btn btn-primary btn-sm">
                            <i class="material-icons">list</i> Ver Todas
                        </a>
                    </div>
                    <div class="table-responsive">
                        <table class="repuestos-table">
                            <thead>
                                <tr>
                                    <th>Fecha</th>
                                    <th>Solicitante</th>
                                    <th>Máquina</th>
                                    <th>Serie</th>
                                    <th>Proveedor</th>
                                    <th>Cantidad</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($solicitudes as $solicitud): ?>
                                <tr>
                                    <td><?php echo date('d/m/Y', strtotime($solicitud['fecha'])); ?></td>
                                    <td><?php echo htmlspecialchars($solicitud['usuario_nombre'] . ' ' . $solicitud['usuario_apellido']); ?></td>
                                    <td><?php echo htmlspecialchars($solicitud['maquina_codigo'] . ' - ' . $solicitud['maquina_nombre']); ?></td>
                                    <td><?php echo !empty($solicitud['codigo_serie']) ? htmlspecialchars($solicitud['codigo_serie']) : 'N/A'; ?></td>
                                    <td><?php echo !empty($solicitud['proveedor_nombre']) ? htmlspecialchars($solicitud['proveedor_nombre'] . ' (' . $solicitud['proveedor_rut'] . ')') : 'N/A'; ?></td>
                                    <td><?php echo $solicitud['cantidad']; ?></td>
                                    <td>
                                        <a href="solicitud_repuestos.php?action=view&id=<?php echo $solicitud['id']; ?>" 
                                           class="action-icon view-icon" title="Ver Solicitud">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;">
                      <a href="repuestos.php" class="btn btn-primary">
                          <i class="material-icons">arrow_back</i> Volver a la Lista
                      </a>
                  </div>';
        }
        break;
        
    default:
        // Parámetros de búsqueda y filtrado
        $search = isset($_GET['search']) ? trim($_GET['search']) : '';
        $stock_filter = isset($_GET['stock']) ? $_GET['stock'] : '';
        
        // Construir la consulta base
        $sql = "SELECT r.* FROM repuestos r WHERE 1=1";
        $params = [];
        
        // Aplicar filtros
        if (!empty($search)) {
            $sql .= " AND (r.codigo LIKE :search OR r.nombre LIKE :search OR r.codigo_pieza LIKE :search)";
            $params[':search'] = "%$search%";
        }
        
        if ($stock_filter === 'disponible') {
            $sql .= " AND r.stock > 0";
        } elseif ($stock_filter === 'agotado') {
            $sql .= " AND r.stock <= 0";
        }
        
        $sql .= " ORDER BY r.codigo";
        
        $stmt = $conn->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        $repuestos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        ?>
        
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><i class="material-icons">inventory_2</i> Gestión de Repuestos</h2>
                <div class="panel-actions">
                    <a href="repuestos.php?action=new" class="btn btn-primary">
                        <i class="material-icons">add</i> Nuevo Repuesto
                    </a>
                    <?php if ($is_admin): ?>
                    <button type="button" class="btn btn-success" data-toggle="modal" data-target="#updateStockModal">
                        <i class="material-icons">sync</i> Actualizar Stock
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Filtros y búsqueda -->
            <div class="search-tab">
                <form method="GET" action="repuestos.php" class="search-form">
                    <div class="search-field">
                        <input type="text" name="search" class="form-control" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Buscar por código, nombre o código de pieza">
                    </div>
                    
                    <div class="search-field">
                        <select name="stock" class="form-control">
                            <option value="">Todos los estados</option>
                            <option value="disponible" <?php echo $stock_filter === 'disponible' ? 'selected' : ''; ?>>Con stock</option>
                            <option value="agotado" <?php echo $stock_filter === 'agotado' ? 'selected' : ''; ?>>Sin stock</option>
                        </select>
                    </div>
                    
                    <div class="search-buttons">
                        <button type="submit" class="btn btn-primary">
                            <i class="material-icons">search</i> Buscar
                        </button>
                        <a href="repuestos.php" class="btn btn-secondary">
                            <i class="material-icons">clear</i> Limpiar
                        </a>
                    </div>
                </form>
            </div>
            
            <!-- Modal para Actualización de Stock -->
            <div class="modal fade" id="updateStockModal" tabindex="-1" role="dialog" aria-labelledby="updateStockModalLabel">
                <div class="modal-dialog modal-lg" role="document">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="updateStockModalLabel">
                                <i class="material-icons">sync</i> Actualización de Stock desde Softland
                            </h5>
                            <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">&times;</span>
                            </button>
                        </div>
                        <div class="modal-body">
                            <div class="alert alert-info">
                                <i class="material-icons">info</i>
                                Esta acción actualizará el stock y la ubicación de todos los repuestos activos desde Softland.
                            </div>
                            
                            <div class="card mb-3">
                                <div class="card-body">
                                    <h6 class="card-title">
                                        <i class="material-icons">update</i> 
                                        Última actualización
                                    </h6>
                                    <p class="text-muted">
                                        <?php
                                        // Obtener la última actualización desde la tabla de historial_cambios
                                        $stmt = $conn->prepare("
                                            SELECT MAX(fecha_hora) as ultima_actualizacion 
                                            FROM historial_cambios 
                                            WHERE tipo = 'actualizacion_stock'
                                        ");
                                        $stmt->execute();
                                        $ultima = $stmt->fetch(PDO::FETCH_ASSOC);
                                        echo $ultima['ultima_actualizacion'] ? date('d/m/Y H:i:s', strtotime($ultima['ultima_actualizacion'])) : 'Sin registros previos';
                                        ?>
                                    </p>
                                </div>
                            </div>
                            
                            <div class="alert alert-warning">
                                <i class="material-icons">warning</i>
                                <strong>Importante:</strong>
                                <ul class="mb-0">
                                    <li>Este proceso puede tomar varios minutos.</li>
                                    <li>Se actualizará el stock y la ubicación de todos los repuestos activos.</li>
                                    <li>Los repuestos que queden sin stock serán marcados como no disponibles.</li>
                                </ul>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-dismiss="modal">
                                <i class="material-icons">close</i> Cancelar
                            </button>
                            <a href="repuestos.php?action=update_stock" class="btn btn-success">
                                <i class="material-icons">sync</i> Iniciar Actualización
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tabla de Repuestos -->
            <div class="table-responsive">
                <table class="table table-striped repuestos-table">
                    <thead>
                        <tr>
                            <th>Código</th>
                            <th>Nombre</th>
                            <th>Stock</th>
                            <th>Ubicación</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($repuestos as $repuesto): 
                        $row_class = !$repuesto['activo'] ? 'inactive-repuesto' : '';
                        $estado_class = $repuesto['stock'] > 0 ? 'text-success' : 'text-danger';
                        $estado_text = $repuesto['stock'] > 0 ? 'Disponible' : 'Sin Stock';
                    ?>
                        <tr class="<?php echo $row_class; ?>">
                            <td><?php echo htmlspecialchars($repuesto['codigo']); ?></td>
                            <td><?php echo htmlspecialchars($repuesto['nombre']); ?></td>
                            <td><?php echo $repuesto['stock']; ?></td>
                            <td><?php echo !empty($repuesto['ubicacion']) ? htmlspecialchars($repuesto['ubicacion']) : '-'; ?></td>
                            <td class="<?php echo $estado_class; ?>"><?php echo $estado_text; ?></td>
                            <td>
                                <div class="btn-group">
                                    <a href="repuestos.php?action=view&id=<?php echo $repuesto['id']; ?>" 
                                       class="action-icon view-icon" title="Ver detalles">
                                        <i class="material-icons">visibility</i>
                                    </a>
                                    <?php if ($is_admin): ?>
                                    <a href="repuestos.php?action=edit&id=<?php echo $repuesto['id']; ?>" 
                                       class="action-icon edit-icon" title="Editar">
                                        <i class="material-icons">edit</i>
                                    </a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
        break;
}
?>

<?php require_once 'includes/footer.php'; ?>