<?php
/**
 * Módulo mejorado de reportes y estadísticas
 * Generación de informes para rendiciones, órdenes de compra, repuestos y más
 * Versión optimizada para mostrar correctamente las relaciones de la base de datos
 */

// Definir el título de la página
$page_title = 'Reportes y Estadísticas';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Verificar permisos
if (!tienePermiso('ver_reportes') && !$is_admin) {
    $_SESSION['flash_message'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['flash_type'] = 'danger';
    header('Location: dashboard.php');
    exit;
}

// Definir mes y año actual como período predeterminado
$mes_actual = date('m');
$ano_actual = date('Y');

// Obtener parámetros de filtrado
$tipo_reporte = isset($_GET['tipo_reporte']) ? $_GET['tipo_reporte'] : 'general';
$periodo = isset($_GET['periodo']) ? $_GET['periodo'] : 'mes';
$mes = isset($_GET['mes']) ? $_GET['mes'] : $mes_actual;
$ano = isset($_GET['ano']) ? $_GET['ano'] : $ano_actual;
$usuario_seleccionado = isset($_GET['usuario']) ? intval($_GET['usuario']) : 0;
$categoria_seleccionada = isset($_GET['categoria']) ? intval($_GET['categoria']) : 0;
$estado_seleccionado = isset($_GET['estado']) ? intval($_GET['estado']) : 0;
$maquina_seleccionada = isset($_GET['maquina']) ? intval($_GET['maquina']) : 0;
$serie_seleccionada = isset($_GET['serie']) ? intval($_GET['serie']) : 0; // Nuevo filtro para series
$proveedor_seleccionado = isset($_GET['proveedor']) ? intval($_GET['proveedor']) : 0;
$fecha_desde = isset($_GET['fecha_desde']) ? $_GET['fecha_desde'] : '';
$fecha_hasta = isset($_GET['fecha_hasta']) ? $_GET['fecha_hasta'] : '';
$formato_exportacion = isset($_GET['exportar']) ? $_GET['exportar'] : '';
$estado_maquina = isset($_GET['estado_maquina']) ? $_GET['estado_maquina'] : ''; // Nuevo filtro para estado de máquinas

// Si no hay fechas específicas, definir según el período seleccionado
if (empty($fecha_desde) || empty($fecha_hasta)) {
    if ($periodo === 'mes') {
        $fecha_desde = date('Y-m-01', strtotime($ano . '-' . $mes . '-01'));
        $fecha_hasta = date('Y-m-t', strtotime($ano . '-' . $mes . '-01'));
    } elseif ($periodo === 'trimestre') {
        $trimestre = ceil($mes / 3);
        $inicio_trimestre = (($trimestre - 1) * 3) + 1;
        $fecha_desde = date('Y-m-01', strtotime($ano . '-' . $inicio_trimestre . '-01'));
        $fecha_hasta = date('Y-m-t', strtotime($ano . '-' . ($inicio_trimestre + 2) . '-01'));
    } elseif ($periodo === 'ano') {
        $fecha_desde = $ano . '-01-01';
        $fecha_hasta = $ano . '-12-31';
    } elseif ($periodo === 'personalizado') {
        // Si es personalizado pero no hay fechas, usar mes actual
        $fecha_desde = date('Y-m-01');
        $fecha_hasta = date('Y-m-t');
    }
}

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Cargar datos para los filtros
    
    // Usuarios
    if ($is_admin) {
        $stmt = $conn->prepare("
            SELECT id, nombre, apellido 
            FROM users 
            WHERE activo = 1 
            ORDER BY nombre, apellido
        ");
        $stmt->execute();
        $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Categorías de gastos (solo para rendiciones)
    $stmt = $conn->prepare("
        SELECT id, nombre 
        FROM categorias_gastos 
        WHERE activo = 1 
        ORDER BY nombre
    ");
    $stmt->execute();
    $categorias = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estados
    $stmt = $conn->prepare("SELECT id, nombre FROM estados ORDER BY id");
    $stmt->execute();
    $estados = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estados de máquina
    $stmt = $conn->prepare("SELECT id, nombre FROM estados_maquina WHERE activo = 1 ORDER BY nombre");
    $stmt->execute();
    $estados_maquina = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Máquinas (para reportes de repuestos)
    $stmt = $conn->prepare("
        SELECT id, codigo, nombre 
        FROM maquinas 
        WHERE activo = 1 
        ORDER BY nombre
    ");
    $stmt->execute();
    $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Series de máquinas (nuevo filtro)
    $stmt = $conn->prepare("
        SELECT sm.id, sm.codigo_serie, m.nombre as maquina_nombre, sm.estado
        FROM series_maquinas sm
        JOIN maquinas m ON sm.maquina_id = m.id
        WHERE sm.activo = 1
        ORDER BY sm.codigo_serie
    ");
    $stmt->execute();
    $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Proveedores (para reportes de proveedores)
    $stmt = $conn->prepare("
        SELECT id, nombre, rut 
        FROM proveedores 
        WHERE activo = 1 
        ORDER BY nombre
    ");
    $stmt->execute();
    $proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Consultas según el tipo de reporte seleccionado
    
    // Variables para almacenar resultados
    $resultados = [];
    $datos_grafico = [
        'labels' => [],
        'data' => [],
        'colors' => []
    ];
    $titulo_reporte = 'Reporte General';
    $subtitulo_reporte = '';
    
    // Función para generar colores aleatorios para gráficos
    function generarColores($cantidad) {
        $colores = [];
        $colores_predefinidos = [
            'rgba(54, 162, 235, 0.7)',
            'rgba(255, 99, 132, 0.7)',
            'rgba(255, 206, 86, 0.7)',
            'rgba(75, 192, 192, 0.7)',
            'rgba(153, 102, 255, 0.7)',
            'rgba(255, 159, 64, 0.7)',
            'rgba(199, 199, 199, 0.7)',
            'rgba(83, 144, 217, 0.7)',
            'rgba(10, 169, 133, 0.7)',
            'rgba(165, 89, 144, 0.7)'
        ];
        
        // Si hay suficientes colores predefinidos, usarlos
        if ($cantidad <= count($colores_predefinidos)) {
            return array_slice($colores_predefinidos, 0, $cantidad);
        }
        
        // Si no, generar colores aleatorios adicionales
        $colores = array_slice($colores_predefinidos, 0);
        for ($i = count($colores_predefinidos); $i < $cantidad; $i++) {
            $r = mt_rand(50, 200);
            $g = mt_rand(50, 200);
            $b = mt_rand(50, 200);
            $colores[] = "rgba($r, $g, $b, 0.7)";
        }
        
        return $colores;
    }
    
    // 1. Reporte General
    if ($tipo_reporte === 'general') {
        $titulo_reporte = 'Reporte General';
        $subtitulo_reporte = 'Resumen de rendiciones, órdenes de compra y solicitudes de repuestos';
        
        // Filtro de fecha para ambas consultas
        $condicion_fecha = "fecha BETWEEN :fecha_desde AND :fecha_hasta";
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Filtro de usuario si no es admin o si se seleccionó un usuario específico
        $condicion_usuario = "";
        if (!$is_admin) {
            $condicion_usuario = "AND user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $condicion_usuario = "AND user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        // Filtro de estado
        $condicion_estado = "";
        if ($estado_seleccionado > 0) {
            $condicion_estado = "AND estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        // Consultar rendiciones
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as total_documentos,
                SUM(monto_total) as monto_total,
                MIN(fecha) as fecha_minima,
                MAX(fecha) as fecha_maxima
            FROM rendiciones
            WHERE $condicion_fecha $condicion_usuario $condicion_estado
        ");
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $resumen_rendiciones = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Consultar órdenes de compra
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as total_documentos,
                SUM(monto_total) as monto_total,
                MIN(fecha) as fecha_minima,
                MAX(fecha) as fecha_maxima
            FROM ordenes_compra
            WHERE $condicion_fecha $condicion_usuario $condicion_estado
        ");
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $resumen_ordenes = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes de repuestos
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as total_documentos,
                MIN(fecha) as fecha_minima,
                MAX(fecha) as fecha_maxima
            FROM solicitud_repuestos
            WHERE $condicion_fecha $condicion_usuario $condicion_estado
        ");
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $resumen_solicitudes = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Consultar devoluciones
        $stmt = $conn->prepare("
            SELECT 
                COUNT(*) as total_documentos,
                SUM(monto_total) as monto_total,
                MIN(fecha) as fecha_minima,
                MAX(fecha) as fecha_maxima
            FROM devoluciones
            WHERE $condicion_fecha $condicion_usuario $condicion_estado
        ");
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $resumen_devoluciones = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Consultar distribución por estado
        $stmt = $conn->prepare("
            SELECT 
                e.id,
                e.nombre,
                e.color,
                COUNT(r.id) as total
            FROM estados e
            LEFT JOIN rendiciones r ON e.id = r.estado_id 
                AND r.fecha BETWEEN :fecha_desde AND :fecha_hasta $condicion_usuario
            GROUP BY e.id
            ORDER BY e.id
        ");
        
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if ((!$is_admin && !empty($condicion_usuario)) || ($usuario_seleccionado > 0 && !empty($condicion_usuario))) {
            $stmt->bindValue(':user_id', $params[':user_id']);
        }
        
        $stmt->execute();
        $estados_rendiciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $stmt = $conn->prepare("
            SELECT 
                e.id,
                e.nombre,
                e.color,
                COUNT(oc.id) as total
            FROM estados e
            LEFT JOIN ordenes_compra oc ON e.id = oc.estado_id 
                AND oc.fecha BETWEEN :fecha_desde AND :fecha_hasta $condicion_usuario
            GROUP BY e.id
            ORDER BY e.id
        ");
        
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if ((!$is_admin && !empty($condicion_usuario)) || ($usuario_seleccionado > 0 && !empty($condicion_usuario))) {
            $stmt->bindValue(':user_id', $params[':user_id']);
        }
        
        $stmt->execute();
        $estados_ordenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar distribución por estado de solicitudes de repuestos
        $stmt = $conn->prepare("
            SELECT 
                e.id,
                e.nombre,
                e.color,
                COUNT(sr.id) as total
            FROM estados e
            LEFT JOIN solicitud_repuestos sr ON e.id = sr.estado_id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta $condicion_usuario
            GROUP BY e.id
            ORDER BY e.id
        ");
        
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if ((!$is_admin && !empty($condicion_usuario)) || ($usuario_seleccionado > 0 && !empty($condicion_usuario))) {
            $stmt->bindValue(':user_id', $params[':user_id']);
        }
        
        $stmt->execute();
        $estados_solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar distribución por estado de devoluciones
        $stmt = $conn->prepare("
            SELECT 
                e.id,
                e.nombre,
                e.color,
                COUNT(d.id) as total
            FROM estados e
            LEFT JOIN devoluciones d ON e.id = d.estado_id 
                AND d.fecha BETWEEN :fecha_desde AND :fecha_hasta $condicion_usuario
            GROUP BY e.id
            ORDER BY e.id
        ");
        
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if ((!$is_admin && !empty($condicion_usuario)) || ($usuario_seleccionado > 0 && !empty($condicion_usuario))) {
            $stmt->bindValue(':user_id', $params[':user_id']);
        }
        
        $stmt->execute();
        $estados_devoluciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para gráfico de estados
        $labels_estados = [];
        $data_rendiciones = [];
        $data_ordenes = [];
        $data_solicitudes = [];
        
        foreach ($estados_rendiciones as $estado) {
            $labels_estados[] = $estado['nombre'];
            $data_rendiciones[] = $estado['total'];
        }
        
        foreach ($estados_ordenes as $estado) {
            $data_ordenes[] = $estado['total'];
        }
        
        foreach ($estados_solicitudes as $estado) {
            $data_solicitudes[] = $estado['total'];
        }
        
        // Almacenar resultados
        $resultados = [
            'rendiciones' => $resumen_rendiciones,
            'ordenes' => $resumen_ordenes,
            'solicitudes' => $resumen_solicitudes,
            'devoluciones' => $resumen_devoluciones,
            'estados_rendiciones' => $estados_rendiciones,
            'estados_ordenes' => $estados_ordenes,
            'estados_solicitudes' => $estados_solicitudes,
            'estados_devoluciones' => $estados_devoluciones
        ];
        
        // Datos para el gráfico
        $datos_grafico = [
            'labels' => $labels_estados,
            'datasets' => [
                [
                    'label' => 'Rendiciones',
                    'data' => $data_rendiciones,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)'
                ],
                [
                    'label' => 'Órdenes de Compra',
                    'data' => $data_ordenes,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)'
                ],
                [
                    'label' => 'Solicitudes de Repuestos',
                    'data' => $data_solicitudes,
                    'backgroundColor' => 'rgba(75, 192, 192, 0.7)'
                ]
            ]
        ];
    }
    
    // 2. Reporte de Rendiciones por Categoría
    elseif ($tipo_reporte === 'rendiciones_categoria') {
        $titulo_reporte = 'Reporte de Rendiciones por Categoría';
        $subtitulo_reporte = 'Distribución de gastos por categoría';
        
        // Construir consulta
        $sql = "
            SELECT 
                c.id,
                c.nombre,
                COUNT(rc.id) as total_documentos,
                SUM(rc.monto) as monto_total,
                AVG(rc.monto) as promedio_gasto
            FROM categorias_gastos c
            LEFT JOIN rendicion_categoria rc ON c.id = rc.categoria_id
            LEFT JOIN rendiciones r ON rc.rendicion_id = r.id
            WHERE r.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND r.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND r.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND r.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($categoria_seleccionada > 0) {
            $sql .= " AND c.id = :categoria_id";
            $params[':categoria_id'] = $categoria_seleccionada;
        }
        
        $sql .= " GROUP BY c.id ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $categorias_rendiciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Obtener el top 5 de gastos más altos por categoría
        $sql = "
            SELECT 
                c.nombre as categoria,
                r.descripcion,
                rc.monto,
                r.fecha,
                u.nombre, u.apellido
            FROM rendicion_categoria rc
            JOIN categorias_gastos c ON rc.categoria_id = c.id
            JOIN rendiciones r ON rc.rendicion_id = r.id
            JOIN users u ON r.user_id = u.id
            WHERE r.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND r.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND r.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND r.estado_id = :estado_id";
        }
        
        if ($categoria_seleccionada > 0) {
            $sql .= " AND c.id = :categoria_id";
        }
        
        $sql .= " ORDER BY rc.monto DESC LIMIT 10";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $top_gastos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico
        $labels = [];
        $data = [];
        $colors = generarColores(count($categorias_rendiciones));
        
        foreach ($categorias_rendiciones as $index => $categoria) {
            $labels[] = $categoria['nombre'];
            $data[] = $categoria['monto_total'] ?: 0;
        }
        
        $resultados = [
            'categorias' => $categorias_rendiciones,
            'top_gastos' => $top_gastos
        ];
        
        $datos_grafico = [
            'labels' => $labels,
            'datasets' => [
                [
                    'data' => $data,
                    'backgroundColor' => $colors
                ]
            ]
        ];
    }
    
    // NUEVO: 9. Reporte de Series de Máquinas
    elseif ($tipo_reporte === 'series_maquinas') {
        $titulo_reporte = 'Reporte de Series de Máquinas';
        $subtitulo_reporte = 'Análisis detallado de series por estado, cliente y proveedor';
        
        // Consultar series con sus estados y relaciones
        $sql = "
            SELECT 
                sm.id,
                sm.codigo_serie,
                sm.estado,
                m.id as maquina_id,
                m.codigo as maquina_codigo,
                m.nombre as maquina_nombre,
                prov.id as proveedor_id,
                prov.nombre as proveedor_nombre,
                prov.rut as proveedor_rut,
                cli.id as cliente_id,
                cli.nombre as cliente_nombre,
                sm.fecha_inicio as fecha_inicio_serie,
                sm.fecha_inicio_estado as fecha_cambio_estado,
                sm.historial_estados
            FROM series_maquinas sm
            LEFT JOIN maquinas m ON sm.maquina_id = m.id
            LEFT JOIN proveedores prov ON sm.proveedor_id = prov.id
            LEFT JOIN proveedores cli ON sm.cliente_id = cli.id
            WHERE 1=1
        ";
        
        $params = [];
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
            $params[':maquina_id'] = $maquina_seleccionada;
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sm.id = :serie_id";
            $params[':serie_id'] = $serie_seleccionada;
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND (sm.proveedor_id = :proveedor_id OR sm.cliente_id = :cliente_id)";
            $params[':proveedor_id'] = $proveedor_seleccionado;
            $params[':cliente_id'] = $proveedor_seleccionado;
        }
        
        if (!empty($estado_maquina)) {
            $sql .= " AND sm.estado = :estado_maquina";
            $params[':estado_maquina'] = $estado_maquina;
        }
        
        $sql .= " ORDER BY m.nombre, sm.codigo_serie";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $series_maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar distribución por estado
        $sql = "
            SELECT 
                sm.estado,
                COUNT(sm.id) as total_series,
                COUNT(DISTINCT m.id) as total_maquinas
            FROM series_maquinas sm
            JOIN maquinas m ON sm.maquina_id = m.id
            WHERE 1=1
        ";
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sm.id = :serie_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND (sm.proveedor_id = :proveedor_id OR sm.cliente_id = :cliente_id)";
        }
        
        $sql .= " GROUP BY sm.estado ORDER BY total_series DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $estados_series = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes de repuestos por serie
        $sql = "
            SELECT 
                sm.codigo_serie,
                m.nombre as maquina_nombre,
                COUNT(sr.id) as total_solicitudes,
                SUM(srd.cantidad) as total_repuestos
            FROM series_maquinas sm
            JOIN maquinas m ON sm.maquina_id = m.id
            LEFT JOIN solicitud_repuestos sr ON sm.id = sr.serie_id
            LEFT JOIN solicitud_repuesto_detalle srd ON sr.id = srd.solicitud_id
            WHERE 1=1
        ";
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sm.id = :serie_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND (sm.proveedor_id = :proveedor_id OR sm.cliente_id = :cliente_id)";
        }
        
        $sql .= " GROUP BY sm.id ORDER BY total_solicitudes DESC LIMIT 10";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $solicitudes_por_serie = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para gráficos
        $labels_estados = [];
        $data_estados = [];
        $colors_estados = generarColores(count($estados_series));
        
        foreach ($estados_series as $index => $estado) {
            $labels_estados[] = $estado['estado'];
            $data_estados[] = $estado['total_series'];
        }
        
        $labels_series = [];
        $data_series = [];
        $colors_series = generarColores(count($solicitudes_por_serie));
        
        foreach ($solicitudes_por_serie as $index => $serie) {
            $labels_series[] = $serie['codigo_serie'] . ' - ' . $serie['maquina_nombre'];
            $data_series[] = $serie['total_solicitudes'];
        }
        
        $resultados = [
            'series_maquinas' => $series_maquinas,
            'estados_series' => $estados_series,
            'solicitudes_por_serie' => $solicitudes_por_serie
        ];
        
        $datos_grafico = [
            'labels' => $labels_estados,
            'datasets' => [
                [
                    'data' => $data_estados,
                    'backgroundColor' => $colors_estados
                ]
            ]
        ];
        
        $datos_grafico_series = [
            'labels' => $labels_series,
            'datasets' => [
                [
                    'label' => 'Solicitudes de Repuestos',
                    'data' => $data_series,
                    'backgroundColor' => $colors_series
                ]
            ]
        ];
    }
    
    // 6. MEJORA: Reporte de Mantenimiento de Máquinas y Repuestos
    elseif ($tipo_reporte === 'mantenimiento_repuestos') {
        $titulo_reporte = 'Mantenimiento y Repuestos';
        $subtitulo_reporte = 'Análisis de solicitudes de repuestos por máquina y serie';
        
        // Construir consulta mejorada para incluir series
        $sql = "
            SELECT 
                m.id as maquina_id,
                m.codigo,
                m.nombre as maquina_nombre,
                COUNT(sr.id) as total_solicitudes,
                COUNT(DISTINCT sr.user_id) as total_usuarios,
                COUNT(DISTINCT sr.serie_id) as total_series,
                SUM(CASE WHEN sr.serie_id IS NOT NULL THEN 1 ELSE 0 END) as solicitudes_con_serie
            FROM maquinas m
            LEFT JOIN solicitud_repuestos sr ON m.id = sr.maquina_id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE m.activo = 1
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND sr.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND sr.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
            $params[':maquina_id'] = $maquina_seleccionada;
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sr.serie_id = :serie_id";
            $params[':serie_id'] = $serie_seleccionada;
        }
        
        $sql .= " GROUP BY m.id ORDER BY total_solicitudes DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $maquinas_solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // NUEVA CONSULTA: Solicitudes por serie de máquina
        $sql = "
            SELECT 
                sm.id as serie_id,
                sm.codigo_serie,
                m.nombre as maquina_nombre,
                sm.estado,
                p.nombre as proveedor_nombre,
                COUNT(sr.id) as total_solicitudes,
                COUNT(DISTINCT srd.repuesto_id) as total_repuestos_distintos,
                SUM(srd.cantidad) as cantidad_total_repuestos
            FROM series_maquinas sm
            JOIN maquinas m ON sm.maquina_id = m.id
            LEFT JOIN proveedores p ON sm.cliente_id = p.id
            LEFT JOIN solicitud_repuestos sr ON sm.id = sr.serie_id
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
            LEFT JOIN solicitud_repuesto_detalle srd ON sr.id = srd.solicitud_id
            WHERE sm.activo = 1
        ";
        
        // Aplicar filtros
        if (!$is_admin && isset($params[':user_id'])) {
            $sql .= " AND sr.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND sr.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sm.id = :serie_id";
        }
        
        if (!empty($estado_maquina)) {
            $sql .= " AND sm.estado = :estado_maquina";
            $params[':estado_maquina'] = $estado_maquina;
        }
        
        $sql .= " GROUP BY sm.id ORDER BY total_solicitudes DESC LIMIT 15";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $series_solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar los repuestos más solicitados (mejorado para incluir series)
        $sql = "
            SELECT 
                r.id as repuesto_id,
                r.codigo,
                r.nombre as repuesto_nombre,
                SUM(srd.cantidad) as total_solicitado,
                COUNT(DISTINCT sr.id) as total_solicitudes,
                COUNT(DISTINCT sr.serie_id) as total_series
            FROM repuestos r
            JOIN solicitud_repuesto_detalle srd ON r.id = srd.repuesto_id
            JOIN solicitud_repuestos sr ON srd.solicitud_id = sr.id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE r.activo = 1
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin && isset($params[':user_id'])) {
            $sql .= " AND sr.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND sr.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND sr.maquina_id = :maquina_id";
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND sr.serie_id = :serie_id";
        }
        
        $sql .= " GROUP BY r.id ORDER BY total_solicitado DESC LIMIT 10";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $repuestos_top = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes por motivo (mejorado para incluir series)
        $sql = "
            SELECT 
                motivo_solicitud,
                COUNT(id) as total,
                COUNT(DISTINCT maquina_id) as total_maquinas,
                COUNT(DISTINCT serie_id) as total_series
            FROM solicitud_repuestos
            WHERE fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin && isset($params[':user_id'])) {
            $sql .= " AND user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND maquina_id = :maquina_id";
        }
        
        if ($serie_seleccionada > 0) {
            $sql .= " AND serie_id = :serie_id";
        }
        
        $sql .= " GROUP BY motivo_solicitud ORDER BY total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $motivos_solicitud = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico de máquinas
        $labels_maquinas = [];
        $data_maquinas = [];
        
        foreach ($maquinas_solicitudes as $maquina) {
            $labels_maquinas[] = $maquina['maquina_nombre'] . ' (' . $maquina['codigo'] . ')';
            $data_maquinas[] = $maquina['total_solicitudes'];
        }
        
        // Preparar datos para el gráfico de series
        $labels_series = [];
        $data_series = [];
        $colors_series = generarColores(count($series_solicitudes));
        
        foreach ($series_solicitudes as $serie) {
            $labels_series[] = $serie['codigo_serie'] . ' (' . $serie['maquina_nombre'] . ')';
            $data_series[] = $serie['total_solicitudes'];
        }
        
        // Preparar datos para el gráfico de repuestos
        $labels_repuestos = [];
        $data_repuestos = [];
        
        foreach ($repuestos_top as $repuesto) {
            $labels_repuestos[] = $repuesto['repuesto_nombre'] . ' (' . $repuesto['codigo'] . ')';
            $data_repuestos[] = $repuesto['total_solicitado'];
        }
        
        // Preparar datos para el gráfico de motivos
        $labels_motivos = [];
        $data_motivos = [];
        
        foreach ($motivos_solicitud as $motivo) {
            // Convertir el valor de la enum a un formato más legible
            $motivo_legible = str_replace('_', ' ', $motivo['motivo_solicitud']);
            $motivo_legible = ucwords($motivo_legible);
            
            $labels_motivos[] = $motivo_legible;
            $data_motivos[] = $motivo['total'];
        }
        
        $resultados = [
            'maquinas' => $maquinas_solicitudes,
            'series' => $series_solicitudes,
            'repuestos' => $repuestos_top,
            'motivos' => $motivos_solicitud
        ];
        
        $datos_grafico = [
            'labels' => $labels_maquinas,
            'datasets' => [
                [
                    'label' => 'Solicitudes por Máquina',
                    'data' => $data_maquinas,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)'
                ]
            ]
        ];
        
        $datos_grafico_series = [
            'labels' => $labels_series,
            'datasets' => [
                [
                    'label' => 'Solicitudes por Serie',
                    'data' => $data_series,
                    'backgroundColor' => $colors_series
                ]
            ]
        ];
        
        $datos_grafico_repuestos = [
            'labels' => $labels_repuestos,
            'datasets' => [
                [
                    'label' => 'Cantidad Solicitada',
                    'data' => $data_repuestos,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)'
                ]
            ]
        ];
        
        $datos_grafico_motivos = [
            'labels' => $labels_motivos,
            'datasets' => [
                [
                    'label' => 'Solicitudes por Motivo',
                    'data' => $data_motivos,
                    'backgroundColor' => generarColores(count($motivos_solicitud))
                ]
            ]
        ];
    }
    
    // 7. MEJORA: Análisis de Proveedores (incluir relación con series de máquinas)
    elseif ($tipo_reporte === 'analisis_proveedores') {
        $titulo_reporte = 'Análisis de Proveedores';
        $subtitulo_reporte = 'Relación con proveedores, máquinas en comodato y tendencias de compra';
        
        // Construir consulta de órdenes por proveedor registrado
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                p.rut,
                COUNT(oc.id) as total_ordenes,
                SUM(oc.monto_total) as monto_total,
                AVG(oc.monto_total) as promedio_orden,
                MIN(oc.monto_total) as orden_minima,
                MAX(oc.monto_total) as orden_maxima,
                (SELECT COUNT(*) FROM series_maquinas sm WHERE sm.proveedor_id = p.id) as maquinas_proveedor,
                (SELECT COUNT(*) FROM series_maquinas sm WHERE sm.cliente_id = p.id) as maquinas_cliente
            FROM proveedores p
            LEFT JOIN ordenes_compra oc ON p.id = oc.proveedor_id 
                AND oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE p.activo = 1
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
            $params[':proveedor_id'] = $proveedor_seleccionado;
        }
        
        $sql .= " GROUP BY p.id ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $proveedores_analisis = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // NUEVA CONSULTA: Series de máquinas por proveedor
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre as proveedor_nombre,
                sm.estado,
                COUNT(sm.id) as total_series,
                COUNT(DISTINCT m.id) as total_maquinas
            FROM proveedores p
            LEFT JOIN series_maquinas sm ON p.id = sm.proveedor_id OR p.id = sm.cliente_id
            LEFT JOIN maquinas m ON sm.maquina_id = m.id
            WHERE p.activo = 1
        ";
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
            $params_series = [':proveedor_id' => $proveedor_seleccionado];
        } else {
            $params_series = [];
        }
        
        if (!empty($estado_maquina)) {
            $sql .= " AND sm.estado = :estado_maquina";
            $params_series[':estado_maquina'] = $estado_maquina;
        }
        
        $sql .= " GROUP BY p.id, sm.estado ORDER BY p.nombre, sm.estado";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params_series as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $series_por_proveedor = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Obtener tendencia de compras por proveedor (últimos 6 meses)
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                MONTH(oc.fecha) as mes,
                YEAR(oc.fecha) as ano,
                COUNT(oc.id) as total_ordenes,
                SUM(oc.monto_total) as monto_total
            FROM proveedores p
            JOIN ordenes_compra oc ON p.id = oc.proveedor_id
            WHERE oc.fecha >= DATE_SUB(:fecha_hasta, INTERVAL 6 MONTH)
                AND oc.fecha <= :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
        }
        
        $sql .= " GROUP BY p.id, YEAR(oc.fecha), MONTH(oc.fecha) ORDER BY p.id, YEAR(oc.fecha), MONTH(oc.fecha)";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $tendencia_proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calcular tasa de aprobación de órdenes por proveedor
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                COUNT(CASE WHEN oc.estado_id = 3 THEN 1 END) as ordenes_aprobadas,
                COUNT(oc.id) as total_ordenes,
                ROUND(COUNT(CASE WHEN oc.estado_id = 3 THEN 1 END) * 100.0 / COUNT(oc.id), 2) as tasa_aprobacion
            FROM proveedores p
            JOIN ordenes_compra oc ON p.id = oc.proveedor_id
            WHERE oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
        }
        
        $sql .= " GROUP BY p.id ORDER BY tasa_aprobacion DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $aprobacion_proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico de proveedores
        $labels_proveedores = [];
        $data_montos = [];
        $data_ordenes = [];
        $data_maquinas = []; // Nuevo dataset para máquinas en comodato
        
        foreach ($proveedores_analisis as $proveedor) {
            if ($proveedor['total_ordenes'] > 0 || $proveedor['maquinas_cliente'] > 0) { // Solo incluir proveedores relevantes
                $labels_proveedores[] = $proveedor['nombre'];
                $data_montos[] = $proveedor['monto_total'] ?: 0;
                $data_ordenes[] = $proveedor['total_ordenes'];
                $data_maquinas[] = $proveedor['maquinas_cliente'] ?: 0;
            }
        }
        
        // Preparar datos para el gráfico de series por estado
        $estados_únicos = [];
        $proveedores_series = [];
        $datos_series_estado = [];
        
        // Primero obtenemos los estados únicos
        foreach ($series_por_proveedor as $serie) {
            if (!empty($serie['estado']) && !in_array($serie['estado'], $estados_únicos)) {
                $estados_únicos[] = $serie['estado'];
            }
        }
        
        // Luego obtenemos los proveedores únicos
        foreach ($series_por_proveedor as $serie) {
            if (!in_array($serie['proveedor_nombre'], $proveedores_series) && !empty($serie['proveedor_nombre'])) {
                $proveedores_series[] = $serie['proveedor_nombre'];
            }
        }
        
        // Limitar a los primeros 10 proveedores
        $proveedores_series = array_slice($proveedores_series, 0, 10);
        
        // Crear un dataset para cada estado
        foreach ($estados_únicos as $estado) {
            $data_estado = array_fill(0, count($proveedores_series), 0);
            
            foreach ($series_por_proveedor as $index => $serie) {
                if ($serie['estado'] === $estado) {
                    $proveedor_idx = array_search($serie['proveedor_nombre'], $proveedores_series);
                    if ($proveedor_idx !== false) {
                        $data_estado[$proveedor_idx] = $serie['total_series'];
                    }
                }
            }
            
            $datos_series_estado[] = [
                'label' => ucfirst($estado),
                'data' => $data_estado,
                'backgroundColor' => generarColores(1)[array_search($estado, $estados_únicos) % 10]
            ];
        }
        
        // Preparar datos para el gráfico de tendencia de proveedores
        $datos_tendencia = [];
        $proveedores_unicos = [];
        
        // Primero identificar los proveedores únicos
        foreach ($tendencia_proveedores as $tendencia) {
            if (!in_array($tendencia['nombre'], $proveedores_unicos)) {
                $proveedores_unicos[] = $tendencia['nombre'];
            }
        }
        
        // Limitar a los 5 principales proveedores para no sobrecargar el gráfico
        $proveedores_unicos = array_slice($proveedores_unicos, 0, 5);
        
        // Preparar etiquetas para los últimos 6 meses
        $meses_labels = [];
        $fecha_fin = new DateTime($fecha_hasta);
        for ($i = 5; $i >= 0; $i--) {
            $fecha_mes = clone $fecha_fin;
            $fecha_mes->modify("-$i months");
            $meses_labels[] = $fecha_mes->format('M Y');
        }
        
        // Crear datasets para cada proveedor
        foreach ($proveedores_unicos as $index => $proveedor_nombre) {
            $data_meses = array_fill(0, 6, 0); // Inicializar con 0 para los 6 meses
            
            foreach ($tendencia_proveedores as $tendencia) {
                if ($tendencia['nombre'] === $proveedor_nombre) {
                    $fecha_tendencia = new DateTime($tendencia['ano'] . '-' . $tendencia['mes'] . '-01');
                    
                    for ($i = 0; $i < 6; $i++) {
                        $fecha_mes = clone $fecha_fin;
                        $fecha_mes->modify("-$i months");
                        
                        if ($fecha_tendencia->format('Y-m') === $fecha_mes->format('Y-m')) {
                            $data_meses[5 - $i] = $tendencia['monto_total'];
                            break;
                        }
                    }
                }
            }
            
            $datos_tendencia[] = [
                'label' => $proveedor_nombre,
                'data' => $data_meses,
                'borderColor' => generarColores(1)[$index % 10],
                'backgroundColor' => 'transparent',
                'tension' => 0.1
            ];
        }
        
        $resultados = [
            'proveedores' => $proveedores_analisis,
            'series_por_proveedor' => $series_por_proveedor,
            'tendencia' => $tendencia_proveedores,
            'aprobacion' => $aprobacion_proveedores
        ];
        
        $datos_grafico = [
            'labels' => $labels_proveedores,
            'datasets' => [
                [
                    'label' => 'Monto Total',
                    'data' => $data_montos,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)',
                    'yAxisID' => 'y'
                ],
                [
                    'label' => 'Cantidad de Órdenes',
                    'data' => $data_ordenes,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)',
                    'yAxisID' => 'y1'
                ],
                [
                    'label' => 'Máquinas en Comodato',
                    'data' => $data_maquinas,
                    'backgroundColor' => 'rgba(75, 192, 192, 0.7)',
                    'yAxisID' => 'y1'
                ]
            ]
        ];
        
        $datos_grafico_series = [
            'labels' => $proveedores_series,
            'datasets' => $datos_series_estado
        ];
        
        $datos_grafico_tendencia = [
            'labels' => $meses_labels,
            'datasets' => $datos_tendencia
        ];
        
        // Preparar datos para el gráfico de tasa de aprobación
        $labels_aprobacion = [];
        $data_aprobacion = [];
        
        foreach ($aprobacion_proveedores as $proveedor) {
            if ($proveedor['total_ordenes'] >= 3) { // Solo incluir proveedores con al menos 3 órdenes
                $labels_aprobacion[] = $proveedor['nombre'];
                $data_aprobacion[] = $proveedor['tasa_aprobacion'];
            }
        }
        
        $datos_grafico_aprobacion = [
            'labels' => $labels_aprobacion,
            'datasets' => [
                [
                    'label' => 'Tasa de Aprobación (%)',
                    'data' => $data_aprobacion,
                    'backgroundColor' => 'rgba(75, 192, 192, 0.7)'
                ]
            ]
        ];
    }
    
    // 3. Reporte de Órdenes de Compra por Proveedor
    elseif ($tipo_reporte === 'ordenes_proveedor') {
        $titulo_reporte = 'Reporte de Órdenes de Compra por Proveedor';
        $subtitulo_reporte = 'Distribución de órdenes por proveedor';
        
        // Construir consulta
        $sql = "
            SELECT 
                proveedor,
                rut_proveedor,
                COUNT(id) as total_documentos,
                SUM(monto_total) as monto_total,
                AVG(monto_total) as promedio_orden,
                MIN(monto_total) as orden_minima,
                MAX(monto_total) as orden_maxima
            FROM ordenes_compra
            WHERE fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND proveedor_id = :proveedor_id";
            $params[':proveedor_id'] = $proveedor_seleccionado;
        }
        
        $sql .= " GROUP BY proveedor, rut_proveedor ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $proveedores_ordenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Obtener top 10 productos más comprados
        $sql = "
            SELECT 
                producto,
                SUM(cantidad) as total_cantidad,
                COUNT(DISTINCT orden_compra_id) as total_ordenes,
                SUM(subtotal) as monto_total
            FROM detalle_orden_compra doc
            JOIN ordenes_compra oc ON doc.orden_compra_id = oc.id
            WHERE oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND oc.proveedor_id = :proveedor_id";
        }
        
        $sql .= " GROUP BY producto ORDER BY total_cantidad DESC LIMIT 10";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $top_productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico
        $labels = [];
        $data = [];
        $colors = generarColores(count($proveedores_ordenes));
        
        foreach ($proveedores_ordenes as $index => $proveedor) {
            $labels[] = $proveedor['proveedor'];
            $data[] = $proveedor['monto_total'] ?: 0;
        }
        
        $resultados = [
            'proveedores' => $proveedores_ordenes,
            'top_productos' => $top_productos
        ];
        
        $datos_grafico = [
            'labels' => $labels,
            'datasets' => [
                [
                    'data' => $data,
                    'backgroundColor' => $colors
                ]
            ]
        ];
    }
    
    // 4. Reporte por Usuario
    elseif ($tipo_reporte === 'por_usuario') {
        $titulo_reporte = 'Reporte por Usuario';
        $subtitulo_reporte = 'Distribución de documentos por usuario';
        
        // Solo disponible para administradores
        if (!$is_admin) {
            $_SESSION['flash_message'] = "No tienes permisos para ver este reporte";
            $_SESSION['flash_type'] = 'danger';
            header('Location: reportes.php');
            exit;
        }
        
        // Consultar rendiciones por usuario
        $sql = "
            SELECT 
                u.id,
                u.nombre,
                u.apellido,
                u.cargo,
                COUNT(r.id) as total_documentos,
                SUM(r.monto_total) as monto_total,
                AVG(r.monto_total) as promedio_rendicion
            FROM users u
            LEFT JOIN rendiciones r ON u.id = r.user_id 
                AND r.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND r.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($usuario_seleccionado > 0) {
            $sql .= " AND u.id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        $sql .= " GROUP BY u.id ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $rendiciones_por_usuario = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar órdenes de compra por usuario
        $sql = "
            SELECT 
                u.id,
                u.nombre,
                u.apellido,
                u.cargo,
                COUNT(oc.id) as total_documentos,
                SUM(oc.monto_total) as monto_total,
                AVG(oc.monto_total) as promedio_orden
            FROM users u
            LEFT JOIN ordenes_compra oc ON u.id = oc.user_id 
                AND oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
        }
        
        if ($usuario_seleccionado > 0) {
            $sql .= " AND u.id = :user_id";
        }
        
        $sql .= " GROUP BY u.id ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $ordenes_por_usuario = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes de repuestos por usuario
        $sql = "
            SELECT 
                u.id,
                u.nombre,
                u.apellido,
                u.cargo,
                COUNT(sr.id) as total_documentos
            FROM users u
            LEFT JOIN solicitud_repuestos sr ON u.id = sr.user_id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
        }
        
        if ($usuario_seleccionado > 0) {
            $sql .= " AND u.id = :user_id";
        }
        
        $sql .= " GROUP BY u.id ORDER BY total_documentos DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $solicitudes_por_usuario = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para gráficos
        $usuarios_labels = [];
        $rendiciones_data = [];
        $ordenes_data = [];
        $solicitudes_data = [];
        
        // Crear un mapa de usuario_id => datos
        $mapa_usuarios = [];
        foreach ($rendiciones_por_usuario as $usuario) {
            $mapa_usuarios[$usuario['id']] = [
                'nombre' => $usuario['nombre'] . ' ' . $usuario['apellido'],
                'cargo' => $usuario['cargo'],
                'rendiciones' => $usuario['monto_total'] ?: 0,
                'ordenes' => 0,
                'solicitudes' => 0,
                'total_rendiciones' => $usuario['total_documentos'] ?: 0,
                'total_ordenes' => 0,
                'total_solicitudes' => 0
            ];
        }
        
        foreach ($ordenes_por_usuario as $usuario) {
            if (isset($mapa_usuarios[$usuario['id']])) {
                $mapa_usuarios[$usuario['id']]['ordenes'] = $usuario['monto_total'] ?: 0;
                $mapa_usuarios[$usuario['id']]['total_ordenes'] = $usuario['total_documentos'] ?: 0;
            } else {
                $mapa_usuarios[$usuario['id']] = [
                    'nombre' => $usuario['nombre'] . ' ' . $usuario['apellido'],
                    'cargo' => $usuario['cargo'],
                    'rendiciones' => 0,
                    'ordenes' => $usuario['monto_total'] ?: 0,
                    'solicitudes' => 0,
                    'total_rendiciones' => 0,
                    'total_ordenes' => $usuario['total_documentos'] ?: 0,
                    'total_solicitudes' => 0
                ];
            }
        }
        
        foreach ($solicitudes_por_usuario as $usuario) {
            if (isset($mapa_usuarios[$usuario['id']])) {
                $mapa_usuarios[$usuario['id']]['solicitudes'] = 0; // No hay monto asociado a solicitudes
                $mapa_usuarios[$usuario['id']]['total_solicitudes'] = $usuario['total_documentos'] ?: 0;
            } else {
                $mapa_usuarios[$usuario['id']] = [
                    'nombre' => $usuario['nombre'] . ' ' . $usuario['apellido'],
                    'cargo' => $usuario['cargo'],
                    'rendiciones' => 0,
                    'ordenes' => 0,
                    'solicitudes' => 0, 
                    'total_rendiciones' => 0,
                    'total_ordenes' => 0,
                    'total_solicitudes' => $usuario['total_documentos'] ?: 0
                ];
            }
        }
        
        // Ordenar por total combinado
        uasort($mapa_usuarios, function($a, $b) {
            $total_a = $a['rendiciones'] + $a['ordenes'];
            $total_b = $b['rendiciones'] + $b['ordenes'];
            return $total_b <=> $total_a; // Orden descendente
        });
        
        // Limitar a los 10 principales usuarios
        $mapa_usuarios = array_slice($mapa_usuarios, 0, 10, true);
        
        foreach ($mapa_usuarios as $usuario) {
            $usuarios_labels[] = $usuario['nombre'];
            $rendiciones_data[] = $usuario['rendiciones'];
            $ordenes_data[] = $usuario['ordenes'];
            $solicitudes_data[] = $usuario['total_solicitudes'];
        }
        
        $resultados = [
            'rendiciones' => $rendiciones_por_usuario,
            'ordenes' => $ordenes_por_usuario,
            'solicitudes' => $solicitudes_por_usuario,
            'combinado' => $mapa_usuarios
        ];
        
        $datos_grafico = [
            'labels' => $usuarios_labels,
            'datasets' => [
                [
                    'label' => 'Rendiciones',
                    'data' => $rendiciones_data,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)'
                ],
                [
                    'label' => 'Órdenes de Compra',
                    'data' => $ordenes_data,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)'
                ]
            ]
        ];
        
        $datos_grafico_solicitudes = [
            'labels' => $usuarios_labels,
            'datasets' => [
                [
                    'label' => 'Solicitudes de Repuestos',
                    'data' => $solicitudes_data,
                    'backgroundColor' => 'rgba(75, 192, 192, 0.7)'
                ]
            ]
        ];
    }
    
    // 5. Reporte de Evolución Mensual
    elseif ($tipo_reporte === 'evolucion_mensual') {
        $titulo_reporte = 'Evolución Mensual';
        $subtitulo_reporte = 'Tendencia mensual de documentos';
        
        // Definir el año para el reporte
        $ano_reporte = $ano;
        
        // Array para almacenar datos mensuales
        $meses = [];
        for ($i = 1; $i <= 12; $i++) {
            $meses[$i] = [
                'nombre' => date('F', mktime(0, 0, 0, $i, 1, 2000)),
                'rendiciones' => 0,
                'ordenes' => 0,
                'solicitudes' => 0
            ];
        }
        
        // Consultar rendiciones por mes
        $sql = "
            SELECT 
                MONTH(fecha) as mes,
                COUNT(id) as total_documentos,
                SUM(monto_total) as monto_total
            FROM rendiciones
            WHERE YEAR(fecha) = :ano
        ";
        
        $params = [':ano' => $ano_reporte];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($categoria_seleccionada > 0) {
            $sql .= " AND id IN (SELECT rendicion_id FROM rendicion_categoria WHERE categoria_id = :categoria_id)";
            $params[':categoria_id'] = $categoria_seleccionada;
        }
        
        $sql .= " GROUP BY MONTH(fecha)";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $rendiciones_por_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar órdenes de compra por mes
        $sql = "
            SELECT 
                MONTH(fecha) as mes,
                COUNT(id) as total_documentos,
                SUM(monto_total) as monto_total
            FROM ordenes_compra
            WHERE YEAR(fecha) = :ano
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND proveedor_id = :proveedor_id";
            $params[':proveedor_id'] = $proveedor_seleccionado;
        }
        
        $sql .= " GROUP BY MONTH(fecha)";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $ordenes_por_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes de repuestos por mes
        $sql = "
            SELECT 
                MONTH(fecha) as mes,
                COUNT(id) as total_documentos
            FROM solicitud_repuestos
            WHERE YEAR(fecha) = :ano
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND maquina_id = :maquina_id";
            $params[':maquina_id'] = $maquina_seleccionada;
        }
        
        $sql .= " GROUP BY MONTH(fecha)";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $solicitudes_por_mes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Llenar el array de meses con los datos obtenidos
        foreach ($rendiciones_por_mes as $rendicion) {
            $mes = intval($rendicion['mes']);
            if (isset($meses[$mes])) {
                $meses[$mes]['rendiciones'] = $rendicion['monto_total'] ?: 0;
            }
        }
        
        foreach ($ordenes_por_mes as $orden) {
            $mes = intval($orden['mes']);
            if (isset($meses[$mes])) {
                $meses[$mes]['ordenes'] = $orden['monto_total'] ?: 0;
            }
        }
        
        foreach ($solicitudes_por_mes as $solicitud) {
            $mes = intval($solicitud['mes']);
            if (isset($meses[$mes])) {
                $meses[$mes]['solicitudes'] = $solicitud['total_documentos'] ?: 0;
            }
        }
        
        // Preparar datos para gráficos
        $meses_labels = [];
        $rendiciones_data = [];
        $ordenes_data = [];
        $solicitudes_data = [];
        
        foreach ($meses as $mes) {
            $meses_labels[] = $mes['nombre'];
            $rendiciones_data[] = $mes['rendiciones'];
            $ordenes_data[] = $mes['ordenes'];
            $solicitudes_data[] = $mes['solicitudes'];
        }
        
        $resultados = $meses;
        
        $datos_grafico = [
            'labels' => $meses_labels,
            'datasets' => [
                [
                    'label' => 'Rendiciones',
                    'data' => $rendiciones_data,
                    'borderColor' => 'rgba(54, 162, 235, 1)',
                    'backgroundColor' => 'rgba(54, 162, 235, 0.2)',
                    'fill' => true
                ],
                [
                    'label' => 'Órdenes de Compra',
                    'data' => $ordenes_data,
                    'borderColor' => 'rgba(255, 99, 132, 1)',
                    'backgroundColor' => 'rgba(255, 99, 132, 0.2)',
                    'fill' => true
                ]
            ]
        ];
        
        $datos_grafico_solicitudes = [
            'labels' => $meses_labels,
            'datasets' => [
                [
                    'label' => 'Solicitudes de Repuestos',
                    'data' => $solicitudes_data,
                    'borderColor' => 'rgba(75, 192, 192, 1)',
                    'backgroundColor' => 'rgba(75, 192, 192, 0.2)',
                    'fill' => true
                ]
            ]
        ];
    }
    
    // 6. NUEVO: Reporte de Mantenimiento de Máquinas y Repuestos
    elseif ($tipo_reporte === 'mantenimiento_repuestos') {
        $titulo_reporte = 'Mantenimiento y Repuestos';
        $subtitulo_reporte = 'Análisis de solicitudes de repuestos por máquina';
        
        // Construir consulta
        $sql = "
            SELECT 
                m.id as maquina_id,
                m.codigo,
                m.nombre as maquina_nombre,
                COUNT(sr.id) as total_solicitudes,
                COUNT(DISTINCT sr.user_id) as total_usuarios
            FROM maquinas m
            LEFT JOIN solicitud_repuestos sr ON m.id = sr.maquina_id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE m.activo = 1
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND sr.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND sr.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND m.id = :maquina_id";
            $params[':maquina_id'] = $maquina_seleccionada;
        }
        
        $sql .= " GROUP BY m.id ORDER BY total_solicitudes DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $maquinas_solicitudes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar los repuestos más solicitados
        $sql = "
            SELECT 
                r.id as repuesto_id,
                r.codigo,
                r.nombre as repuesto_nombre,
                SUM(srd.cantidad) as total_solicitado,
                COUNT(DISTINCT sr.id) as total_solicitudes
            FROM repuestos r
            JOIN solicitud_repuesto_detalle srd ON r.id = srd.repuesto_id
            JOIN solicitud_repuestos sr ON srd.solicitud_id = sr.id 
                AND sr.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE r.activo = 1
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND sr.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND sr.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND sr.estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND sr.maquina_id = :maquina_id";
        }
        
        $sql .= " GROUP BY r.id ORDER BY total_solicitado DESC LIMIT 10";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $repuestos_top = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Consultar solicitudes por motivo
        $sql = "
            SELECT 
                motivo_solicitud,
                COUNT(id) as total,
                COUNT(DISTINCT maquina_id) as total_maquinas
            FROM solicitud_repuestos
            WHERE fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND estado_id = :estado_id";
        }
        
        if ($maquina_seleccionada > 0) {
            $sql .= " AND maquina_id = :maquina_id";
        }
        
        $sql .= " GROUP BY motivo_solicitud ORDER BY total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $motivos_solicitud = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico de máquinas
        $labels_maquinas = [];
        $data_maquinas = [];
        
        foreach ($maquinas_solicitudes as $maquina) {
            $labels_maquinas[] = $maquina['maquina_nombre'] . ' (' . $maquina['codigo'] . ')';
            $data_maquinas[] = $maquina['total_solicitudes'];
        }
        
        // Preparar datos para el gráfico de repuestos
        $labels_repuestos = [];
        $data_repuestos = [];
        
        foreach ($repuestos_top as $repuesto) {
            $labels_repuestos[] = $repuesto['repuesto_nombre'] . ' (' . $repuesto['codigo'] . ')';
            $data_repuestos[] = $repuesto['total_solicitado'];
        }
        
        // Preparar datos para el gráfico de motivos
        $labels_motivos = [];
        $data_motivos = [];
        
        foreach ($motivos_solicitud as $motivo) {
            // Convertir el valor de la enum a un formato más legible
            $motivo_legible = str_replace('_', ' ', $motivo['motivo_solicitud']);
            $motivo_legible = ucwords($motivo_legible);
            
            $labels_motivos[] = $motivo_legible;
            $data_motivos[] = $motivo['total'];
        }
        
        $resultados = [
            'maquinas' => $maquinas_solicitudes,
            'repuestos' => $repuestos_top,
            'motivos' => $motivos_solicitud
        ];
        
        $datos_grafico = [
            'labels' => $labels_maquinas,
            'datasets' => [
                [
                    'label' => 'Solicitudes por Máquina',
                    'data' => $data_maquinas,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)'
                ]
            ]
        ];
        
        $datos_grafico_repuestos = [
            'labels' => $labels_repuestos,
            'datasets' => [
                [
                    'label' => 'Cantidad Solicitada',
                    'data' => $data_repuestos,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)'
                ]
            ]
        ];
        
        $datos_grafico_motivos = [
            'labels' => $labels_motivos,
            'datasets' => [
                [
                    'label' => 'Solicitudes por Motivo',
                    'data' => $data_motivos,
                    'backgroundColor' => generarColores(count($motivos_solicitud))
                ]
            ]
        ];
    }
    
    // 7. NUEVO: Análisis de Proveedores
    elseif ($tipo_reporte === 'analisis_proveedores') {
        $titulo_reporte = 'Análisis de Proveedores';
        $subtitulo_reporte = 'Relación con proveedores y tendencias de compra';
        
        // Construir consulta de órdenes por proveedor registrado
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                p.rut,
                COUNT(oc.id) as total_ordenes,
                SUM(oc.monto_total) as monto_total,
                AVG(oc.monto_total) as promedio_orden,
                MIN(oc.monto_total) as orden_minima,
                MAX(oc.monto_total) as orden_maxima
            FROM proveedores p
            LEFT JOIN ordenes_compra oc ON p.id = oc.proveedor_id 
                AND oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
            WHERE p.activo = 1
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
            $params[':proveedor_id'] = $proveedor_seleccionado;
        }
        
        $sql .= " GROUP BY p.id ORDER BY monto_total DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $proveedores_analisis = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Obtener tendencia de compras por proveedor (últimos 6 meses)
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                MONTH(oc.fecha) as mes,
                YEAR(oc.fecha) as ano,
                COUNT(oc.id) as total_ordenes,
                SUM(oc.monto_total) as monto_total
            FROM proveedores p
            JOIN ordenes_compra oc ON p.id = oc.proveedor_id
            WHERE oc.fecha >= DATE_SUB(:fecha_hasta, INTERVAL 6 MONTH)
                AND oc.fecha <= :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND oc.estado_id = :estado_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
        }
        
        $sql .= " GROUP BY p.id, YEAR(oc.fecha), MONTH(oc.fecha) ORDER BY p.id, YEAR(oc.fecha), MONTH(oc.fecha)";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $tendencia_proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calcular tasa de aprobación de órdenes por proveedor
        $sql = "
            SELECT 
                p.id as proveedor_id,
                p.nombre,
                COUNT(CASE WHEN oc.estado_id = 3 THEN 1 END) as ordenes_aprobadas,
                COUNT(oc.id) as total_ordenes,
                ROUND(COUNT(CASE WHEN oc.estado_id = 3 THEN 1 END) * 100.0 / COUNT(oc.id), 2) as tasa_aprobacion
            FROM proveedores p
            JOIN ordenes_compra oc ON p.id = oc.proveedor_id
            WHERE oc.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND oc.user_id = :user_id";
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND oc.user_id = :user_id";
        }
        
        if ($proveedor_seleccionado > 0) {
            $sql .= " AND p.id = :proveedor_id";
        }
        
        $sql .= " GROUP BY p.id ORDER BY tasa_aprobacion DESC";
        
        $stmt = $conn->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $aprobacion_proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para el gráfico de proveedores
        $labels_proveedores = [];
        $data_montos = [];
        $data_ordenes = [];
        
        foreach ($proveedores_analisis as $proveedor) {
            if ($proveedor['total_ordenes'] > 0) { // Solo incluir proveedores con órdenes
                $labels_proveedores[] = $proveedor['nombre'];
                $data_montos[] = $proveedor['monto_total'] ?: 0;
                $data_ordenes[] = $proveedor['total_ordenes'];
            }
        }
        
        // Preparar datos para el gráfico de tendencia de proveedores
        $datos_tendencia = [];
        $proveedores_unicos = [];
        
        // Primero identificar los proveedores únicos
        foreach ($tendencia_proveedores as $tendencia) {
            if (!in_array($tendencia['nombre'], $proveedores_unicos)) {
                $proveedores_unicos[] = $tendencia['nombre'];
            }
        }
        
        // Limitar a los 5 principales proveedores para no sobrecargar el gráfico
        $proveedores_unicos = array_slice($proveedores_unicos, 0, 5);
        
        // Preparar etiquetas para los últimos 6 meses
        $meses_labels = [];
        $fecha_fin = new DateTime($fecha_hasta);
        for ($i = 5; $i >= 0; $i--) {
            $fecha_mes = clone $fecha_fin;
            $fecha_mes->modify("-$i months");
            $meses_labels[] = $fecha_mes->format('M Y');
        }
        
        // Crear datasets para cada proveedor
        foreach ($proveedores_unicos as $index => $proveedor_nombre) {
            $data_meses = array_fill(0, 6, 0); // Inicializar con 0 para los 6 meses
            
            foreach ($tendencia_proveedores as $tendencia) {
                if ($tendencia['nombre'] === $proveedor_nombre) {
                    $fecha_tendencia = new DateTime($tendencia['ano'] . '-' . $tendencia['mes'] . '-01');
                    
                    for ($i = 0; $i < 6; $i++) {
                        $fecha_mes = clone $fecha_fin;
                        $fecha_mes->modify("-$i months");
                        
                        if ($fecha_tendencia->format('Y-m') === $fecha_mes->format('Y-m')) {
                            $data_meses[5 - $i] = $tendencia['monto_total'];
                            break;
                        }
                    }
                }
            }
            
            $datos_tendencia[] = [
                'label' => $proveedor_nombre,
                'data' => $data_meses,
                'borderColor' => generarColores(1)[$index % 10],
                'backgroundColor' => 'transparent',
                'tension' => 0.1
            ];
        }
        
        $resultados = [
            'proveedores' => $proveedores_analisis,
            'tendencia' => $tendencia_proveedores,
            'aprobacion' => $aprobacion_proveedores
        ];
        
        $datos_grafico = [
            'labels' => $labels_proveedores,
            'datasets' => [
                [
                    'label' => 'Monto Total',
                    'data' => $data_montos,
                    'backgroundColor' => 'rgba(54, 162, 235, 0.7)',
                    'yAxisID' => 'y'
                ],
                [
                    'label' => 'Cantidad de Órdenes',
                    'data' => $data_ordenes,
                    'backgroundColor' => 'rgba(255, 99, 132, 0.7)',
                    'yAxisID' => 'y1'
                ]
            ]
        ];
        
        $datos_grafico_tendencia = [
            'labels' => $meses_labels,
            'datasets' => $datos_tendencia
        ];
        
        // Preparar datos para el gráfico de tasa de aprobación
        $labels_aprobacion = [];
        $data_aprobacion = [];
        
        foreach ($aprobacion_proveedores as $proveedor) {
            if ($proveedor['total_ordenes'] >= 3) { // Solo incluir proveedores con al menos 3 órdenes
                $labels_aprobacion[] = $proveedor['nombre'];
                $data_aprobacion[] = $proveedor['tasa_aprobacion'];
            }
        }
        
        $datos_grafico_aprobacion = [
            'labels' => $labels_aprobacion,
            'datasets' => [
                [
                    'label' => 'Tasa de Aprobación (%)',
                    'data' => $data_aprobacion,
                    'backgroundColor' => 'rgba(75, 192, 192, 0.7)'
                ]
            ]
        ];
    }
    
    // 8. NUEVO: Reporte de Devoluciones
    elseif ($tipo_reporte === 'devoluciones') {
        $titulo_reporte = 'Análisis de Devoluciones';
        $subtitulo_reporte = 'Detalle y estadísticas de devoluciones';
        
        // Construir consulta principal
        $sql = "
            SELECT 
                d.id,
                d.fecha,
                d.monto_total,
                d.motivo_id,
                md.nombre as motivo_nombre,
                d.estado_id,
                e.nombre as estado,
                u.nombre as usuario_nombre,
                u.apellido as usuario_apellido
            FROM devoluciones d
            LEFT JOIN estados e ON d.estado_id = e.id
            LEFT JOIN users u ON d.user_id = u.id
            LEFT JOIN motivos_devolucion md ON d.motivo_id = md.id
            WHERE d.fecha BETWEEN :fecha_desde AND :fecha_hasta
        ";
        
        $params = [
            ':fecha_desde' => $fecha_desde,
            ':fecha_hasta' => $fecha_hasta
        ];
        
        // Agregar filtros adicionales
        if (!$is_admin) {
            $sql .= " AND d.user_id = :user_id";
            $params[':user_id'] = $user_id;
        } elseif ($usuario_seleccionado > 0) {
            $sql .= " AND d.user_id = :user_id";
            $params[':user_id'] = $usuario_seleccionado;
        }
        
        if ($estado_seleccionado > 0) {
            $sql .= " AND d.estado_id = :estado_id";
            $params[':estado_id'] = $estado_seleccionado;
        }
        
        $sql .= " ORDER BY d.fecha DESC";
        
        $stmt = $conn->prepare($sql);
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        $stmt->execute();
        $devoluciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Estadísticas por estado
        $sql = "
            SELECT 
                e.nombre as estado,
                COUNT(d.id) as total,
                SUM(d.monto_total) as monto_total,
                AVG(d.monto_total) as promedio
            FROM estados e
            LEFT JOIN devoluciones d ON e.id = d.estado_id 
                AND d.fecha BETWEEN :fecha_desde AND :fecha_hasta
            " . (!$is_admin ? "AND d.user_id = :user_id" : "") . "
            GROUP BY e.id
            ORDER BY total DESC
        ";
        
        $stmt = $conn->prepare($sql);
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if (!$is_admin) {
            $stmt->bindValue(':user_id', $user_id);
        }
        $stmt->execute();
        $estados_devoluciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Estadísticas por motivo
        $sql = "
            SELECT 
                md.nombre as motivo,
                COUNT(d.id) as total,
                SUM(d.monto_total) as monto_total,
                AVG(d.monto_total) as promedio
            FROM motivos_devolucion md
            LEFT JOIN devoluciones d ON md.id = d.motivo_id 
                AND d.fecha BETWEEN :fecha_desde AND :fecha_hasta
                " . (!$is_admin ? "AND d.user_id = :user_id" : "") . "
            WHERE md.activo = 1
            GROUP BY md.id, md.nombre
            ORDER BY total DESC
        ";
        
        $stmt = $conn->prepare($sql);
        $stmt->bindValue(':fecha_desde', $fecha_desde);
        $stmt->bindValue(':fecha_hasta', $fecha_hasta);
        if (!$is_admin) {
            $stmt->bindValue(':user_id', $user_id);
        }
        $stmt->execute();
        $motivos_devoluciones = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Preparar datos para gráficos
        $labels_estados = [];
        $data_estados = [];
        $colors_estados = [];
        
        foreach ($estados_devoluciones as $estado) {
            $labels_estados[] = $estado['estado'];
            $data_estados[] = $estado['total'];
            $colors_estados[] = generarColores(1)[0];
        }
        
        $labels_motivos = [];
        $data_motivos = [];
        $colors_motivos = [];
        
        foreach ($motivos_devoluciones as $motivo) {
            $labels_motivos[] = $motivo['motivo'];
            $data_motivos[] = $motivo['total'];
            $colors_motivos[] = generarColores(1)[0];
        }
        
        $resultados = [
            'devoluciones' => $devoluciones,
            'estados' => $estados_devoluciones,
            'motivos' => $motivos_devoluciones
        ];
        
        $datos_grafico = [
            'labels' => $labels_estados,
            'datasets' => [
                [
                    'label' => 'Devoluciones por Estado',
                    'data' => $data_estados,
                    'backgroundColor' => $colors_estados
                ]
            ]
        ];
        
        $datos_grafico_motivos = [
            'labels' => $labels_motivos,
            'datasets' => [
                [
                    'label' => 'Devoluciones por Motivo',
                    'data' => $data_motivos,
                    'backgroundColor' => $colors_motivos
                ]
            ]
        ];
    }
    
    // Exportación a Excel
    if ($formato_exportacion === 'excel') {
        // Implementar exportación a Excel aquí
        // Por simplicidad, esta funcionalidad se podría implementar en un archivo separado
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    /* Estilos personalizados para reportes */
    .filtros-container {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 20px;
    }
    
    .filtro-item {
        flex: 1;
        min-width: 200px;
    }
    
    .reporte-card {
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        padding: 20px;
        margin-bottom: 20px;
        transition: all 0.3s;
    }
    
    .reporte-card:hover {
        box-shadow: 0 6px 12px rgba(0, 0, 0, 0.15);
    }
    
    .reporte-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 15px;
        border-bottom: 1px solid #eee;
        padding-bottom: 15px;
    }
    
    .reporte-title {
        font-size: 18px;
        font-weight: 600;
        margin: 0;
    }
    
    .reporte-subtitle {
        color: #6c757d;
        font-size: 14px;
        margin-top: 5px;
    }
    
    .stat-cards {
        display: flex;
        flex-wrap: wrap;
        gap: 15px;
        margin-bottom: 20px;
    }
    
    .stat-card {
        flex: 1;
        min-width: 200px;
        background-color: #f8f9fa;
        border-radius: 6px;
        padding: 15px;
        border-left: 4px solid #1e88e5;
    }
    
    .stat-card.rendiciones {
        border-left-color: #1e88e5;
    }
    
    .stat-card.ordenes {
        border-left-color: #ef5350;
    }
    
    .stat-card.solicitudes {
        border-left-color: #26a69a;
    }
    
    .stat-card.devoluciones {
        border-left-color: #ff9800;
    }
    
    .stat-card.series {
        border-left-color: #9c27b0;
    }
    
    .stat-card h3 {
        font-size: 16px;
        margin: 0 0 10px 0;
        color: #495057;
    }
    
    .stat-value {
        font-size: 24px;
        font-weight: 700;
        margin-bottom: 5px;
        color: #333;
    }
    
    .stat-label {
        font-size: 13px;
        color: #6c757d;
    }
    
    .chart-container {
        position: relative;
        height: 350px;
        margin-bottom: 20px;
    }
    
    .chart-container-small {
        position: relative;
        height: 250px;
        margin-bottom: 20px;
    }
    
    .table-reportes {
        width: 100%;
        border-collapse: collapse;
        margin-bottom: 20px;
    }
    
    .table-reportes th, .table-reportes td {
        padding: 10px;
        border: 1px solid #ddd;
    }
    
    .table-reportes th {
        background-color: #f8f9fa;
        font-weight: 600;
    }
    
    .table-reportes tr:nth-child(even) {
        background-color: #f9f9f9;
    }
    
    .table-reportes tr:hover {
        background-color: rgba(0,0,0,0.03);
    }
    
    .tipo-reporte-selector {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 20px;
        justify-content: center;
    }
    
    .tipo-reporte-btn {
        display: flex;
        flex-direction: column;
        align-items: center;
        padding: 15px;
        border-radius: 8px;
        border: 1px solid #ddd;
        background-color: white;
        cursor: pointer;
        transition: all 0.3s;
        flex: 1;
        min-width: 150px;
        max-width: 200px;
        text-decoration: none;
        color: #333;
    }
    
    .tipo-reporte-btn:hover, .tipo-reporte-btn.active {
        background-color: #f8f9fa;
        border-color: #1e88e5;
        transform: translateY(-3px);
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    }
    
    .tipo-reporte-btn.active {
        background-color: #e3f2fd;
    }
    
    .tipo-reporte-btn i {
        font-size: 28px;
        margin-bottom: 10px;
        color: #1e88e5;
    }
    
    .tipo-reporte-btn h4 {
        font-size: 14px;
        margin: 0;
        text-align: center;
    }
    
    .tipo-reporte-btn p {
        font-size: 12px;
        color: #6c757d;
        margin: 5px 0 0 0;
        text-align: center;
    }
    
    .report-columns {
        display: flex;
        flex-wrap: wrap;
        gap: 20px;
        margin-bottom: 20px;
    }
    
    .report-column {
        flex: 1;
        min-width: 300px;
    }
    
    .status-badge {
        display: inline-block;
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 13px;
    }
    
    .tabs-container {
        margin-bottom: 20px;
    }
    
    .tab-buttons {
        display: flex;
        gap: 5px;
        margin-bottom: 15px;
    }
    
    .tab-button {
        padding: 10px 15px;
        background-color: #f8f9fa;
        border: 1px solid #ddd;
        border-radius: 4px;
        cursor: pointer;
        transition: all 0.3s;
    }
    
    .tab-button.active {
        background-color: #e3f2fd;
        border-color: #1e88e5;
        color: #1e88e5;
    }
    
    .tab-content {
        display: none;
    }
    
    .tab-content.active {
        display: block;
    }
    
    @media (max-width: 768px) {
        .tipo-reporte-btn {
            min-width: 120px;
        }
        
        .report-column {
            min-width: 100%;
        }
    }
    
    @media (max-width: 576px) {
        .tipo-reporte-btn {
            min-width: 100%;
        }
    }

    /* Estilos para el indicador de carga */
    .loading-indicator {
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background: rgba(255, 255, 255, 0.9);
        padding: 20px;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        z-index: 9999;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .spin {
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<div class="card">
    <div class="panel-header">
        <h2 class="panel-title">Reportes y Estadísticas</h2>
        <div>
            <?php if (!empty($resultados)): ?>
            <a href="reportes.php?<?php echo http_build_query($_GET) . '&exportar=excel'; ?>" 
               class="btn btn-success btn-sm">
                <i class="material-icons">file_download</i> Exportar a Excel
            </a>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Selector de tipo de reporte -->
    <div class="tipo-reporte-selector">
        <a href="reportes.php?tipo_reporte=general<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'general' ? 'active' : ''; ?>">
            <i class="material-icons">dashboard</i>
            <h4>General</h4>
            <p>Resumen global</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=rendiciones_categoria<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'rendiciones_categoria' ? 'active' : ''; ?>">
            <i class="material-icons">receipt</i>
            <h4>Rendiciones</h4>
            <p>Por categoría</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=ordenes_proveedor<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'ordenes_proveedor' ? 'active' : ''; ?>">
            <i class="material-icons">shopping_cart</i>
            <h4>Órdenes</h4>
            <p>Por proveedor</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=series_maquinas<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'series_maquinas' ? 'active' : ''; ?>">
            <i class="material-icons">router</i>
            <h4>Series</h4>
            <p>Detalle de series</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=devoluciones<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'devoluciones' ? 'active' : ''; ?>">
            <i class="material-icons">assignment_return</i>
            <h4>Devoluciones</h4>
            <p>Análisis</p>
        </a>
        
        <?php if ($is_admin): ?>
        <a href="reportes.php?tipo_reporte=por_usuario<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'por_usuario' ? 'active' : ''; ?>">
            <i class="material-icons">people</i>
            <h4>Usuarios</h4>
            <p>Comparativa</p>
        </a>
        <?php endif; ?>
        
        <a href="reportes.php?tipo_reporte=evolucion_mensual<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'evolucion_mensual' ? 'active' : ''; ?>">
            <i class="material-icons">trending_up</i>
            <h4>Evolución</h4>
            <p>Tendencia mensual</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=mantenimiento_repuestos<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'mantenimiento_repuestos' ? 'active' : ''; ?>">
            <i class="material-icons">build</i>
            <h4>Repuestos</h4>
            <p>Mantenimiento</p>
        </a>
        
        <a href="reportes.php?tipo_reporte=analisis_proveedores<?php echo isset($_GET['periodo']) ? '&periodo=' . $_GET['periodo'] : ''; ?>" 
           class="tipo-reporte-btn <?php echo $tipo_reporte === 'analisis_proveedores' ? 'active' : ''; ?>">
            <i class="material-icons">business</i>
            <h4>Proveedores</h4>
            <p>Análisis</p>
        </a>
    </div>
    
    <!-- Filtros -->
    <div class="card" style="margin-bottom: 20px;">
        <div class="panel-header">
            <h3 class="panel-title">Filtros</h3>
        </div>
        
        <form method="GET" action="reportes.php" id="filtros-form">
            <input type="hidden" name="tipo_reporte" value="<?php echo $tipo_reporte; ?>">
            
            <div class="filtros-container">
                <?php if ($tipo_reporte !== 'series_maquinas'): ?>
                <div class="filtro-item">
                    <label for="periodo" class="form-label">Período</label>
                    <select id="periodo" name="periodo" class="form-control">
                        <option value="mes" <?php echo $periodo === 'mes' ? 'selected' : ''; ?>>Mes</option>
                        <option value="trimestre" <?php echo $periodo === 'trimestre' ? 'selected' : ''; ?>>Trimestre</option>
                        <option value="ano" <?php echo $periodo === 'ano' ? 'selected' : ''; ?>>Año</option>
                        <option value="personalizado" <?php echo $periodo === 'personalizado' ? 'selected' : ''; ?>>Personalizado</option>
                    </select>
                </div>
                
                <div class="filtro-item" id="mes-container" <?php echo $periodo !== 'mes' && $periodo !== 'trimestre' ? 'style="display:none"' : ''; ?>>
                    <label for="mes" class="form-label">Mes</label>
                    <select id="mes" name="mes" class="form-control">
                        <?php for ($i = 1; $i <= 12; $i++): ?>
                        <option value="<?php echo str_pad($i, 2, '0', STR_PAD_LEFT); ?>" <?php echo $mes == str_pad($i, 2, '0', STR_PAD_LEFT) ? 'selected' : ''; ?>>
                            <?php echo date('F', mktime(0, 0, 0, $i, 1, 2000)); ?>
                        </option>
                        <?php endfor; ?>
                    </select>
                </div>
                
                <div class="filtro-item">
                    <label for="ano" class="form-label">Año</label>
                    <select id="ano" name="ano" class="form-control">
                        <?php for ($i = date('Y'); $i >= date('Y') - 5; $i--): ?>
                        <option value="<?php echo $i; ?>" <?php echo $ano == $i ? 'selected' : ''; ?>>
                            <?php echo $i; ?>
                        </option>
                        <?php endfor; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($is_admin): ?>
                <div class="filtro-item">
                    <label for="usuario" class="form-label">Usuario</label>
                    <select id="usuario" name="usuario" class="form-control">
                        <option value="0">Todos los usuarios</option>
                        <?php foreach ($usuarios as $usuario): ?>
                        <option value="<?php echo $usuario['id']; ?>" <?php echo $usuario_seleccionado == $usuario['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($tipo_reporte === 'rendiciones_categoria' || $tipo_reporte === 'general'): ?>
                <div class="filtro-item">
                    <label for="categoria" class="form-label">Categoría</label>
                    <select id="categoria" name="categoria" class="form-control">
                        <option value="0">Todas las categorías</option>
                        <?php foreach ($categorias as $categoria): ?>
                        <option value="<?php echo $categoria['id']; ?>" <?php echo $categoria_seleccionada == $categoria['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($categoria['nombre']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($tipo_reporte === 'mantenimiento_repuestos' || $tipo_reporte === 'series_maquinas'): ?>
                <div class="filtro-item">
                    <label for="maquina" class="form-label">Máquina</label>
                    <select id="maquina" name="maquina" class="form-control">
                        <option value="0">Todas las máquinas</option>
                        <?php foreach ($maquinas as $maquina): ?>
                        <option value="<?php echo $maquina['id']; ?>" <?php echo $maquina_seleccionada == $maquina['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($maquina['nombre'] . ' (' . $maquina['codigo'] . ')'); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filtro-item">
                    <label for="serie" class="form-label">Serie</label>
                    <select id="serie" name="serie" class="form-control">
                        <option value="0">Todas las series</option>
                        <?php foreach ($series as $serie): ?>
                        <option value="<?php echo $serie['id']; ?>" <?php echo $serie_seleccionada == $serie['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($serie['codigo_serie'] . ' - ' . $serie['maquina_nombre']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filtro-item">
                    <label for="estado_maquina" class="form-label">Estado de Máquina</label>
                    <select id="estado_maquina" name="estado_maquina" class="form-control">
                        <option value="">Todos los estados</option>
                        <?php foreach ($estados_maquina as $estado): ?>
                        <option value="<?php echo $estado['nombre']; ?>" <?php echo $estado_maquina === $estado['nombre'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars(ucfirst($estado['nombre'])); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <?php if ($tipo_reporte === 'ordenes_proveedor' || $tipo_reporte === 'analisis_proveedores'): ?>
                <div class="filtro-item">
                    <label for="proveedor" class="form-label">Proveedor</label>
                    <select id="proveedor" name="proveedor" class="form-control">
                        <option value="0">Todos los proveedores</option>
                        <?php foreach ($proveedores as $proveedor): ?>
                        <option value="<?php echo $proveedor['id']; ?>" <?php echo $proveedor_seleccionado == $proveedor['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($proveedor['nombre'] . ' (' . $proveedor['rut'] . ')'); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php endif; ?>
                
                <div class="filtro-item">
                    <label for="estado" class="form-label">Estado</label>
                    <select id="estado" name="estado" class="form-control">
                        <option value="0">Todos los estados</option>
                        <?php foreach ($estados as $estado): ?>
                        <option value="<?php echo $estado['id']; ?>" <?php echo $estado_seleccionado == $estado['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($estado['nombre']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <?php if ($tipo_reporte !== 'series_maquinas'): ?>
                <div class="filtro-item fecha-container" <?php echo $periodo !== 'personalizado' ? 'style="display:none"' : ''; ?>>
                    <label for="fecha_desde" class="form-label">Desde</label>
                    <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" 
                           value="<?php echo $fecha_desde; ?>">
                </div>
                
                <div class="filtro-item fecha-container" <?php echo $periodo !== 'personalizado' ? 'style="display:none"' : ''; ?>>
                    <label for="fecha_hasta" class="form-label">Hasta</label>
                    <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" 
                           value="<?php echo $fecha_hasta; ?>">
                </div>
                <?php endif; ?>
            </div>
            
            <div style="text-align: center;">
                <button type="submit" class="btn btn-primary">
                    <i class="material-icons">search</i> Generar Reporte
                </button>
            </div>
        </form>
    </div>
    
    <?php if (!empty($resultados)): ?>
    <!-- Resultados del reporte -->
    <div class="reporte-card">
        <div class="reporte-header">
            <div>
                <h2 class="reporte-title"><?php echo $titulo_reporte; ?></h2>
                <p class="reporte-subtitle"><?php echo $subtitulo_reporte; ?></p>
            </div>
            <div>
                <?php if ($tipo_reporte !== 'series_maquinas'): ?>
                <span>Período: 
                    <?php 
                    if ($periodo === 'mes') {
                        echo date('F Y', strtotime($ano . '-' . $mes . '-01'));
                    } elseif ($periodo === 'trimestre') {
                        $trimestre = ceil(intval($mes) / 3);
                        echo 'T' . $trimestre . ' ' . $ano;
                    } elseif ($periodo === 'ano') {
                        echo $ano;
                    } else {
                        echo formatoFecha($fecha_desde) . ' a ' . formatoFecha($fecha_hasta);
                    }
                    ?>
                </span>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if ($tipo_reporte === 'general'): ?>
            <!-- Código del reporte general -->
            <div class="stat-cards">
                <div class="stat-card rendiciones">
                    <h3>Rendiciones</h3>
                    <div class="stat-value"><?php echo formatoMoneda($resultados['rendiciones']['monto_total'] ?? 0); ?></div>
                    <div class="stat-label"><?php echo $resultados['rendiciones']['total_documentos'] ?? 0; ?> documentos</div>
                </div>
                
                <div class="stat-card ordenes">
                    <h3>Órdenes de Compra</h3>
                    <div class="stat-value"><?php echo formatoMoneda($resultados['ordenes']['monto_total'] ?? 0); ?></div>
                    <div class="stat-label"><?php echo $resultados['ordenes']['total_documentos'] ?? 0; ?> documentos</div>
                </div>
                
                <div class="stat-card solicitudes">
                    <h3>Solicitudes de Repuestos</h3>
                    <div class="stat-value"><?php echo $resultados['solicitudes']['total_documentos'] ?? 0; ?></div>
                    <div class="stat-label">Documentos gestionados</div>
                </div>
                
                <div class="stat-card devoluciones">
                    <h3>Devoluciones</h3>
                    <div class="stat-value"><?php echo formatoMoneda($resultados['devoluciones']['monto_total'] ?? 0); ?></div>
                    <div class="stat-label"><?php echo $resultados['devoluciones']['total_documentos'] ?? 0; ?> documentos</div>
                </div>
                
                <div class="stat-card">
                    <h3>Total</h3>
                    <div class="stat-value">
                        <?php 
                        $total_general = ($resultados['rendiciones']['monto_total'] ?? 0) + 
                                         ($resultados['ordenes']['monto_total'] ?? 0) +
                                         ($resultados['devoluciones']['monto_total'] ?? 0);
                        echo formatoMoneda($total_general);
                        ?>
                    </div>
                    <div class="stat-label">
                        <?php 
                        $total_documentos = ($resultados['rendiciones']['total_documentos'] ?? 0) + 
                                            ($resultados['ordenes']['total_documentos'] ?? 0) +
                                            ($resultados['devoluciones']['total_documentos'] ?? 0);
                        echo $total_documentos; ?> documentos
                    </div>
                </div>
            </div>
            
            <div class="chart-container">
                <canvas id="estadosChart"></canvas>
            </div>
            
            <h3>Distribución por Estado</h3>
            <div class="table-responsive">
                <table class="table-reportes">
                    <thead>
                        <tr>
                            <th>Estado</th>
                            <th>Rendiciones</th>
                            <th>Órdenes de Compra</th>
                            <th>Solicitudes</th>
                            <th>Devoluciones</th>
                            <th>Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($resultados['estados_rendiciones'] as $index => $estado): ?>
                        <tr>
                            <td>
                                <span class="status-badge" style="background-color: <?php echo $estado['color']; ?>20; color: <?php echo $estado['color']; ?>;">
                                    <?php echo $estado['nombre']; ?>
                                </span>
                            </td>
                            <td><?php echo $estado['total']; ?></td>
                            <td><?php echo $resultados['estados_ordenes'][$index]['total']; ?></td>
                            <td><?php echo $resultados['estados_solicitudes'][$index]['total']; ?></td>
                            <td><?php echo $resultados['estados_devoluciones'][$index]['total']; ?></td>
                            <td><?php echo $estado['total'] + $resultados['estados_ordenes'][$index]['total'] + $resultados['estados_solicitudes'][$index]['total'] + $resultados['estados_devoluciones'][$index]['total']; ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
        <?php elseif ($tipo_reporte === 'series_maquinas'): ?>
            <!-- Reporte de Series de Máquinas (NUEVO) -->
            <div class="stat-cards">
                <div class="stat-card series">
                    <h3>Series Registradas</h3>
                    <div class="stat-value"><?php echo count($resultados['series_maquinas']); ?></div>
                    <div class="stat-label">Total de series en el sistema</div>
                </div>
                
                <?php
                $series_comodato = 0;
                $series_servicio = 0;
                $series_nuevas = 0;
                
                foreach ($resultados['estados_series'] as $estado) {
                    if ($estado['estado'] === 'comodato') {
                        $series_comodato = $estado['total_series'];
                    } elseif ($estado['estado'] === 'servicio_tecnico') {
                        $series_servicio = $estado['total_series'];
                    } elseif ($estado['estado'] === 'nueva') {
                        $series_nuevas = $estado['total_series'];
                    }
                }
                ?>
                
                <div class="stat-card rendiciones">
                    <h3>En Comodato</h3>
                    <div class="stat-value"><?php echo $series_comodato; ?></div>
                    <div class="stat-label">Series en comodato con clientes</div>
                </div>
                
                <div class="stat-card ordenes">
                    <h3>En Servicio Técnico</h3>
                    <div class="stat-value"><?php echo $series_servicio; ?></div>
                    <div class="stat-label">Series en reparación/mantenimiento</div>
                </div>
                
                <div class="stat-card solicitudes">
                    <h3>Nuevas</h3>
                    <div class="stat-value"><?php echo $series_nuevas; ?></div>
                    <div class="stat-label">Series disponibles sin usar</div>
                </div>
            </div>
            
            <div class="report-columns">
                <div class="report-column">
                    <h3>Distribución por Estado</h3>
                    <div class="chart-container">
                        <canvas id="estadosSeriesChart"></canvas>
                    </div>
                </div>
                <div class="report-column">
                    <h3>Series con Más Solicitudes de Repuestos</h3>
                    <div class="chart-container">
                        <canvas id="seriesSolicitudesChart"></canvas>
                    </div>
                </div>
            </div>
            
            <h3>Detalle de Series por Estado</h3>
            <div class="table-responsive">
                <table class="table-reportes">
                    <thead>
                        <tr>
                            <th>Estado</th>
                            <th>Total Series</th>
                            <th>Total Máquinas</th>
                            <th>% del Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $total_series = 0;
                        foreach ($resultados['estados_series'] as $estado) {
                            $total_series += $estado['total_series'];
                        }
                        
                        foreach ($resultados['estados_series'] as $estado): 
                            $porcentaje = ($total_series > 0) ? 
                                round(($estado['total_series'] / $total_series) * 100, 1) : 0;
                        ?>
                        <tr>
                            <td>
                                <span class="status-badge" style="background-color: #e3f2fd; color: #1e88e5;">
                                    <?php echo ucfirst($estado['estado']); ?>
                                </span>
                            </td>
                            <td><?php echo $estado['total_series']; ?></td>
                            <td><?php echo $estado['total_maquinas']; ?></td>
                            <td><?php echo $porcentaje; ?>%</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr>
                            <th>Total</th>
                            <th><?php echo $total_series; ?></th>
                            <th>-</th>
                            <th>100%</th>
                        </tr>
                    </tfoot>
                </table>
            </div>
            
            <div class="tabs-container">
                <div class="tab-buttons">
                    <button type="button" class="tab-button active" data-tab="series-tab">Todas las Series</button>
                    <button type="button" class="tab-button" data-tab="comodato-tab">Series en Comodato</button>
                    <button type="button" class="tab-button" data-tab="solicitudes-tab">Solicitudes de Repuestos</button>
                </div>
                
                <div id="series-tab" class="tab-content active">
                    <h3>Listado de Series de Máquinas</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código Serie</th>
                                    <th>Máquina</th>
                                    <th>Estado</th>
                                    <th>Fecha Inicio</th>
                                    <th>Última Actualización</th>
                                    <th>Cliente/Proveedor</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['series_maquinas'] as $serie): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($serie['codigo_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($serie['maquina_nombre'] . ' (' . $serie['maquina_codigo'] . ')'); ?></td>
                                    <td>
                                        <span class="status-badge" style="background-color: #e3f2fd; color: #1e88e5;">
                                            <?php echo ucfirst($serie['estado']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo formatoFecha($serie['fecha_inicio_serie']); ?></td>
                                    <td><?php echo formatoFecha($serie['fecha_cambio_estado']); ?></td>
                                    <td>
                                        <?php
                                        if ($serie['estado'] === 'comodato' && !empty($serie['cliente_nombre'])) {
                                            echo 'Cliente: ' . htmlspecialchars($serie['cliente_nombre']);
                                        } elseif (!empty($serie['proveedor_nombre'])) {
                                            echo 'Proveedor: ' . htmlspecialchars($serie['proveedor_nombre']);
                                        } else {
                                            echo '-';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="comodato-tab" class="tab-content">
                    <h3>Series en Comodato con Clientes</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código Serie</th>
                                    <th>Máquina</th>
                                    <th>Cliente</th>
                                    <th>RUT Cliente</th>
                                    <th>Fecha Inicio Comodato</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $series_comodato_filtradas = array_filter($resultados['series_maquinas'], function($serie) {
                                    return $serie['estado'] === 'comodato' && !empty($serie['cliente_nombre']);
                                });
                                
                                foreach ($series_comodato_filtradas as $serie): 
                                ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($serie['codigo_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($serie['maquina_nombre'] . ' (' . $serie['maquina_codigo'] . ')'); ?></td>
                                    <td><?php echo htmlspecialchars($serie['cliente_nombre']); ?></td>
                                    <td><?php echo htmlspecialchars($serie['cliente_rut'] ?? '-'); ?></td>
                                    <td><?php echo formatoFecha($serie['fecha_cambio_estado']); ?></td>
                                </tr>
                                <?php endforeach; ?>
                                
                                <?php if (empty($series_comodato_filtradas)): ?>
                                <tr>
                                    <td colspan="5" style="text-align: center;">No hay series en comodato según los filtros seleccionados</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="solicitudes-tab" class="tab-content">
                    <h3>Series con Solicitudes de Repuestos</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código Serie</th>
                                    <th>Máquina</th>
                                    <th>Estado</th>
                                    <th>Cliente</th>
                                    <th>Solicitudes</th>
                                    <th>Repuestos Solicitados</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['solicitudes_por_serie'] as $solicitud): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($solicitud['codigo_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($solicitud['maquina_nombre']); ?></td>
                                    <td>
                                        <span class="status-badge" style="background-color: #e3f2fd; color: #1e88e5;">
                                            <?php echo ucfirst($solicitud['estado']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($solicitud['proveedor_nombre'] ?? '-'); ?></td>
                                    <td><?php echo $solicitud['total_solicitudes']; ?></td>
                                    <td><?php echo $solicitud['cantidad_total_repuestos'] ?? 0; ?></td>
                                </tr>
                                <?php endforeach; ?>
                                
                                <?php if (empty($resultados['solicitudes_por_serie'])): ?>
                                <tr>
                                    <td colspan="6" style="text-align: center;">No hay solicitudes de repuestos según los filtros seleccionados</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
        <?php elseif ($tipo_reporte === 'mantenimiento_repuestos'): ?>
            <!-- Reporte de Mantenimiento de Máquinas y Repuestos (MEJORADO) -->
            <div class="stat-cards">
                <div class="stat-card series">
                    <h3>Máquinas</h3>
                    <div class="stat-value"><?php echo count($resultados['maquinas']); ?></div>
                    <div class="stat-label">Con solicitudes de repuestos</div>
                </div>
                
                <div class="stat-card rendiciones">
                    <h3>Series</h3>
                    <div class="stat-value">
                        <?php 
                        $total_series = 0;
                        foreach ($resultados['maquinas'] as $maquina) {
                            $total_series += $maquina['total_series'];
                        }
                        echo $total_series;
                        ?>
                    </div>
                    <div class="stat-label">Con solicitudes de repuestos</div>
                </div>
                
                <div class="stat-card ordenes">
                    <h3>Solicitudes</h3>
                    <div class="stat-value">
                        <?php 
                        $total_solicitudes = 0;
                        foreach ($resultados['maquinas'] as $maquina) {
                            $total_solicitudes += $maquina['total_solicitudes'];
                        }
                        echo $total_solicitudes;
                        ?>
                    </div>
                    <div class="stat-label">Total en el período</div>
                </div>
                
                <div class="stat-card solicitudes">
                    <h3>Repuestos</h3>
                    <div class="stat-value">
                        <?php 
                        $total_repuestos = 0;
                        foreach ($resultados['repuestos'] as $repuesto) {
                            $total_repuestos += $repuesto['total_solicitado'];
                        }
                        echo $total_repuestos;
                        ?>
                    </div>
                    <div class="stat-label">Unidades solicitadas</div>
                </div>
            </div>
            
            <div class="tabs-container">
                <div class="tab-buttons">
                    <button type="button" class="tab-button active" data-tab="maquinas-tab">Por Máquina</button>
                    <button type="button" class="tab-button" data-tab="series-tab">Por Serie</button>
                    <button type="button" class="tab-button" data-tab="repuestos-tab">Por Repuesto</button>
                    <button type="button" class="tab-button" data-tab="motivos-tab">Por Motivo</button>
                </div>
                
                <div id="maquinas-tab" class="tab-content active">
                    <div class="chart-container">
                        <canvas id="maquinasChart"></canvas>
                    </div>
                    
                    <h3>Detalle por Máquina</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código</th>
                                    <th>Máquina</th>
                                    <th>Solicitudes</th>
                                    <th>Series</th>
                                    <th>Usuarios</th>
                                    <th>% Con Series</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $total_solicitudes = 0;
                                foreach ($resultados['maquinas'] as $maquina) {
                                    $total_solicitudes += $maquina['total_solicitudes'];
                                }
                                
                                foreach ($resultados['maquinas'] as $maquina): 
                                    $porcentaje_series = ($maquina['total_solicitudes'] > 0) ? 
                                        round(($maquina['solicitudes_con_serie'] / $maquina['total_solicitudes']) * 100, 1) : 0;
                                ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($maquina['codigo']); ?></td>
                                    <td><?php echo htmlspecialchars($maquina['maquina_nombre']); ?></td>
                                    <td><?php echo $maquina['total_solicitudes']; ?></td>
                                    <td><?php echo $maquina['total_series']; ?></td>
                                    <td><?php echo $maquina['total_usuarios']; ?></td>
                                    <td><?php echo $porcentaje_series; ?>%</td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="2">Total</th>
                                    <th><?php echo $total_solicitudes; ?></th>
                                    <th><?php echo $total_series; ?></th>
                                    <th>-</th>
                                    <th>-</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
                
                <div id="series-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="seriesChart"></canvas>
                    </div>
                    
                    <h3>Detalle por Serie</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código Serie</th>
                                    <th>Máquina</th>
                                    <th>Estado</th>
                                    <th>Cliente</th>
                                    <th>Solicitudes</th>
                                    <th>Repuestos Distintos</th>
                                    <th>Total Repuestos</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['series'] as $serie): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($serie['codigo_serie']); ?></td>
                                    <td><?php echo htmlspecialchars($serie['maquina_nombre']); ?></td>
                                    <td>
                                        <span class="status-badge" style="background-color: #e3f2fd; color: #1e88e5;">
                                            <?php echo ucfirst($serie['estado']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($serie['proveedor_nombre'] ?? '-'); ?></td>
                                    <td><?php echo $serie['total_solicitudes']; ?></td>
                                    <td><?php echo $serie['total_repuestos_distintos']; ?></td>
                                    <td><?php echo $serie['cantidad_total_repuestos']; ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="repuestos-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="repuestosChart"></canvas>
                    </div>
                    
                    <h3>Detalle de Repuestos Más Solicitados</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Código</th>
                                    <th>Repuesto</th>
                                    <th>Cantidad Solicitada</th>
                                    <th>Solicitudes</th>
                                    <th>Series Diferentes</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['repuestos'] as $repuesto): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($repuesto['codigo']); ?></td>
                                    <td><?php echo htmlspecialchars($repuesto['repuesto_nombre']); ?></td>
                                    <td><?php echo $repuesto['total_solicitado']; ?></td>
                                    <td><?php echo $repuesto['total_solicitudes']; ?></td>
                                    <td><?php echo $repuesto['total_series']; ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="motivos-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="motivosChart"></canvas>
                    </div>
                    
                    <h3>Detalle por Motivo de Solicitud</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Motivo</th>
                                    <th>Total Solicitudes</th>
                                    <th>Máquinas</th>
                                    <th>Series</th>
                                    <th>% del Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $total_motivos = 0;
                                foreach ($resultados['motivos'] as $motivo) {
                                    $total_motivos += $motivo['total'];
                                }
                                
                                foreach ($resultados['motivos'] as $motivo): 
                                    // Convertir el valor de la enum a un formato más legible
                                    $motivo_legible = str_replace('_', ' ', $motivo['motivo_solicitud']);
                                    $motivo_legible = ucwords($motivo_legible);
                                    
                                    $porcentaje = ($total_motivos > 0) ? 
                                        round(($motivo['total'] / $total_motivos) * 100, 1) : 0;
                                ?>
                                <tr>
                                    <td><?php echo $motivo_legible; ?></td>
                                    <td><?php echo $motivo['total']; ?></td>
                                    <td><?php echo $motivo['total_maquinas']; ?></td>
                                    <td><?php echo $motivo['total_series']; ?></td>
                                    <td><?php echo $porcentaje; ?>%</td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th>Total</th>
                                    <th><?php echo $total_motivos; ?></th>
                                    <th>-</th>
                                    <th>-</th>
                                    <th>100%</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        
        <?php elseif ($tipo_reporte === 'analisis_proveedores'): ?>
            <!-- Reporte de Análisis de Proveedores (MEJORADO) -->
            <div class="stat-cards">
                <div class="stat-card">
                    <h3>Proveedores</h3>
                    <div class="stat-value"><?php echo count($resultados['proveedores']); ?></div>
                    <div class="stat-label">Total analizados</div>
                </div>
                
                <div class="stat-card ordenes">
                    <h3>Órdenes de Compra</h3>
                    <div class="stat-value">
                        <?php 
                        $total_ordenes = 0;
                        foreach ($resultados['proveedores'] as $proveedor) {
                            $total_ordenes += $proveedor['total_ordenes'];
                        }
                        echo $total_ordenes;
                        ?>
                    </div>
                    <div class="stat-label">En el período</div>
                </div>
                
                <div class="stat-card rendiciones">
                    <h3>Monto Total</h3>
                    <div class="stat-value">
                        <?php 
                        $total_monto = 0;
                        foreach ($resultados['proveedores'] as $proveedor) {
                            $total_monto += $proveedor['monto_total'] ?? 0;
                        }
                        echo formatoMoneda($total_monto);
                        ?>
                    </div>
                    <div class="stat-label">En compras</div>
                </div>
                
                <div class="stat-card series">
                    <h3>Máquinas en Comodato</h3>
                    <div class="stat-value">
                        <?php 
                        $total_maquinas_cliente = 0;
                        foreach ($resultados['proveedores'] as $proveedor) {
                            $total_maquinas_cliente += $proveedor['maquinas_cliente'] ?? 0;
                        }
                        echo $total_maquinas_cliente;
                        ?>
                    </div>
                    <div class="stat-label">Con clientes</div>
                </div>
            </div>
            
            <div class="tabs-container">
                <div class="tab-buttons">
                    <button type="button" class="tab-button active" data-tab="compras-tab">Compras</button>
                    <button type="button" class="tab-button" data-tab="series-tab">Máquinas</button>
                    <button type="button" class="tab-button" data-tab="tendencia-tab">Tendencia</button>
                    <button type="button" class="tab-button" data-tab="aprobacion-tab">Aprobación</button>
                </div>
                
                <div id="compras-tab" class="tab-content active">
                    <div class="chart-container">
                        <canvas id="proveedoresAnalisisChart"></canvas>
                    </div>
                    
                    <h3>Detalle de Compras por Proveedor</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Proveedor</th>
                                    <th>RUT</th>
                                    <th>Órdenes</th>
                                    <th>Monto Total</th>
                                    <th>Promedio</th>
                                    <th>% del Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                foreach ($resultados['proveedores'] as $proveedor): 
                                    $porcentaje = ($total_monto > 0) ? 
                                        round((($proveedor['monto_total'] ?? 0) / $total_monto) * 100, 1) : 0;
                                ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($proveedor['nombre']); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['rut']); ?></td>
                                    <td><?php echo $proveedor['total_ordenes']; ?></td>
                                    <td><?php echo formatoMoneda($proveedor['monto_total'] ?? 0); ?></td>
                                    <td><?php echo formatoMoneda($proveedor['promedio_orden'] ?? 0); ?></td>
                                    <td><?php echo $porcentaje; ?>%</td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="2">Total</th>
                                    <th><?php echo $total_ordenes; ?></th>
                                    <th><?php echo formatoMoneda($total_monto); ?></th>
                                    <th>-</th>
                                    <th>100%</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
                
                <div id="series-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="seriesProveedorChart"></canvas>
                    </div>
                    
                    <h3>Máquinas por Proveedor y Estado</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Proveedor</th>
                                    <th>Estado</th>
                                    <th>Series</th>
                                    <th>Máquinas</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                // Agrupar por proveedor y luego por estado
                                $series_agrupadas = [];
                                foreach ($resultados['series_por_proveedor'] as $serie) {
                                    if (!isset($series_agrupadas[$serie['proveedor_id']])) {
                                        $series_agrupadas[$serie['proveedor_id']] = [
                                            'nombre' => $serie['proveedor_nombre'],
                                            'estados' => []
                                        ];
                                    }
                                    
                                    if (!empty($serie['estado'])) {
                                        if (!isset($series_agrupadas[$serie['proveedor_id']]['estados'][$serie['estado']])) {
                                            $series_agrupadas[$serie['proveedor_id']]['estados'][$serie['estado']] = [
                                                'series' => 0,
                                                'maquinas' => 0
                                            ];
                                        }
                                        
                                        $series_agrupadas[$serie['proveedor_id']]['estados'][$serie['estado']]['series'] += $serie['total_series'];
                                        $series_agrupadas[$serie['proveedor_id']]['estados'][$serie['estado']]['maquinas'] += $serie['total_maquinas'];
                                    }
                                }
                                
                                foreach ($series_agrupadas as $proveedor_id => $proveedor): 
                                    if (empty($proveedor['estados'])) {
                                        continue;
                                    }
                                    
                                    $first_row = true;
                                    foreach ($proveedor['estados'] as $estado => $datos):
                                ?>
                                <tr>
                                    <?php if ($first_row): ?>
                                    <td rowspan="<?php echo count($proveedor['estados']); ?>"><?php echo htmlspecialchars($proveedor['nombre']); ?></td>
                                    <?php $first_row = false; endif; ?>
                                    <td>
                                        <span class="status-badge" style="background-color: #e3f2fd; color: #1e88e5;">
                                            <?php echo ucfirst($estado); ?>
                                        </span>
                                    </td>
                                    <td><?php echo $datos['series']; ?></td>
                                    <td><?php echo $datos['maquinas']; ?></td>
                                </tr>
                                <?php 
                                    endforeach;
                                endforeach; 
                                ?>
                                
                                <?php if (empty($series_agrupadas)): ?>
                                <tr>
                                    <td colspan="4" style="text-align: center;">No hay series asociadas a proveedores según los filtros seleccionados</td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="tendencia-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="tendenciaChart"></canvas>
                    </div>
                    
                    <p class="mt-3 text-muted">
                        El gráfico muestra la tendencia de compras por proveedor en los últimos 6 meses.
                        Solo se muestran los 5 proveedores con mayor volumen de compras.
                    </p>
                </div>
                
                <div id="aprobacion-tab" class="tab-content">
                    <div class="chart-container">
                        <canvas id="aprobacionChart"></canvas>
                    </div>
                    
                    <h3>Tasa de Aprobación por Proveedor</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Proveedor</th>
                                    <th>Órdenes Aprobadas</th>
                                    <th>Total Órdenes</th>
                                    <th>Tasa de Aprobación</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['aprobacion'] as $aprobacion): ?>
                                <?php if ($aprobacion['total_ordenes'] >= 3): // Solo mostrar proveedores con al menos 3 órdenes ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($aprobacion['nombre']); ?></td>
                                    <td><?php echo $aprobacion['ordenes_aprobadas']; ?></td>
                                    <td><?php echo $aprobacion['total_ordenes']; ?></td>
                                    <td><?php echo $aprobacion['tasa_aprobacion']; ?>%</td>
                                </tr>
                                <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
        <?php elseif ($tipo_reporte === 'rendiciones_categoria'): ?>
            <!-- Código para el reporte de rendiciones por categoría -->
            <div class="report-columns">
                <div class="report-column">
                    <div class="chart-container">
                        <canvas id="categoriasChart"></canvas>
                    </div>
                </div>
                <div class="report-column">
                    <h3>Top 10 Gastos</h3>
                    <div class="table-responsive">
                        <table class="table-reportes">
                            <thead>
                                <tr>
                                    <th>Categoría</th>
                                    <th>Descripción</th>
                                    <th>Monto</th>
                                    <th>Usuario</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($resultados['top_gastos'] as $gasto): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($gasto['categoria']); ?></td>
                                    <td><?php echo htmlspecialchars(substr($gasto['descripcion'], 0, 50) . (strlen($gasto['descripcion']) > 50 ? '...' : '')); ?></td>
                                    <td><?php echo formatoMoneda($gasto['monto']); ?></td>
                                    <td><?php echo htmlspecialchars($gasto['nombre'] . ' ' . $gasto['apellido']); ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <h3>Detalle por Categoría</h3>
            <div class="table-responsive">
                <table class="table-reportes">
                    <thead>
                        <tr>
                            <th>Categoría</th>
                            <th>Documentos</th>
                            <th>Monto Total</th>
                            <th>Promedio</th>
                            <th>% del Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $total_monto = 0;
                        foreach ($resultados['categorias'] as $categoria) {
                            $total_monto += $categoria['monto_total'] ?? 0;
                        }
                        ?>
                        
                        <?php foreach ($resultados['categorias'] as $categoria): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($categoria['nombre']); ?></td>
                            <td><?php echo $categoria['total_documentos']; ?></td>
                            <td><?php echo formatoMoneda($categoria['monto_total'] ?? 0); ?></td>
                            <td><?php echo formatoMoneda($categoria['promedio_gasto'] ?? 0); ?></td>
                            <td>
                                <?php 
                                $porcentaje = ($total_monto > 0) ? 
                                    round(($categoria['monto_total'] / $total_monto) * 100, 1) : 0;
                                echo $porcentaje . '%';
                                ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr>
                            <th>Total</th>
                            <th>
                                <?php 
                                $total_docs = 0;
                                foreach ($resultados['categorias'] as $categoria) {
                                    $total_docs += $categoria['total_documentos'] ?? 0;
                                }
                                echo $total_docs;
                                ?>
                            </th>
                            <th><?php echo formatoMoneda($total_monto); ?></th>
                            <th>-</th>
                            <th>100%</th>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php endif; ?>
        <!-- Aquí continuaría el código para los demás tipos de reportes -->
    </div>
    <?php else: ?>
    <div class="alert alert-info">
        Seleccione los filtros y haga clic en "Generar Reporte" para ver los resultados.
    </div>
    <?php endif; ?>
</div>

<script>
    // Función para formatear moneda en los gráficos
    function formatoMoneda(valor) {
        return new Intl.NumberFormat('es-CL', {
            style: 'currency',
            currency: 'CLP'
        }).format(valor);
    }

    // Función para manejar datos vacíos
    function validarDatosGrafico(datos) {
        if (!datos || !datos.labels || !datos.datasets) {
            return {
                labels: ['Sin datos'],
                datasets: [{
                    data: [0],
                    backgroundColor: ['#e0e0e0']
                }]
            };
        }
        return datos;
    }

    document.addEventListener('DOMContentLoaded', function() {
        // Manejo de filtros
        const periodoSelect = document.getElementById('periodo');
        const mesContainer = document.getElementById('mes-container');
        const fechaContainer = document.querySelectorAll('.fecha-container');

        if (periodoSelect) {
            periodoSelect.addEventListener('change', function() {
                if (this.value === 'mes' || this.value === 'trimestre') {
                    mesContainer.style.display = 'block';
                    fechaContainer.forEach(el => el.style.display = 'none');
                } else if (this.value === 'personalizado') {
                    mesContainer.style.display = 'none';
                    fechaContainer.forEach(el => el.style.display = 'block');
                } else {
                    mesContainer.style.display = 'none';
                    fechaContainer.forEach(el => el.style.display = 'none');
                }
            });
        }
        
        // Manejo de tabs
        const tabButtons = document.querySelectorAll('.tab-button');
        tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const tabId = this.getAttribute('data-tab');
                
                // Desactivar todos los tabs y botones
                document.querySelectorAll('.tab-content').forEach(tab => {
                    tab.classList.remove('active');
                });
                document.querySelectorAll('.tab-button').forEach(btn => {
                    btn.classList.remove('active');
                });
                
                // Activar el tab seleccionado
                document.getElementById(tabId).classList.add('active');
                this.classList.add('active');
            });
        });

        <?php if ($tipo_reporte === 'general' && !empty($datos_grafico)): ?>
            // Gráfico de estados
            const estadosCtx = document.getElementById('estadosChart').getContext('2d');
            const estadosData = validarDatosGrafico(<?php echo json_encode($datos_grafico); ?>);
            const estadosChart = new Chart(estadosCtx, {
                type: 'bar',
                data: estadosData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        },
                        title: {
                            display: true,
                            text: 'Distribución por Estados'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.dataset.label || '';
                                    const value = context.raw || 0;
                                    return `${label}: ${value} documentos`;
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
        <?php endif; ?>

        <?php if ($tipo_reporte === 'rendiciones_categoria' && !empty($datos_grafico)): ?>
            // Gráfico de categorías
            const categoriasCtx = document.getElementById('categoriasChart').getContext('2d');
            const categoriasData = validarDatosGrafico(<?php echo json_encode($datos_grafico); ?>);
            const categoriasChart = new Chart(categoriasCtx, {
                type: 'doughnut',
                data: categoriasData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right',
                            align: 'center'
                        },
                        title: {
                            display: true,
                            text: 'Distribución por Categorías'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                                    const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                    return `${label}: ${formatoMoneda(value)} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        <?php endif; ?>

        <?php if ($tipo_reporte === 'series_maquinas' && !empty($datos_grafico)): ?>
            // Gráfico de estados de series
            const estadosSeriesCtx = document.getElementById('estadosSeriesChart').getContext('2d');
            const estadosSeriesData = validarDatosGrafico(<?php echo json_encode($datos_grafico); ?>);
            const estadosSeriesChart = new Chart(estadosSeriesCtx, {
                type: 'pie',
                data: estadosSeriesData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right',
                            align: 'center'
                        },
                        title: {
                            display: true,
                            text: 'Series por Estado'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                                    const percentage = total > 0 ? Math.round((value / total) * 100) : 0;
                                    return `${label}: ${value} series (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
            
            // Gráfico de solicitudes por serie
            const seriesSolicitudesCtx = document.getElementById('seriesSolicitudesChart').getContext('2d');
            const seriesSolicitudesData = validarDatosGrafico(<?php echo json_encode($datos_grafico_series); ?>);
            const seriesSolicitudesChart = new Chart(seriesSolicitudesCtx, {
                type: 'bar',
                data: seriesSolicitudesData,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        title: {
                            display: true,
                            text: 'Solicitudes de Repuestos por Serie'
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
        <?php endif; ?>

        <?php if ($tipo_reporte === 'mantenimiento_repuestos'): ?>
            // Gráfico de máquinas
            const maquinasCtx = document.getElementById('maquinasChart')?.getContext('2d');
            if (maquinasCtx) {
                const maquinasData = validarDatosGrafico(<?php echo json_encode($datos_grafico); ?>);
                const maquinasChart = new Chart(maquinasCtx, {
                    type: 'bar',
                    data: maquinasData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            },
                            title: {
                                display: true,
                                text: 'Solicitudes por Máquina'
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de series
            const seriesCtx = document.getElementById('seriesChart')?.getContext('2d');
            if (seriesCtx) {
                const seriesData = validarDatosGrafico(<?php echo json_encode($datos_grafico_series); ?>);
                const seriesChart = new Chart(seriesCtx, {
                    type: 'bar',
                    data: seriesData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            },
                            title: {
                                display: true,
                                text: 'Solicitudes por Serie'
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de repuestos
            const repuestosCtx = document.getElementById('repuestosChart')?.getContext('2d');
            if (repuestosCtx) {
                const repuestosData = validarDatosGrafico(<?php echo json_encode($datos_grafico_repuestos); ?>);
                const repuestosChart = new Chart(repuestosCtx, {
                    type: 'bar',
                    data: repuestosData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            },
                            title: {
                                display: true,
                                text: 'Repuestos Más Solicitados'
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de motivos
            const motivosCtx = document.getElementById('motivosChart')?.getContext('2d');
            if (motivosCtx) {
                const motivosData = validarDatosGrafico(<?php echo json_encode($datos_grafico_motivos); ?>);
                const motivosChart = new Chart(motivosCtx, {
                    type: 'pie',
                    data: motivosData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'right',
                                align: 'center'
                            },
                            title: {
                                display: true,
                                text: 'Distribución por Motivo'
                            }
                        }
                    }
                });
            }
        <?php endif; ?>

        <?php if ($tipo_reporte === 'analisis_proveedores'): ?>
            // Gráfico de análisis de proveedores
            const proveedoresAnalisisCtx = document.getElementById('proveedoresAnalisisChart')?.getContext('2d');
            if (proveedoresAnalisisCtx) {
                const proveedoresAnalisisData = validarDatosGrafico(<?php echo json_encode($datos_grafico); ?>);
                const proveedoresAnalisisChart = new Chart(proveedoresAnalisisCtx, {
                    type: 'bar',
                    data: proveedoresAnalisisData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            title: {
                                display: true,
                                text: 'Órdenes y Montos por Proveedor'
                            },
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        const label = context.dataset.label || '';
                                        const value = context.raw || 0;
                                        if (label === 'Monto Total') {
                                            return `${label}: ${formatoMoneda(value)}`;
                                        }
                                        return `${label}: ${value}`;
                                    }
                                }
                            }
                        },
                        scales: {
                            y: {
                                position: 'left',
                                beginAtZero: true,
                                title: {
                                    display: true,
                                    text: 'Monto Total'
                                },
                                ticks: {
                                    callback: function(value) {
                                        return formatoMoneda(value);
                                    }
                                }
                            },
                            y1: {
                                position: 'right',
                                beginAtZero: true,
                                title: {
                                    display: true,
                                    text: 'Cantidad'
                                },
                                grid: {
                                    drawOnChartArea: false
                                },
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de series por proveedor
            const seriesProveedorCtx = document.getElementById('seriesProveedorChart')?.getContext('2d');
            if (seriesProveedorCtx) {
                const seriesProveedorData = validarDatosGrafico(<?php echo json_encode($datos_grafico_series); ?>);
                const seriesProveedorChart = new Chart(seriesProveedorCtx, {
                    type: 'bar',
                    data: seriesProveedorData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            title: {
                                display: true,
                                text: 'Máquinas por Proveedor y Estado'
                            }
                        },
                        scales: {
                            x: {
                                stacked: true
                            },
                            y: {
                                stacked: true,
                                beginAtZero: true,
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de tendencia
            const tendenciaCtx = document.getElementById('tendenciaChart')?.getContext('2d');
            if (tendenciaCtx) {
                const tendenciaData = validarDatosGrafico(<?php echo json_encode($datos_grafico_tendencia); ?>);
                const tendenciaChart = new Chart(tendenciaCtx, {
                    type: 'line',
                    data: tendenciaData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            title: {
                                display: true,
                                text: 'Tendencia de Compras (Últimos 6 Meses)'
                            },
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        const label = context.dataset.label || '';
                                        const value = context.raw || 0;
                                        return `${label}: ${formatoMoneda(value)}`;
                                    }
                                }
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    callback: function(value) {
                                        return formatoMoneda(value);
                                    }
                                }
                            }
                        }
                    }
                });
            }
            
            // Gráfico de tasa de aprobación
            const aprobacionCtx = document.getElementById('aprobacionChart')?.getContext('2d');
            if (aprobacionCtx) {
                const aprobacionData = validarDatosGrafico(<?php echo json_encode($datos_grafico_aprobacion); ?>);
                const aprobacionChart = new Chart(aprobacionCtx, {
                    type: 'bar',
                    data: aprobacionData,
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            },
                            title: {
                                display: true,
                                text: 'Tasa de Aprobación de Órdenes por Proveedor'
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                max: 100,
                                ticks: {
                                    callback: function(value) {
                                        return value + '%';
                                    }
                                }
                            }
                        }
                    }
                });
            }
        <?php endif; ?>

        // Función para mostrar indicador de carga durante la generación de reportes
        const filtrosForm = document.getElementById('filtros-form');
        if (filtrosForm) {
            filtrosForm.addEventListener('submit', function() {
                // Mostrar indicador de carga
                const loadingIndicator = document.createElement('div');
                loadingIndicator.className = 'loading-indicator';
                loadingIndicator.innerHTML = '<i class="material-icons spin">refresh</i> Generando reporte...';
                document.body.appendChild(loadingIndicator);
            });
        }
    });
</script>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>