<?php
/**
 * Gestión de órdenes de compra
 * CRUD completo para órdenes de compra
 */

$page_title = 'Gestión de Órdenes de Compra';
require_once 'includes/config.php';
require_once 'includes/functions.php';

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $estados = obtenerEstados($conn);
    $estados_map = [];
    foreach ($estados as $estado) {
        $estados_map[$estado['id']] = ['nombre' => $estado['nombre'], 'color' => $estado['color']];
    }
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        $proveedor = trim($_POST['proveedor'] ?? '');
        $rut_proveedor = trim($_POST['rut_proveedor'] ?? '');
        $proveedor_id = !empty($_POST['proveedor_id']) ? intval($_POST['proveedor_id']) : null;
        $fecha = $_POST['fecha'] ?? '';
        $descripcion = trim($_POST['descripcion'] ?? '');
        $numero_documento = trim($_POST['numero_documento'] ?? '');
        $condiciones_pago = trim($_POST['condiciones_pago'] ?? '');
        $direccion_despacho = trim($_POST['direccion_despacho'] ?? '');
        $motivo_orden = trim($_POST['motivo_orden'] ?? '');
        $area_solicitud = trim($_POST['area_solicitud'] ?? '');
        $tiempo_entrega = trim($_POST['tiempo_entrega'] ?? '');
        
        $codigos_producto = $_POST['codigo_producto'] ?? [];
        $productos = $_POST['producto'] ?? [];
        $cantidades = $_POST['cantidad'] ?? [];
        $precios = $_POST['precio_unitario'] ?? [];
        $descuentos = $_POST['descuento'] ?? [];
        $precios_finales = $_POST['precio_final'] ?? [];
        $subtotales = $_POST['subtotal'] ?? [];
        
        $errores = [];
        if (empty($proveedor)) $errores[] = "El nombre del proveedor es obligatorio";
        if (empty($rut_proveedor)) $errores[] = "El RUT del proveedor es obligatorio";
        elseif (!validarRut($rut_proveedor)) $errores[] = "El RUT del proveedor no es válido";
        if (empty($fecha)) $errores[] = "La fecha es obligatoria";
        if (empty($numero_documento)) $errores[] = "El número de documento es obligatorio";
        if (empty($area_solicitud)) $errores[] = "El área de solicitud es obligatoria";
        if (empty($productos) || count($productos) < 1) $errores[] = "Debe agregar al menos un producto o servicio";
        
        $monto_total = 0;
        $total_descuentos = 0;
        $iva = 0.19; // IVA del 19%
        
        foreach ($productos as $index => $producto) {
            $producto = trim($producto);
            if (empty($producto)) $errores[] = "La descripción del producto en la línea " . ($index + 1) . " es obligatoria";
            $cantidad = isset($cantidades[$index]) ? intval($cantidades[$index]) : 0;
            $precio = isset($precios[$index]) ? floatval($precios[$index]) : 0;
            $descuento = isset($descuentos[$index]) ? floatval($descuentos[$index]) : 0;
            $precio_final = $precio * (1 - ($descuento / 100));
            $subtotal = $cantidad * $precio_final;
            
            // Calcular el IVA directamente sobre el subtotal
            $subtotal_bruto = $subtotal * (1 + $iva);
            
            $monto_total += $subtotal_bruto;
            $total_descuentos += ($precio - $precio_final) * $cantidad;
        }
        
        if (empty($errores)) {
            $conn->beginTransaction();
            try {
                if ($action === 'edit' && $id > 0) {
                    $stmt = $conn->prepare("SELECT * FROM ordenes_compra WHERE id = :id");
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    $orden = $stmt->fetch(PDO::FETCH_ASSOC);
                    if (!$orden) throw new Exception("La orden de compra no existe");
                    if ($orden['user_id'] != $user_id && !$is_admin) throw new Exception("No tiene permisos para editar esta orden de compra");
                    if ($orden['estado_id'] != 1 && $orden['estado_id'] != 4) throw new Exception("No se puede editar una orden que ya ha sido aprobada o está en revisión");
                    if (!$is_admin && !estaEnTiempoEdicion($orden['fecha_creacion'])) throw new Exception("Solo puede editar órdenes dentro de las 24 horas");
                    
                    $stmt = $conn->prepare("
                        UPDATE ordenes_compra 
                        SET proveedor = :proveedor, rut_proveedor = :rut_proveedor, proveedor_id = :proveedor_id,
                            fecha = :fecha, descripcion = :descripcion, numero_documento = :numero_documento,
                            condiciones_pago = :condiciones_pago, monto_total = :monto_total, 
                            direccion_despacho = :direccion_despacho, motivo_orden = :motivo_orden,
                            area_solicitud = :area_solicitud, tiempo_entrega = :tiempo_entrega,
                            ultima_modificacion = NOW()
                        WHERE id = :id
                    ");
                    $stmt->bindParam(':proveedor', $proveedor);
                    $stmt->bindParam(':rut_proveedor', $rut_proveedor);
                    $stmt->bindParam(':proveedor_id', $proveedor_id, PDO::PARAM_INT);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':numero_documento', $numero_documento);
                    $stmt->bindParam(':condiciones_pago', $condiciones_pago);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':direccion_despacho', $direccion_despacho);
                    $stmt->bindParam(':motivo_orden', $motivo_orden);
                    $stmt->bindParam(':area_solicitud', $area_solicitud);
                    $stmt->bindParam(':tiempo_entrega', $tiempo_entrega);
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    
                    // Primero eliminar los detalles existentes
                    $stmt = $conn->prepare("DELETE FROM detalle_orden_compra WHERE orden_compra_id = :id");
                    $stmt->bindParam(':id', $id);
                    $stmt->execute();
                    
                    registrarActividad($conn, $user_id, 'editar', 'Edición de orden de compra', $id, 'orden_compra');
                    $mensaje = "Orden de compra actualizada correctamente";
                } else {
                    $stmt = $conn->prepare("
                        INSERT INTO ordenes_compra 
                        (user_id, proveedor, rut_proveedor, proveedor_id, fecha, descripcion, numero_documento, 
                         condiciones_pago, monto_total, estado_id, fecha_creacion, ultima_modificacion,
                         direccion_despacho, motivo_orden, area_solicitud, tiempo_entrega) 
                        VALUES (:user_id, :proveedor, :rut_proveedor, :proveedor_id, :fecha, :descripcion, :numero_documento, 
                                :condiciones_pago, :monto_total, 1, NOW(), NOW(), 
                                :direccion_despacho, :motivo_orden, :area_solicitud, :tiempo_entrega)
                    ");
                    $stmt->bindParam(':user_id', $user_id);
                    $stmt->bindParam(':proveedor', $proveedor);
                    $stmt->bindParam(':rut_proveedor', $rut_proveedor);
                    $stmt->bindParam(':proveedor_id', $proveedor_id, PDO::PARAM_INT);
                    $stmt->bindParam(':fecha', $fecha);
                    $stmt->bindParam(':descripcion', $descripcion);
                    $stmt->bindParam(':numero_documento', $numero_documento);
                    $stmt->bindParam(':condiciones_pago', $condiciones_pago);
                    $stmt->bindParam(':monto_total', $monto_total);
                    $stmt->bindParam(':direccion_despacho', $direccion_despacho);
                    $stmt->bindParam(':motivo_orden', $motivo_orden);
                    $stmt->bindParam(':area_solicitud', $area_solicitud);
                    $stmt->bindParam(':tiempo_entrega', $tiempo_entrega);
                    $stmt->execute();
                    
                    $id = $conn->lastInsertId();
                    registrarActividad($conn, $user_id, 'crear', 'Creación de orden de compra', $id, 'orden_compra');
                    $mensaje = "Orden de compra creada correctamente";
                }
                
                // Añadir los detalles de la orden
                foreach ($productos as $index => $producto) {
                    if (empty($producto)) continue;
                    
                    $codigo_producto = isset($codigos_producto[$index]) ? trim($codigos_producto[$index]) : '';
                    $cantidad = intval($cantidades[$index]);
                    $precio = floatval($precios[$index]);
                    $descuento = floatval($descuentos[$index] ?? 0);
                    $precio_final = $precio * (1 - ($descuento / 100));
                    $subtotal = $cantidad * $precio_final;
                    
                    $stmt = $conn->prepare("
                        INSERT INTO detalle_orden_compra 
                        (orden_compra_id, producto, codigo_producto, cantidad, precio_unitario, 
                         descuento, precio_final, subtotal) 
                        VALUES (:orden_compra_id, :producto, :codigo_producto, :cantidad, :precio_unitario, 
                                :descuento, :precio_final, :subtotal)
                    ");
                    
                    $stmt->bindParam(':orden_compra_id', $id);
                    $stmt->bindParam(':producto', $producto);
                    $stmt->bindParam(':codigo_producto', $codigo_producto);
                    $stmt->bindParam(':cantidad', $cantidad);
                    $stmt->bindParam(':precio_unitario', $precio);
                    $stmt->bindParam(':descuento', $descuento);
                    $stmt->bindParam(':precio_final', $precio_final);
                    $stmt->bindParam(':subtotal', $subtotal);
                    $stmt->execute();
                }
                
                $conn->commit();
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: ordenes_compra.php');
                exit;
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = "Error al guardar: " . $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Resto del código PHP (review, delete, etc.) permanece igual
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'review' && $is_admin) {
        $decision = $_POST['decision'] ?? '';
        $comentario = trim($_POST['comentario'] ?? '');
        
        if ($decision != 'aprobar' && $decision != 'rechazar') {
            $error_message = "Acción no válida";
        } else {
            try {
                $estado_id = ($decision === 'aprobar') ? 3 : 4;
                $stmt = $conn->prepare("
                    UPDATE ordenes_compra 
                    SET estado_id = :estado_id, revisor_id = :revisor_id, comentario_revision = :comentario,
                        fecha_revision = NOW()
                    WHERE id = :id
                ");
                $stmt->bindParam(':estado_id', $estado_id);
                $stmt->bindParam(':revisor_id', $user_id);
                $stmt->bindParam(':comentario', $comentario);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                $tipo_accion = ($decision === 'aprobar') ? 'aprobar' : 'rechazar';
                $descripcion = ($decision === 'aprobar') ? 'Aprobación de orden de compra' : 'Rechazo de orden de compra';
                registrarActividad($conn, $user_id, $tipo_accion, $descripcion, $id, 'orden_compra');
                
                $mensaje = ($decision === 'aprobar') ? "Orden de compra aprobada correctamente" : "Orden de compra rechazada correctamente";
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: ordenes_compra.php');
                exit;
            } catch (Exception $e) {
                $error_message = "Error al procesar la solicitud: " . $e->getMessage();
            }
        }
    }
    
    if ($action === 'delete' && $is_admin && $id > 0) {
        try {
            $stmt = $conn->prepare("SELECT * FROM ordenes_compra WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $orden = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$orden) throw new Exception("La orden de compra no existe");
            if ($orden['estado_id'] == 2 || $orden['estado_id'] == 3 || $orden['estado_id'] == 5) throw new Exception("No se puede eliminar una orden en revisión, aprobada o pagada");
            
            $stmt = $conn->prepare("DELETE FROM detalle_orden_compra WHERE orden_compra_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            $stmt = $conn->prepare("DELETE FROM ordenes_compra WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de orden de compra', $id, 'orden_compra');
            $_SESSION['flash_message'] = "Orden de compra eliminada correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: ordenes_compra.php');
            exit;
        } catch (Exception $e) {
            $error_message = "Error al eliminar: " . $e->getMessage();
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

require_once 'includes/header.php';
?>

<style>
    .ordenes-table td, .ordenes-table th { padding: 8px 10px; }
    .ordenes-table tr:hover { background-color: rgba(0,0,0,0.03); }
    .action-icon {
        display: inline-flex; align-items: center; justify-content: center; width: 28px; height: 28px;
        border-radius: 50%; margin-right: 5px; transition: all 0.2s; color: #555 !important; background-color: #f0f0f0;
    }
    .action-icon:hover { background-color: #e0e0e0; transform: scale(1.1); }
    .action-icon.view-icon:hover { color: #1e88e5 !important; }
    .action-icon.pdf-icon:hover { color: #66bb6a !important; }
    .action-icon.edit-icon:hover { color: #ffa726 !important; }
    .action-icon.delete-icon:hover { color: #ef5350 !important; }
    .action-icon i { font-size: 18px; }
    .form-control:focus { border-color: #1e88e5; box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25); }
    .form-control:required { border-left: 3px solid #1e88e5; }
    .input-sm { padding: 8px 10px; height: auto; }
    .proveedor-info {
        background-color: #f9f9f9;
        border: 1px solid #ddd;
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }
    .proveedor-info p {
        margin-bottom: 5px;
    }
    .proveedor-info strong {
        display: inline-block;
        width: 100px;
    }
    tfoot tr {
        background-color: #f5f5f5;
        font-weight: bold;
    }
    .table-totals th, .table-totals td {
        font-weight: bold;
        text-align: right;
    }
    .totales-section {
        margin-top: 20px;
        padding: 15px;
        background-color: #f8f9fa;
        border-radius: 4px;
        border: 1px solid #dee2e6;
    }

    .total-row {
        display: flex;
        justify-content: flex-end;
        align-items: center;
        margin-bottom: 8px;
        font-size: 14px;
    }

    .total-row.total-final {
        margin-top: 10px;
        padding-top: 10px;
        border-top: 1px solid #dee2e6;
        font-weight: bold;
        font-size: 16px;
    }

    .total-label {
        margin-right: 15px;
        color: #495057;
    }

    .total-value {
        min-width: 120px;
        text-align: right;
        font-family: monospace;
    }

    .total-final .total-value {
        color: #28a745;
        font-size: 18px;
    }
</style>

<?php if (isset($error_message)): ?>
<div class="alert alert-danger"><?php echo $error_message; ?></div>
<?php endif; ?>

<?php
switch ($action) {
    case 'new':
        ?>
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title">Nueva Orden de Compra</h2>
                <a href="ordenes_compra.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="ordenes_compra.php?action=new" id="form-orden">
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="rut_proveedor" class="form-label">RUT del Proveedor</label>
            <div class="input-group">
                <input type="text" id="rut_proveedor" name="rut_proveedor" class="form-control input-sm" 
                       placeholder="Ej: 12.345.678-5" required autocomplete="off" 
                       value="<?php echo isset($orden) ? htmlspecialchars($orden['rut_proveedor']) : ''; ?>">
            </div>
            <div id="mensaje-proveedor" class="form-text"></div>
        </div>
        <div class="form-group">
            <label for="proveedor" class="form-label">Nombre del Proveedor</label>
            <input type="text" id="proveedor" name="proveedor" class="form-control input-sm" 
                   placeholder="Nombre de la empresa proveedora" required autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['proveedor']) : ''; ?>">
            <input type="hidden" id="proveedor_id" name="proveedor_id"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['proveedor_id'] ?? '') : ''; ?>">
        </div>
        <div class="form-group">
            <label for="fecha" class="form-label">Fecha</label>
            <input type="date" id="fecha" name="fecha" class="form-control input-sm" 
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['fecha']) : date('Y-m-d'); ?>" required>
        </div>
        <div class="form-group">
            <label for="area_solicitud" class="form-label">Área de Solicitud</label>
            <select id="area_solicitud" name="area_solicitud" class="form-control input-sm" required>
                <option value="">Seleccione área...</option>
                <option value="bodega" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'bodega') ? 'selected' : ''; ?>>Bodega</option>
                <option value="servicio_tecnico" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'servicio_tecnico') ? 'selected' : ''; ?>>Servicio Técnico</option>
                <option value="tiendas" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'tiendas') ? 'selected' : ''; ?>>Tiendas</option>
                <option value="otros" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'otros') ? 'selected' : ''; ?>>Otros</option>
            </select>
        </div>
    </div>
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="numero_documento" class="form-label">N° Documento</label>
            <input type="text" id="numero_documento" name="numero_documento" class="form-control input-sm" 
                   placeholder="Número de documento" required autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['numero_documento']) : ''; ?>">
        </div>
        <div class="form-group">
            <label for="condiciones_pago" class="form-label">Condiciones de Pago</label>
            <input type="text" id="condiciones_pago" name="condiciones_pago" class="form-control input-sm" 
                   placeholder="Ej: 30 días" autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['condiciones_pago'] ?? '') : ''; ?>">
        </div>
        <div class="form-group">
            <label for="direccion_despacho" class="form-label">Dirección de Despacho</label>
            <select id="direccion_despacho" name="direccion_despacho" class="form-control input-sm">
                <option value="">Seleccione dirección...</option>
                <option value="Bodega Principal - Lira 1133, Santiago" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Bodega Principal - Lira 1133, Santiago') ? 'selected' : ''; ?>>Bodega Principal - Lira 1133, Santiago</option>
                <option value="Tienda Las Condes - Puerta del sol 180, local 2, Las Condes" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Tienda Las Condes - Puerta del sol 180, local 2, Las Condes') ? 'selected' : ''; ?>>Tienda Las Condes - Puerta del sol 180, local 2, Las Condes</option>
                <option value="Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura') ? 'selected' : ''; ?>>Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura</option>
            </select>
        </div>
        <div class="form-group">
            <label for="tiempo_entrega" class="form-label">Tiempo de Entrega</label>
            <input type="text" id="tiempo_entrega" name="tiempo_entrega" class="form-control input-sm" 
                  placeholder="Ej: 5 días hábiles" autocomplete="off"
                  value="<?php echo isset($orden) ? htmlspecialchars($orden['tiempo_entrega'] ?? '') : ''; ?>">
        </div>
    </div>
</div>

<div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="motivo_orden" class="form-label">Motivo de la Orden</label>
            <select id="motivo_orden" name="motivo_orden" class="form-control input-sm">
                <option value="">Seleccione motivo...</option>
                <option value="Para cotización" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Para cotización') ? 'selected' : ''; ?>>Para cotización</option>
                <option value="Compras" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Compras') ? 'selected' : ''; ?>>Compras</option>
                <option value="Reposición" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Reposición') ? 'selected' : ''; ?>>Reposición</option>
                <option value="Otras" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Otras') ? 'selected' : ''; ?>>Otras</option>
            </select>
        </div>
    </div>
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="descripcion" class="form-label">Descripción</label>
            <textarea id="descripcion" name="descripcion" class="form-control" 
                      rows="3" placeholder="Descripción de la orden"><?php echo isset($orden) ? htmlspecialchars($orden['descripcion'] ?? '') : ''; ?></textarea>
        </div>
    </div>
</div>

<!-- Información del proveedor (se mostrará automáticamente cuando se seleccione un proveedor) -->
<div id="info-proveedor" class="proveedor-info" style="display: none; margin-top: 10px;">
    <h4>Información del Proveedor</h4>
    <p><strong>Dirección:</strong> <span id="prov-direccion"></span></p>
    <p><strong>Teléfono:</strong> <span id="prov-telefono"></span></p>
    <p><strong>Email:</strong> <span id="prov-email"></span></p>
    <p><strong>Contacto:</strong> <span id="prov-contacto"></span></p>
    <p><strong>Giro:</strong> <span id="prov-giro"></span></p>
</div>

                    
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr style="background-color: #f5f5f5;">
                                    <th style="width: 120px;">Código</th>
                                    <th>Descripción</th>
                                    <th style="width: 90px;">Cantidad</th>
                                    <th style="width: 130px;">Precio</th>
                                    <th style="width: 90px;">Descuento %</th>
                                    <th style="width: 130px;">Precio Final</th>
                                    <th style="width: 130px;">Total</th>
                                    <th style="width: 80px;">Acciones</th>
                                </tr>
                            </thead>
                            <tbody id="productos-container">
                                <tr>
                                    <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" autocomplete="off"></td>
                                    <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" required autocomplete="off"></td>
                                    <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="1" required autocomplete="off"></td>
                                    <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="0" required autocomplete="off"></td>
                                    <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="0" autocomplete="off"></td>
                                    <td><span class="precio-final">$0</span><input type="hidden" name="precio_final[]" class="precio-final-input" value="0"></td>
                                    <td><span class="subtotal">$0</span><input type="hidden" name="subtotal[]" class="subtotal-input" value="0"></td>
                                    <td><button type="button" class="btn btn-danger btn-sm" onclick="eliminarProducto(this.parentNode.parentNode)"><i class="material-icons">delete</i></button></td>
                                </tr>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="6" style="text-align: right;">Subtotal Neto:</th>
                                    <th id="subtotal-neto" style="text-align: right;">$0</th>
                                    <th></th>
                                </tr>
                                <tr>
                                    <th colspan="6" style="text-align: right;">Total Descuentos:</th>
                                    <th id="total-descuentos" style="text-align: right;">$0</th>
                                    <th></th>
                                </tr>
                                <tr>
                                    <th colspan="6" style="text-align: right;">Total Neto:</th>
                                    <th id="total-orden" style="text-align: right;">$0</th>
                                    <th></th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Orden de Compra
                    </button>
                </div>
            </form>
        </div>
        
        <div id="producto-template" style="display: none;">
            <tr>
                <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" autocomplete="off"></td>
                <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" required autocomplete="off"></td>
                <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="1" required autocomplete="off"></td>
                <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="0" required autocomplete="off"></td>
                <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="0" autocomplete="off"></td>
                <td><span class="precio-final">$0</span><input type="hidden" name="precio_final[]" class="precio-final-input" value="0"></td>
                <td><span class="subtotal">$0</span><input type="hidden" name="subtotal[]" class="subtotal-input" value="0"></td>
                <td><button type="button" class="btn btn-danger btn-sm" onclick="eliminarProducto(this.parentNode.parentNode)"><i class="material-icons">delete</i></button></td>
            </tr>
        </div>
        <?php
        break;
        
    case 'edit':
        try {
            $stmt = $conn->prepare("SELECT * FROM ordenes_compra WHERE id = :id " . (!$is_admin ? "AND user_id = :user_id" : ""));
            $stmt->bindParam(':id', $id);
            if (!$is_admin) $stmt->bindParam(':user_id', $user_id);
            $stmt->execute();
            $orden = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$orden) throw new Exception("Orden de compra no encontrada o sin permisos");
            if ($orden['estado_id'] != 1 && $orden['estado_id'] != 4) throw new Exception("No se puede editar una orden aprobada o en revisión");
            if (!$is_admin && !estaEnTiempoEdicion($orden['fecha_creacion'])) throw new Exception("Solo se puede editar dentro de las 24 horas");
            
            $stmt = $conn->prepare("SELECT * FROM detalle_orden_compra WHERE orden_compra_id = :id ORDER BY id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Editar Orden de Compra</h2>
                    <a href="ordenes_compra.php" class="btn btn-primary btn-sm">
                        <i class="material-icons">arrow_back</i> Volver a la Lista
                    </a>
                </div>
                
                <form method="POST" action="ordenes_compra.php?action=edit&id=<?php echo $id; ?>" id="form-orden">
                    <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="rut_proveedor" class="form-label">RUT del Proveedor</label>
            <div class="input-group">
                <input type="text" id="rut_proveedor" name="rut_proveedor" class="form-control input-sm" 
                       placeholder="Ej: 12.345.678-5" required autocomplete="off" 
                       value="<?php echo isset($orden) ? htmlspecialchars($orden['rut_proveedor']) : ''; ?>">
            </div>
            <div id="mensaje-proveedor" class="form-text"></div>
        </div>
        <div class="form-group">
            <label for="proveedor" class="form-label">Nombre del Proveedor</label>
            <input type="text" id="proveedor" name="proveedor" class="form-control input-sm" 
                   placeholder="Nombre de la empresa proveedora" required autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['proveedor']) : ''; ?>">
            <input type="hidden" id="proveedor_id" name="proveedor_id"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['proveedor_id'] ?? '') : ''; ?>">
        </div>
        <div class="form-group">
            <label for="fecha" class="form-label">Fecha</label>
            <input type="date" id="fecha" name="fecha" class="form-control input-sm" 
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['fecha']) : date('Y-m-d'); ?>" required>
        </div>
        <div class="form-group">
            <label for="area_solicitud" class="form-label">Área de Solicitud</label>
            <select id="area_solicitud" name="area_solicitud" class="form-control input-sm" required>
                <option value="">Seleccione área...</option>
                <option value="bodega" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'bodega') ? 'selected' : ''; ?>>Bodega</option>
                <option value="servicio_tecnico" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'servicio_tecnico') ? 'selected' : ''; ?>>Servicio Técnico</option>
                <option value="tiendas" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'tiendas') ? 'selected' : ''; ?>>Tiendas</option>
                <option value="otros" <?php echo (isset($orden) && isset($orden['area_solicitud']) && $orden['area_solicitud'] == 'otros') ? 'selected' : ''; ?>>Otros</option>
            </select>
        </div>
    </div>
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="numero_documento" class="form-label">N° Documento</label>
            <input type="text" id="numero_documento" name="numero_documento" class="form-control input-sm" 
                   placeholder="Número de documento" required autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['numero_documento']) : ''; ?>">
        </div>
        <div class="form-group">
            <label for="condiciones_pago" class="form-label">Condiciones de Pago</label>
            <input type="text" id="condiciones_pago" name="condiciones_pago" class="form-control input-sm" 
                   placeholder="Ej: 30 días" autocomplete="off"
                   value="<?php echo isset($orden) ? htmlspecialchars($orden['condiciones_pago'] ?? '') : ''; ?>">
        </div>
        <div class="form-group">
            <label for="direccion_despacho" class="form-label">Dirección de Despacho</label>
            <select id="direccion_despacho" name="direccion_despacho" class="form-control input-sm">
                <option value="">Seleccione dirección...</option>
                <option value="Bodega Principal - Lira 1133, Santiago" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Bodega Principal - Lira 1133, Santiago') ? 'selected' : ''; ?>>Bodega Principal - Lira 1133, Santiago</option>
                <option value="Tienda Las Condes - Puerta del sol 180, local 2, Las Condes" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Tienda Las Condes - Puerta del sol 180, local 2, Las Condes') ? 'selected' : ''; ?>>Tienda Las Condes - Puerta del sol 180, local 2, Las Condes</option>
                <option value="Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura" <?php echo (isset($orden) && isset($orden['direccion_despacho']) && $orden['direccion_despacho'] == 'Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura') ? 'selected' : ''; ?>>Tienda Vitacura - Alonso de Córdova 2350, local 2, Vitacura</option>
            </select>
        </div>
        <div class="form-group">
            <label for="tiempo_entrega" class="form-label">Tiempo de Entrega</label>
            <input type="text" id="tiempo_entrega" name="tiempo_entrega" class="form-control input-sm" 
                  placeholder="Ej: 5 días hábiles" autocomplete="off"
                  value="<?php echo isset($orden) ? htmlspecialchars($orden['tiempo_entrega'] ?? '') : ''; ?>">
        </div>
    </div>
</div>

<div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="motivo_orden" class="form-label">Motivo de la Orden</label>
            <select id="motivo_orden" name="motivo_orden" class="form-control input-sm">
                <option value="">Seleccione motivo...</option>
                <option value="Para cotización" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Para cotización') ? 'selected' : ''; ?>>Para cotización</option>
                <option value="Compras" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Compras') ? 'selected' : ''; ?>>Compras</option>
                <option value="Reposición" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Reposición') ? 'selected' : ''; ?>>Reposición</option>
                <option value="Otras" <?php echo (isset($orden) && isset($orden['motivo_orden']) && $orden['motivo_orden'] == 'Otras') ? 'selected' : ''; ?>>Otras</option>
            </select>
        </div>
    </div>
    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
        <div class="form-group">
            <label for="descripcion" class="form-label">Descripción</label>
            <textarea id="descripcion" name="descripcion" class="form-control" 
                      rows="3" placeholder="Descripción de la orden"><?php echo isset($orden) ? htmlspecialchars($orden['descripcion'] ?? '') : ''; ?></textarea>
        </div>
    </div>
</div>

<!-- Información del proveedor (se mostrará automáticamente cuando se seleccione un proveedor) -->
<div id="info-proveedor" class="proveedor-info" style="display: none; margin-top: 10px;">
    <h4>Información del Proveedor</h4>
    <p><strong>Dirección:</strong> <span id="prov-direccion"></span></p>
    <p><strong>Teléfono:</strong> <span id="prov-telefono"></span></p>
    <p><strong>Email:</strong> <span id="prov-email"></span></p>
    <p><strong>Contacto:</strong> <span id="prov-contacto"></span></p>
    <p><strong>Giro:</strong> <span id="prov-giro"></span></p>
</div>

                    
                    <div class="card" style="margin-top: 20px;">
                        <div class="panel-header">
                            <h3 class="panel-title">Detalle de Productos y Servicios</h3>
                            <button type="button" id="btn-agregar-producto" class="btn btn-success btn-sm">
                                <i class="material-icons">add</i> Agregar Producto
                            </button>
                        </div>
                        
                        <div class="table-responsive">
                            <table>
                                <thead>
                                    <tr style="background-color: #f5f5f5;">
                                        <th style="width: 120px;">Código</th>
                                        <th>Descripción</th>
                                        <th style="width: 90px;">Cantidad</th>
                                        <th style="width: 130px;">Precio</th>
                                        <th style="width: 90px;">Descuento %</th>
                                        <th style="width: 130px;">Precio Final</th>
                                        <th style="width: 130px;">Total</th>
                                        <th style="width: 80px;">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody id="productos-container">
                                    <?php if (empty($productos)): ?>
                                        <tr>
                                            <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" autocomplete="off"></td>
                                            <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" required autocomplete="off"></td>
                                            <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="1" required autocomplete="off"></td>
                                            <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="0" required autocomplete="off"></td>
                                            <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="0" autocomplete="off"></td>
                                            <td><span class="precio-final">$0</span><input type="hidden" name="precio_final[]" class="precio-final-input" value="0"></td>
                                            <td><span class="subtotal">$0</span><input type="hidden" name="subtotal[]" class="subtotal-input" value="0"></td>
                                            <td><button type="button" class="btn btn-danger btn-sm" onclick="eliminarProducto(this.parentNode.parentNode)"><i class="material-icons">delete</i></button></td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($productos as $producto): ?>
                                            <tr>
                                                <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" value="<?php echo htmlspecialchars($producto['codigo_producto'] ?? ''); ?>" autocomplete="off"></td>
                                                <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" value="<?php echo htmlspecialchars($producto['producto']); ?>" required autocomplete="off"></td>
                                                <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="<?php echo $producto['cantidad']; ?>" required autocomplete="off"></td>
                                                <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="<?php echo $producto['precio_unitario']; ?>" required autocomplete="off"></td>
                                                <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="<?php echo isset($producto['descuento']) ? $producto['descuento'] : 0; ?>" autocomplete="off"></td>
                                                <td><span class="precio-final"><?php echo formatoMoneda(isset($producto['precio_final']) ? $producto['precio_final'] : $producto['precio_unitario']); ?></span><input type="hidden" name="precio_final[]" class="precio-final-input" value="<?php echo isset($producto['precio_final']) ? $producto['precio_final'] : $producto['precio_unitario']; ?>"></td>
                                                <td><span class="subtotal"><?php echo formatoMoneda($producto['subtotal']); ?></span><input type="hidden" name="subtotal[]" class="subtotal-input" value="<?php echo $producto['subtotal']; ?>"></td>
                                                <td><button type="button" class="btn btn-danger btn-sm" onclick="eliminarProducto(this.parentNode.parentNode)"><i class="material-icons">delete</i></button></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <th colspan="6" style="text-align: right;">Subtotal Neto:</th>
                                        <th id="subtotal-neto" style="text-align: right;"><?php echo formatoMoneda($orden['monto_total']); ?></th>
                                        <th></th>
                                    </tr>
                                    <tr>
                                        <th colspan="6" style="text-align: right;">Total Descuentos:</th>
                                        <th id="total-descuentos" style="text-align: right;">$0</th>
                                        <th></th>
                                    </tr>
                                    <tr>
                                        <th colspan="6" style="text-align: right;">Total Neto:</th>
                                        <th id="total-orden" style="text-align: right;"><?php echo formatoMoneda($orden['monto_total']); ?></th>
                                        <th></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px; text-align: center;">
                        <button type="submit" class="btn btn-primary">
                            <i class="material-icons">save</i> Actualizar Orden de Compra
                        </button>
                    </div>
                </form>
            </div>
            
            <div id="producto-template" style="display: none;">
                <tr>
                    <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" autocomplete="off"></td>
                    <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" required autocomplete="off"></td>
                    <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="1" required autocomplete="off"></td>
                    <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="0" required autocomplete="off"></td>
                    <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="0" autocomplete="off"></td>
                    <td><span class="precio-final">$0</span><input type="hidden" name="precio_final[]" class="precio-final-input" value="0"></td>
                    <td><span class="subtotal">$0</span><input type="hidden" name="subtotal[]" class="subtotal-input" value="0"></td>
                    <td><button type="button" class="btn btn-danger btn-sm" onclick="eliminarProducto(this.parentNode.parentNode)"><i class="material-icons">delete</i></button></td>
                </tr>
            </div>
            <?php
        } catch (Exception $e) {
            echo '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
            echo '<div style="text-align: center; margin-top: 20px;"><a href="ordenes_compra.php" class="btn btn-primary"><i class="material-icons">arrow_back</i> Volver a la Lista</a></div>';
        }
        break;
        
    case 'view':
        try {
            $stmt = $conn->prepare("
                SELECT oc.*, u.nombre, u.apellido, u.rut, u.cargo,
                       p.direccion, p.telefono, p.email, p.contacto, p.giro,
                       rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
                FROM ordenes_compra oc
                JOIN users u ON oc.user_id = u.id
                LEFT JOIN proveedores p ON oc.proveedor_id = p.id
                LEFT JOIN users rev ON oc.revisor_id = rev.id
                WHERE oc.id = :id " . (!$is_admin ? "AND (oc.user_id = :user_id OR oc.revisor_id = :user_id)" : "")
            );
            $stmt->bindParam(':id', $id);
            if (!$is_admin) $stmt->bindParam(':user_id', $user_id);
            $stmt->execute();
            $orden = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$orden) throw new Exception("Orden de compra no encontrada o sin permisos");
            
            $stmt = $conn->prepare("SELECT * FROM detalle_orden_compra WHERE orden_compra_id = :id ORDER BY id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $codigo = generarCodigo('OC', $id);
            
            $iva = 0.19;
            $subtotal_neto = 0;
            $total_iva = 0;
            $total_bruto = 0;
            
            foreach ($productos as $producto) {
                $subtotal_neto += $producto['subtotal'];
                $iva_monto = $producto['subtotal'] * $iva;
                $total_iva += $iva_monto;
                $total_bruto += $producto['subtotal'] + $iva_monto;
            }
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Orden de Compra #<?php echo $codigo; ?></h2>
                    <div class="panel-actions">
                        <?php if ($orden['estado_id'] == 3 || $orden['estado_id'] == 5): ?>
                        <a href="generate_pdf.php?tipo=orden_compra&id=<?php echo $id; ?>" class="btn btn-primary btn-sm">
                            <i class="material-icons">picture_as_pdf</i> Generar PDF
                        </a>
                        <?php endif; ?>
                        <?php if (($orden['estado_id'] == 1 || $orden['estado_id'] == 4) && ($orden['user_id'] == $user_id || $is_admin)): ?>
                        <a href="ordenes_compra.php?action=edit&id=<?php echo $id; ?>" class="btn btn-warning btn-sm">
                            <i class="material-icons">edit</i> Editar
                        </a>
                        <?php endif; ?>
                        <?php if ($is_admin && $orden['estado_id'] == 1): ?>
                        <a href="ordenes_compra.php?action=delete&id=<?php echo $id; ?>" class="btn btn-danger btn-sm" 
                           onclick="return confirm('¿Está seguro de eliminar esta orden de compra?')">
                            <i class="material-icons">delete</i> Eliminar
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h3>Información del Proveedor</h3>
                            <table class="table table-bordered">
                                <tr>
                                    <th>Nombre:</th>
                                    <td><?php echo htmlspecialchars($orden['proveedor']); ?></td>
                                </tr>
                                <tr>
                                    <th>RUT:</th>
                                    <td><?php echo htmlspecialchars($orden['rut_proveedor']); ?></td>
                                </tr>
                                <?php if (!empty($orden['direccion'])): ?>
                                <tr>
                                    <th>Dirección:</th>
                                    <td><?php echo htmlspecialchars($orden['direccion']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['telefono'])): ?>
                                <tr>
                                    <th>Teléfono:</th>
                                    <td><?php echo htmlspecialchars($orden['telefono']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['email'])): ?>
                                <tr>
                                    <th>Email:</th>
                                    <td><?php echo htmlspecialchars($orden['email']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['contacto'])): ?>
                                <tr>
                                    <th>Contacto:</th>
                                    <td><?php echo htmlspecialchars($orden['contacto']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['giro'])): ?>
                                <tr>
                                    <th>Giro:</th>
                                    <td><?php echo htmlspecialchars($orden['giro']); ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                        
                        <div class="col-md-6">
                            <h3>Información de la Orden</h3>
                            <table class="table table-bordered">
                                <tr>
                                    <th>Estado:</th>
                                    <td>
                                        <span class="badge" style="background-color: <?php echo $estados_map[$orden['estado_id']]['color']; ?>">
                                            <?php echo htmlspecialchars($estados_map[$orden['estado_id']]['nombre']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Fecha:</th>
                                    <td><?php echo date('d/m/Y', strtotime($orden['fecha'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Número:</th>
                                    <td><?php echo htmlspecialchars($orden['numero_documento']); ?></td>
                                </tr>
                                <tr>
                                    <th>Solicitante:</th>
                                    <td><?php echo htmlspecialchars($orden['nombre'] . ' ' . $orden['apellido']); ?></td>
                                </tr>
                                <?php if (!empty($orden['descripcion'])): ?>
                                <tr>
                                    <th>Descripción:</th>
                                    <td><?php echo nl2br(htmlspecialchars($orden['descripcion'])); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['condiciones_pago'])): ?>
                                <tr>
                                    <th>Condiciones de Pago:</th>
                                    <td><?php echo htmlspecialchars($orden['condiciones_pago']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['direccion_despacho'])): ?>
                                <tr>
                                    <th>Dirección de Despacho:</th>
                                    <td><?php echo htmlspecialchars($orden['direccion_despacho']); ?></td>
                                </tr>
                                <?php endif; ?>
                                <?php if (!empty($orden['tiempo_entrega'])): ?>
                                <tr>
                                    <th>Tiempo de Entrega:</th>
                                    <td><?php echo htmlspecialchars($orden['tiempo_entrega']); ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                    </div>
                    
                    <h3>Detalle de Productos</h3>
                    <div class="table-responsive">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Producto</th>
                                    <th>Código</th>
                                    <th>Cantidad</th>
                                    <th>Precio Unit. Neto</th>
                                    <th>Descuento</th>
                                    <th>Subtotal Neto</th>
                                    <th>IVA (19%)</th>
                                    <th>Total Bruto</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $iva = 0.19;
                                $subtotal_neto = 0;
                                $total_iva = 0;
                                $total_bruto = 0;
                                
                                foreach ($productos as $index => $producto): 
                                    $precio_unitario = $producto['precio_unitario'];
                                    $descuento = isset($producto['descuento']) ? $producto['descuento'] : 0;
                                    $precio_final = $precio_unitario * (1 - ($descuento / 100));
                                    $subtotal_linea = $precio_final * $producto['cantidad'];
                                    $iva_linea = $subtotal_linea * $iva;
                                    $total_linea = $subtotal_linea + $iva_linea;
                                    
                                    $subtotal_neto += $subtotal_linea;
                                    $total_iva += $iva_linea;
                                    $total_bruto += $total_linea;
                                ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td><?php echo htmlspecialchars($producto['producto']); ?></td>
                                    <td><?php echo htmlspecialchars($producto['codigo_producto'] ?? ''); ?></td>
                                    <td class="text-right"><?php echo $producto['cantidad']; ?></td>
                                    <td class="text-right"><?php echo formatoMoneda($precio_unitario); ?></td>
                                    <td class="text-right"><?php echo $descuento . '%'; ?></td>
                                    <td class="text-right"><?php echo formatoMoneda($subtotal_linea); ?></td>
                                    <td class="text-right"><?php echo formatoMoneda($iva_linea); ?></td>
                                    <td class="text-right"><?php echo formatoMoneda($total_linea); ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th colspan="6" class="text-right">Subtotal Neto:</th>
                                    <td class="text-right"><?php echo formatoMoneda($subtotal_neto); ?></td>
                                    <td></td>
                                    <td></td>
                                </tr>
                                <tr>
                                    <th colspan="6" class="text-right">IVA (19%):</th>
                                    <td></td>
                                    <td class="text-right"><?php echo formatoMoneda($total_iva); ?></td>
                                    <td></td>
                                </tr>
                                <tr>
                                    <th colspan="6" class="text-right">Total Bruto:</th>
                                    <td></td>
                                    <td></td>
                                    <td class="text-right"><?php echo formatoMoneda($total_bruto); ?></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                    
                    <?php if ($orden['estado_id'] == 3 || $orden['estado_id'] == 5): ?>
                    <div class="alert alert-success">
                        <strong>Aprobado por:</strong> <?php echo htmlspecialchars($orden['revisor_nombre'] . ' ' . $orden['revisor_apellido']); ?>
                        <br>
                        <strong>Fecha de aprobación:</strong> <?php echo date('d/m/Y H:i', strtotime($orden['fecha_revision'])); ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php
        } catch (Exception $e) {
            $error_message = "Error al cargar la orden de compra: " . $e->getMessage();
        }
        break;
        
    default:
        try {
            $estado = isset($_GET['estado']) ? intval($_GET['estado']) : 0;
            $fecha_desde = $_GET['fecha_desde'] ?? '';
            $fecha_hasta = $_GET['fecha_hasta'] ?? '';
            $busqueda = trim($_GET['busqueda'] ?? '');
            
            $sql = "SELECT oc.*, u.nombre, u.apellido, u.rut FROM ordenes_compra oc JOIN users u ON oc.user_id = u.id WHERE 1=1";
            $params = [];
            if (!$is_admin) {
                $sql .= " AND oc.user_id = :user_id";
                $params[':user_id'] = $user_id;
            }
            if ($estado > 0) {
                $sql .= " AND oc.estado_id = :estado";
                $params[':estado'] = $estado;
            }
            if (!empty($fecha_desde)) {
                $sql .= " AND oc.fecha >= :fecha_desde";
                $params[':fecha_desde'] = $fecha_desde;
            }
            if (!empty($fecha_hasta)) {
                $sql .= " AND oc.fecha <= :fecha_hasta";
                $params[':fecha_hasta'] = $fecha_hasta;
            }
            if (!empty($busqueda)) {
                $sql .= " AND (oc.proveedor LIKE :busqueda OR oc.rut_proveedor LIKE :busqueda OR oc.numero_documento LIKE :busqueda OR oc.descripcion LIKE :busqueda OR u.nombre LIKE :busqueda OR u.apellido LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            $sql .= " ORDER BY oc.fecha_creacion DESC";
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) $stmt->bindValue($param, $value);
            $stmt->execute();
            $ordenes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Órdenes de Compra</h2>
                    <?php if (tienePermiso('crear_orden_compra') || $is_admin): ?>
                    <a href="ordenes_compra.php?action=new" class="btn btn-primary btn-sm"><i class="material-icons">add</i> Nueva Orden de Compra</a>
                    <?php endif; ?>
                </div>
                
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header"><h3 class="panel-title">Filtros de Búsqueda</h3></div>
                    <form method="GET" action="ordenes_compra.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px;">
                        <div style="flex: 1; min-width: 200px;">
                            <label for="estado" class="form-label">Estado</label>
                            <select id="estado" name="estado" class="form-control">
                                <option value="0">Todos</option>
                                <?php foreach ($estados as $est): ?>
                                <option value="<?php echo $est['id']; ?>" <?php echo $estado == $est['id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($est['nombre']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_desde" class="form-label">Fecha Desde</label>
                            <input type="date" id="fecha_desde" name="fecha_desde" class="form-control" value="<?php echo $fecha_desde; ?>">
                        </div>
                        <div style="flex: 1; min-width: 200px;">
                            <label for="fecha_hasta" class="form-label">Fecha Hasta</label>
                            <input type="date" id="fecha_hasta" name="fecha_hasta" class="form-control" value="<?php echo $fecha_hasta; ?>">
                        </div>
                        <div style="flex: 2; min-width: 300px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" placeholder="Proveedor, N° Documento, RUT..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary"><i class="material-icons">search</i> Buscar</button>
                            <a href="ordenes_compra.php" class="btn btn-danger"><i class="material-icons">clear</i> Limpiar</a>
                        </div>
                    </form>
                </div>
                
                <div class="table-responsive">
                    <table class="ordenes-table">
                        <thead><tr><th>Código</th><th>Fecha</th><th>Proveedor</th><th>N° Documento</th><th>Monto</th><?php if ($is_admin): ?><th>Usuario</th><?php endif; ?><th>Estado</th><th>Acciones</th></tr></thead>
                        <tbody>
                            <?php if (empty($ordenes)): ?>
                            <tr><td colspan="<?php echo $is_admin ? '8' : '7'; ?>" style="text-align: center;">No se encontraron órdenes de compra</td></tr>
                            <?php else: ?>
                                <?php foreach ($ordenes as $orden): $codigo = generarCodigo('OC', $orden['id']); ?>
                                <tr>
                                    <td><?php echo $codigo; ?></td>
                                    <td><?php echo formatoFecha($orden['fecha']); ?></td>
                                    <td><?php echo htmlspecialchars(substr($orden['proveedor'], 0, 30)) . (strlen($orden['proveedor']) > 30 ? '...' : ''); ?></td>
                                    <td><?php echo htmlspecialchars($orden['numero_documento']); ?></td>
                                    <td><?php 
                                        $monto_neto = $orden['monto_total'];
                                        $iva = 0.19;
                                        $monto_bruto = $monto_neto * (1 + $iva);
                                        echo formatoMoneda($monto_bruto); 
                                    ?></td>
                                    <?php if ($is_admin): ?><td><?php echo htmlspecialchars($orden['nombre'] . ' ' . $orden['apellido']); ?></td><?php endif; ?>
                                    <td><span class="status-badge badge-estado-<?php echo $orden['estado_id']; ?>"><?php echo $estados_map[$orden['estado_id']]['nombre'] ?? 'Desconocido'; ?></span></td>
                                    <td>
                                        <a href="ordenes_compra.php?action=view&id=<?php echo $orden['id']; ?>" title="Ver detalles" class="action-icon view-icon"><i class="material-icons">visibility</i></a>
                                        <?php if ($orden['estado_id'] == 3 || $orden['estado_id'] == 5): ?>
                                        <a href="generate_pdf.php?tipo=orden_compra&id=<?php echo $orden['id']; ?>" title="Generar PDF" class="action-icon pdf-icon"><i class="material-icons">picture_as_pdf</i></a>
                                        <?php endif; ?>
                                        <?php if (($orden['estado_id'] == 1 || $orden['estado_id'] == 4) && ($orden['user_id'] == $user_id || $is_admin)): ?>
                                        <a href="ordenes_compra.php?action=edit&id=<?php echo $orden['id']; ?>" title="Editar" class="action-icon edit-icon"><i class="material-icons">edit</i></a>
                                        <a href="generate_pdf.php?tipo=orden_compra&id=<?php echo $orden['id']; ?>" class="btn btn-info btn-sm"><i class="material-icons">picture_as_pdf</i> Vista Previa PDF</a>
                                        <?php endif; ?>
                                        <?php if ($is_admin && ($orden['estado_id'] == 1 || $orden['estado_id'] == 4)): ?>
                                        <a href="ordenes_compra.php?action=delete&id=<?php echo $orden['id']; ?>" onclick="return confirm('¿Está seguro de eliminar esta orden de compra?')" title="Eliminar" class="action-icon delete-icon"><i class="material-icons">delete</i></a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar órdenes de compra: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Inicialización
    document.querySelectorAll('#productos-container tr').forEach(actualizarCalculos);
    actualizarTotales();

    // Botón para mostrar/ocultar datos adicionales
    const btnDatosAdicionales = document.getElementById('btn-datos-adicionales');
    if (btnDatosAdicionales) {
        btnDatosAdicionales.addEventListener('click', function(e) {
            e.preventDefault();
            const datosAdicionales = document.getElementById('datos-adicionales');
            if (datosAdicionales) {
                datosAdicionales.style.display = datosAdicionales.style.display === 'none' ? 'block' : 'none';
            }
        });
    }

    // Botón para agregar producto
    const btnAgregar = document.getElementById('btn-agregar-producto');
    if (btnAgregar) {
        btnAgregar.addEventListener('click', function(e) {
            e.preventDefault();
            agregarProductoNuevo();
        });
    }

    // Event listeners para actualizar cálculos
    document.querySelectorAll('.cantidad-input, .precio-input, .descuento-input').forEach(input => {
        input.addEventListener('input', function() {
            actualizarCalculos(this.closest('tr'));
            actualizarTotales();
        });
    });

    // Validación del formulario
    const formOrden = document.getElementById('form-orden');
    if (formOrden) {
        formOrden.addEventListener('submit', function(e) {
            const rutInput = document.getElementById('rut_proveedor');
            if (rutInput && rutInput.value.trim().length > 0) {
                rutInput.value = formatearRut(rutInput.value);
                if (!validaRut(rutInput.value)) {
                    e.preventDefault();
                    alert('El RUT del proveedor no es válido. Ejemplo válido: 12.345.678-5');
                    rutInput.focus();
                    return;
                }
            }

            const invalidFields = this.querySelectorAll(':invalid');
            if (invalidFields.length > 0) {
                e.preventDefault();
                let errorMessages = [];
                invalidFields.forEach(field => {
                    if (field.name === 'proveedor') errorMessages.push('El nombre del proveedor es obligatorio.');
                    else if (field.name === 'rut_proveedor') errorMessages.push('El RUT del proveedor es obligatorio.');
                    else if (field.name === 'fecha') errorMessages.push('La fecha es obligatoria.');
                    else if (field.name === 'numero_documento') errorMessages.push('El número de documento es obligatorio.');
                    else if (field.name === 'area_solicitud') errorMessages.push('El área de solicitud es obligatoria.');
                    else if (field.name === 'producto[]') errorMessages.push('Falta la descripción de un producto o servicio.');
                    else if (field.name === 'cantidad[]') errorMessages.push('La cantidad debe ser mayor a 0.');
                    else if (field.name === 'precio_unitario[]') errorMessages.push('El precio unitario debe ser mayor a 0.');
                });
                alert('Por favor, corrige los siguientes errores:\n- ' + errorMessages.join('\n- '));
                invalidFields[0].focus();
                return;
            }
        });
    }

    // Autodetección del proveedor al ingresar RUT completo
    const rutInput = document.getElementById('rut_proveedor');
    if (rutInput) {
        rutInput.addEventListener('input', function() {
            this.value = formatearRut(this.value);
            // Buscar proveedor automáticamente cuando el RUT está completo (tiene un guión y al menos un dígito después)
            if (this.value.includes('-') && this.value.split('-')[1].length > 0) {
                buscarProveedorPorRut();
            }
        });

        // Buscar proveedor al cargar la página si ya hay un RUT
        if (rutInput.value.includes('-') && rutInput.value.split('-')[1].length > 0) {
            buscarProveedorPorRut();
        }
    }
});

// Reemplaza la función agregarProductoNuevo con esta versión mejorada
function agregarProductoNuevo() {
    const tabla = document.getElementById('productos-container');
    if (!tabla) {
        console.error('No se encontró el contenedor de productos');
        return;
    }
    
    // Crear una nueva fila directamente con HTML
    const fila = document.createElement('tr');
    fila.innerHTML = `
        <td><input type="text" name="codigo_producto[]" class="form-control input-sm" placeholder="Código" autocomplete="off"></td>
        <td><input type="text" name="producto[]" class="form-control input-sm" placeholder="Descripción del producto" required autocomplete="off"></td>
        <td><input type="number" name="cantidad[]" class="form-control cantidad-input input-sm" min="0" step="1" value="1" required autocomplete="off"></td>
        <td><input type="number" name="precio_unitario[]" class="form-control precio-input input-sm" min="0" step="1" value="0" required autocomplete="off"></td>
        <td><input type="number" name="descuento[]" class="form-control descuento-input input-sm" min="0" max="100" step="1" value="0" autocomplete="off"></td>
        <td><span class="precio-final">$0</span><input type="hidden" name="precio_final[]" class="precio-final-input" value="0"></td>
        <td><span class="subtotal">$0</span><input type="hidden" name="subtotal[]" class="subtotal-input" value="0"></td>
        <td><button type="button" class="btn btn-danger btn-sm"><i class="material-icons">delete</i></button></td>
    `;
    
    // Agregar la fila a la tabla
    tabla.appendChild(fila);
    
    // Configurar event listeners para inputs
    fila.querySelectorAll('.cantidad-input, .precio-input, .descuento-input').forEach(input => {
        input.addEventListener('input', function() {
            actualizarCalculos(this.closest('tr'));
            actualizarTotales();
        });
    });
    
    // Configurar event listener para el botón de eliminar
    const btnEliminar = fila.querySelector('button');
    if (btnEliminar) {
        btnEliminar.addEventListener('click', function() {
            eliminarProducto(this.closest('tr'));
        });
    }
    
    // Actualizar cálculos iniciales
    actualizarCalculos(fila);
    actualizarTotales();
}

// También mejoramos la función eliminarProducto para mayor robustez
function eliminarProducto(fila) {
    if (!fila) {
        console.error('No se especificó una fila para eliminar');
        return;
    }
    
    const tabla = document.getElementById('productos-container');
    if (!tabla) {
        console.error('No se encontró el contenedor de productos');
        return;
    }
    
    const filas = tabla.querySelectorAll('tr');
    if (filas.length <= 1) {
        alert('Debe haber al menos un producto.');
        return;
    }
    
    fila.remove();
    actualizarTotales();
}

function eliminarProducto(fila) {
    if (document.querySelectorAll('#productos-container tr').length <= 1) {
        alert('Debe haber al menos un producto.');
        return;
    }
    fila.remove();
    actualizarTotales();
}

function actualizarCalculos(fila) {
    const cantidad = parseInt(fila.querySelector('.cantidad-input').value) || 0;
    const precio = parseFloat(fila.querySelector('.precio-input').value) || 0;
    const descuento = parseFloat(fila.querySelector('.descuento-input').value) || 0;
    
    const precioFinal = precio * (1 - (descuento / 100));
    const subtotal = cantidad * precioFinal;
    
    fila.querySelector('.precio-final').textContent = formatoMoneda(precioFinal);
    fila.querySelector('.precio-final-input').value = precioFinal;
    fila.querySelector('.subtotal').textContent = formatoMoneda(subtotal);
    fila.querySelector('.subtotal-input').value = subtotal;
}

function actualizarTotales() {
    let subtotalNeto = 0;
    let totalDescuentos = 0;
    let totalNeto = 0;
    
    document.querySelectorAll('#productos-container tr').forEach(fila => {
        const cantidad = parseInt(fila.querySelector('.cantidad-input').value) || 0;
        const precio = parseFloat(fila.querySelector('.precio-input').value) || 0;
        const descuento = parseFloat(fila.querySelector('.descuento-input').value) || 0;
        
        const montoSinDescuento = cantidad * precio;
        const precioFinal = precio * (1 - (descuento / 100));
        const subtotal = cantidad * precioFinal;
        const montoDescuento = montoSinDescuento - subtotal;
        
        subtotalNeto += montoSinDescuento;
        totalDescuentos += montoDescuento;
        totalNeto += subtotal;
    });
    
    document.getElementById('subtotal-neto').textContent = formatoMoneda(subtotalNeto);
    document.getElementById('total-descuentos').textContent = formatoMoneda(totalDescuentos);
    document.getElementById('total-orden').textContent = formatoMoneda(totalNeto);
}

function formatoMoneda(valor) {
    return '$' + parseInt(valor).toLocaleString('es-CL');
}

function validaRut(rut) {
    if (!rut || typeof rut !== 'string') return false;
    
    // Acepta RUT con o sin formato (12.345.678-5 o 123456785)
    const cleanRut = rut.replace(/[^0-9kK]/g, '');
    if (cleanRut.length < 2) return false;
    
    const cuerpo = cleanRut.slice(0, -1);
    const dv = cleanRut.slice(-1).toLowerCase();
    if (!/^\d+$/.test(cuerpo)) return false;
    
    let suma = 0, multiplo = 2;
    for (let i = cuerpo.length - 1; i >= 0; i--) {
        suma += parseInt(cuerpo.charAt(i)) * multiplo;
        multiplo = multiplo < 7 ? multiplo + 1 : 2;
    }
    
    const dvEsperado = 11 - (suma % 11);
    const dvCalculado = dvEsperado === 11 ? '0' : dvEsperado === 10 ? 'k' : dvEsperado.toString();
    return dvCalculado === dv;
}

function formatearRut(rut) {
    if (!rut) return '';
    const cleanRut = rut.replace(/[^0-9kK]/g, '');
    if (cleanRut.length <= 1) return cleanRut;
    
    const dv = cleanRut.slice(-1);
    let cuerpo = cleanRut.slice(0, -1);
    cuerpo = cuerpo.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    return `${cuerpo}-${dv}`;
}

function buscarProveedorPorRut() {
    const rutInput = document.getElementById('rut_proveedor');
    const mensajeProveedor = document.getElementById('mensaje-proveedor');
    const infoProveedor = document.getElementById('info-proveedor');
    
    if (!rutInput || !rutInput.value) {
        if (mensajeProveedor) mensajeProveedor.innerHTML = 'Debe ingresar un RUT para buscar';
        if (infoProveedor) infoProveedor.style.display = 'none';
        return;
    }
    
    if (mensajeProveedor) mensajeProveedor.innerHTML = 'Buscando proveedor...';
    
    fetch('buscar_proveedor.php?rut=' + encodeURIComponent(rutInput.value))
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('proveedor').value = data.proveedor.nombre;
                document.getElementById('proveedor_id').value = data.proveedor.id;
                
                // Mostrar información del proveedor
                if (infoProveedor) {
                    infoProveedor.style.display = 'block';
                    document.getElementById('prov-direccion').textContent = data.proveedor.direccion || 'No especificada';
                    document.getElementById('prov-telefono').textContent = data.proveedor.telefono || 'No especificado';
                    document.getElementById('prov-email').textContent = data.proveedor.email || 'No especificado';
                    document.getElementById('prov-contacto').textContent = data.proveedor.contacto || 'No especificado';
                    document.getElementById('prov-giro').textContent = data.proveedor.giro || 'No especificado';
                }
                