<?php
/**
 * get_series.php
 * Obtiene las series asociadas a una máquina específica
 * Devuelve los datos en formato JSON
 */

require_once 'includes/config.php';
require_once 'includes/functions.php';

verificarSesion();

header('Content-Type: application/json');

try {
    // Validar parámetro
    $maquina_id = isset($_GET['maquina_id']) ? intval($_GET['maquina_id']) : 0;
    if ($maquina_id <= 0) {
        throw new Exception("ID de máquina no válido");
    }

    // Conexión a la base de datos
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    
    // Consulta para obtener las series de la máquina
    $stmt = $conn->prepare("
        SELECT 
            s.id,
            s.codigo_serie,
            s.descripcion,
            s.estado,
            s.proveedor_id as cliente_id,
            CASE 
                WHEN s.proveedor_id IS NOT NULL THEN p.nombre 
                ELSE NULL 
            END as cliente_nombre,
            CASE 
                WHEN s.proveedor_id IS NOT NULL THEN p.rut 
                ELSE NULL 
            END as cliente_rut
        FROM series_maquinas s
        LEFT JOIN proveedores p ON s.proveedor_id = p.id
        WHERE s.maquina_id = :maquina_id 
        AND s.activo = 1
        ORDER BY s.codigo_serie
    ");

    $stmt->bindParam(':maquina_id', $maquina_id, PDO::PARAM_INT);
    $stmt->execute();
    $series = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Formatear la respuesta
    $response = [
        'success' => true,
        'series' => array_map(function($serie) {
            return [
                'id' => $serie['id'],
                'codigo_serie' => $serie['codigo_serie'],
                'descripcion' => $serie['descripcion'],
                'estado' => $serie['estado'],
                'cliente_id' => $serie['cliente_id'],
                'cliente_nombre' => $serie['cliente_nombre'] ?? '',
                'cliente_rut' => $serie['cliente_rut'] ?? ''
            ];
        }, $series)
    ];

    echo json_encode($response);
    
} catch (Exception $e) {
    error_log("Error en get_series.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}