<?php
/**
 * Generador mejorado de PDFs para rendiciones, órdenes de compra y guías de entrada
 * Usando tFPDF para soporte nativo de UTF-8
 * Versión mejorada con mejor formato visual y soporte para nuevos campos
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$log_file = 'pdf_debug.log';
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - PDF solicitado: tipo=' . ($_GET['tipo'] ?? 'no_definido') . ', id=' . ($_GET['id'] ?? 'no_definido') . "\n", FILE_APPEND);

require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'tfpdf/tfpdf.php';

file_put_contents($log_file, date('Y-m-d H:i:s') . ' - tFPDF cargado correctamente' . "\n", FILE_APPEND);

verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');
file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Sesión verificada: user_id=' . $user_id . ', is_admin=' . ($is_admin ? 'true' : 'false') . "\n", FILE_APPEND);

$tipo = $_GET['tipo'] ?? '';
$id = $_GET['id'] ?? 0;
$save = isset($_GET['save']) ? true : false;
$redirect = isset($_GET['redirect']) ? $_GET['redirect'] : '';

if (empty($tipo) || empty($id)) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Parámetros incorrectos' . "\n", FILE_APPEND);
    die('Parámetros incorrectos');
}

if ($tipo != 'rendicion' && $tipo != 'orden_compra' && $tipo != 'guia_entrada' && $tipo != 'devolucion') {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Tipo de documento no válido: ' . $tipo . "\n", FILE_APPEND);
    die('Tipo de documento no válido');
}

if (!file_exists('temp')) {
    mkdir('temp', 0777, true);
}

$company_name = 'COMERCIAL FIGARO LIMITADA';
$company_address = 'Puerta del Sol 180 Of.02';
$company_rut = '76.080.987-K';
$company_phone = '+562 32212511';
$company_email = 'CONTACTO@FIGAROCAFE.CL';
$company_business = 'Comercializadora de Café y otros Productos Gourmet';
$company_logo = 'assets/img/logo.png';

if (!defined('COMPANY_NAME')) {
    define('COMPANY_NAME', 'Nombre de tu empresa');
}
define('COMPANY_ADDRESS', $company_address);
define('COMPANY_RUT', $company_rut);
define('COMPANY_PHONE', $company_phone);
define('COMPANY_EMAIL', $company_email);
define('COMPANY_BUSINESS', $company_business);
define('COMPANY_LOGO', $company_logo);

class EnhancedPDF extends tFPDF {
    private $logo = null;
    public $titulo;
    public $codigo;
    public $footer_text;
    public $documento;

    function __construct($documento = [], $titulo = '', $codigo = '', $orientation = 'P', $unit = 'mm', $size = 'A4') {
        setlocale(LC_ALL, 'es_ES.UTF-8');
        parent::__construct($orientation, $unit, $size);
        $this->AddFont('DejaVu', '', 'DejaVuSans.ttf', true);
        $this->AddFont('DejaVu', 'B', 'DejaVuSans-Bold.ttf', true);
        $this->SetFont('DejaVu', '', 8);
        if (file_exists(COMPANY_LOGO)) {
            $this->logo = COMPANY_LOGO;
        }
        $this->footer_text = 'Documento generado por ' . APP_NAME . ' - ' . date('d/m/Y H:i:s');
        $this->SetMargins(10, 10, 10);
        $this->documento = $documento;
        $this->titulo = $titulo;
        $this->codigo = $codigo;
        $this->AddPage();
    }

    function RoundedRect($x, $y, $w, $h, $r, $corners = '1234', $style = '') {
        $k = $this->k;
        $hp = $this->h;
        if($style=='F')
            $op='f';
        elseif($style=='FD' || $style=='DF')
            $op='B';
        else
            $op='S';
        $MyArc = 4/3 * (sqrt(2) - 1);
        $this->_out(sprintf('%.2F %.2F m',($x+$r)*$k,($hp-$y)*$k ));

        $xc = $x+$w-$r;
        $yc = $y+$r;
        $this->_out(sprintf('%.2F %.2F l', $xc*$k,($hp-$y)*$k ));

        if (strpos($corners, '2')===false)
            $this->_out(sprintf('%.2F %.2F l', ($x+$w)*$k,($hp-$y)*$k ));
        else
            $this->_Arc($xc + $r*$MyArc, $yc - $r, $xc + $r, $yc - $r*$MyArc, $xc + $r, $yc);

        $xc = $x+$w-$r;
        $yc = $y+$h-$r;
        $this->_out(sprintf('%.2F %.2F l',($x+$w)*$k,($hp-($y+$h))*$k));

        if (strpos($corners, '3')===false)
            $this->_out(sprintf('%.2F %.2F l',($x+$w)*$k,($hp-($y+$h))*$k));
        else
            $this->_Arc($xc + $r, $yc + $r*$MyArc, $xc + $r*$MyArc, $yc + $r, $xc, $yc + $r);

        $xc = $x+$r;
        $yc = $y+$h-$r;
        $this->_out(sprintf('%.2F %.2F l',$xc*$k,($hp-($y+$h))*$k));

        if (strpos($corners, '4')===false)
            $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-($y+$h))*$k));
        else
            $this->_Arc($xc - $r*$MyArc, $yc + $r, $xc - $r, $yc + $r*$MyArc, $xc - $r, $yc);

        $xc = $x+$r ;
        $yc = $y+$r;
        $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-$yc)*$k ));

        if (strpos($corners, '1')===false)
        {
            $this->_out(sprintf('%.2F %.2F l',($x)*$k,($hp-$y)*$k ));
            $this->_out(sprintf('%.2F %.2F l',($x+$r)*$k,($hp-$y)*$k ));
        }
        else
            $this->_Arc($xc - $r, $yc - $r*$MyArc, $xc - $r*$MyArc, $yc - $r, $xc, $yc - $r);
        $this->_out($op);
    }

    function _Arc($x1, $y1, $x2, $y2, $x3, $y3) {
        $h = $this->h;
        $this->_out(sprintf('%.2F %.2F %.2F %.2F %.2F %.2F c ', 
            $x1*$this->k, ($h-$y1)*$this->k,
            $x2*$this->k, ($h-$y2)*$this->k,
            $x3*$this->k, ($h-$y3)*$this->k));
    }

    function Header() {
        // Establecer colores corporativos
        $this->SetDrawColor(30, 136, 229);
        $this->SetFillColor(240, 248, 255);
        
        $y_initial = $this->GetY();
        if ($this->logo) {
            $this->Image($this->logo, (210 - 40) / 2, 10, 40); // Logo más pequeño
            $this->SetY($y_initial + 25); // Ajustar posición después del logo
        }

        // Información de la empresa con formato más compacto
        $this->SetFont('DejaVu', 'B', 10);
        $this->SetTextColor(30, 82, 140);
        $this->Cell(0, 5, COMPANY_NAME, 0, 1, 'C');

        // Información adicional de la empresa más compacta
        $this->SetFont('DejaVu', '', 7);
        $this->SetTextColor(80, 80, 80);
        $this->Cell(0, 3, 'RUT: ' . COMPANY_RUT . ' | ' . COMPANY_ADDRESS, 0, 1, 'C');
        $this->Cell(0, 3, 'Tel: ' . COMPANY_PHONE . ' | Email: ' . COMPANY_EMAIL . ' | Giro: ' . COMPANY_BUSINESS, 0, 1, 'C');
        
        // Espacio en blanco después de la información de la empresa
        $this->Ln(5);
        

        // Código y estado en la misma línea
        $this->SetFont('DejaVu', 'B', 7);
        $this->SetTextColor(255, 255, 255);
        $this->SetFillColor(30, 136, 229);
        
        // Código a la izquierda
        $code_text = 'COD: ' . ($this->codigo ?? 'No especificado');
        $code_width = $this->GetStringWidth($code_text) + 6;
        $this->Cell($code_width, 6, $code_text, 1, 0, 'C', true);
        
        // Título centrado
        $this->SetTextColor(30, 82, 140);
        $this->SetFillColor(255, 255, 255);
        $title_width = $this->GetPageWidth() - 2*$code_width - 20;
        $this->Cell($title_width, 6, $this->titulo, 'B', 0, 'C');
        
        // Estado a la derecha
        $this->SetTextColor(255, 255, 255);
        $estado_color = '#FFC107'; // Color por defecto (amarillo)
        $estado_nombre = 'Pendiente'; // Estado por defecto

        if (isset($this->documento['estado_nombre'])) {
            $estado_nombre = $this->documento['estado_nombre'];
        }
        
        if (isset($this->documento['estado_id'])) {
            switch($this->documento['estado_id']) {
                case 1: // Enviada
                    $estado_color = '#FFC107'; // Amarillo
                    break;
                case 2: // En revisión
                    $estado_color = '#3498DB'; // Azul
                    break;
                case 3: // Aprobada
                    $estado_color = '#28A745'; // Verde
                    break;
                case 4: // Rechazada
                    $estado_color = '#DC3545'; // Rojo
                    break;
                case 5: // Pagada
                    $estado_color = '#6C757D'; // Gris
                    break;
                default:
                    $estado_color = '#6C757D'; // Gris por defecto
            }
        }
        
        // Convertir color hexadecimal a RGB
        if (preg_match('/^#[a-fA-F0-9]{6}$/', $estado_color)) {
            $r = hexdec(substr($estado_color, 1, 2));
            $g = hexdec(substr($estado_color, 3, 2));
            $b = hexdec(substr($estado_color, 5, 2));
        } else {
            // Color por defecto si el formato es inválido
            $r = 108; // #6C
            $g = 117; // #75
            $b = 125; // #7D
        }
        
        $this->SetFillColor($r, $g, $b);
        $this->Cell($code_width, 6, 'ESTADO: ' . strtoupper($estado_nombre), 1, 1, 'C', true);
        
        $this->Ln(2);
    }

    function Footer() {
        $this->SetY(-15);
        $this->SetDrawColor(30, 136, 229); // Azul para la línea
        $this->Line(15, $this->GetY(), 195, $this->GetY());
        $this->Ln(1);
        $this->SetFont('DejaVu', '', 8);
        $this->SetTextColor(80, 80, 80); // Gris oscuro para el pie de página
        $this->Cell(0, 10, $this->footer_text, 0, 0, 'C');
        $this->Cell(0, 10, 'Página ' . $this->PageNo() . ' de {nb}', 0, 0, 'R');
    }

    function AddSection($title, $border = 0, $fill = false) {
        $this->SetFont('DejaVu', 'B', 10);
        $this->SetFillColor(30, 136, 229); // Azul para el fondo
        $this->SetTextColor(255, 255, 255); // Texto blanco
        $this->Cell(0, 7, $title, 1, 1, 'L', true);
        $this->SetFont('DejaVu', '', 9);
        $this->SetTextColor(0, 0, 0); // Restaurar color de texto
        $this->Ln(2);
    }

    function FormatoMoneda($monto) {
        return '$' . number_format($monto, 0, ',', '.');
    }

    function Cell($w, $h = 0, $txt = '', $border = 0, $ln = 0, $align = '', $fill = false, $link = '') {
        parent::Cell($w, $h, $txt, $border, $ln, $align, $fill, $link);
    }

    function MultiCell($w, $h, $txt, $border = 0, $align = 'J', $fill = false) {
        parent::MultiCell($w, $h, $txt, $border, $align, $fill);
    }

    function AddSignatureSection() {
        $this->Ln(15);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(90, 6, '____________________________', 0, 0, 'C');
        $this->Cell(90, 6, '____________________________', 0, 1, 'C');
        $this->Cell(90, 6, 'Firma Emisor', 0, 0, 'C');
        $this->Cell(90, 6, 'Firma Aprobador', 0, 1, 'C');
    }
    
    // Función para crear tabla con colores alternos en las filas
    function FancyTable($header, $data, $widths = array(), $aligns = array()) {
        // Colores, ancho de línea y fuente en negrita
        $this->SetFillColor(240, 248, 255); // Azul muy claro para filas alternas
        $this->SetTextColor(0);
        $this->SetDrawColor(30, 136, 229); // Azul para bordes
        $this->SetLineWidth(0.3);
        $this->SetFont('DejaVu', 'B', 8);
        
        // Encabezado
        for($i=0; $i<count($header); $i++) {
            $w = isset($widths[$i]) ? $widths[$i] : ($this->GetPageWidth() - 30) / count($header);
            $align = isset($aligns[$i]) ? $aligns[$i] : 'C';
            $this->SetFillColor(30, 136, 229); // Azul para encabezado
            $this->SetTextColor(255); // Texto blanco
            $this->Cell($w, 7, $header[$i], 1, 0, $align, true);
        }
        $this->Ln();
        
        // Restaurar colores y fuente
        $this->SetFillColor(240, 248, 255); // Azul muy claro para filas alternas
        $this->SetTextColor(0); // Texto negro
        $this->SetFont('DejaVu', '', 8);
        
        // Datos con colores alternos
        $fill = false;
        foreach($data as $row) {
            for($i=0; $i<count($row); $i++) {
                $w = isset($widths[$i]) ? $widths[$i] : ($this->GetPageWidth() - 30) / count($header);
                $align = isset($aligns[$i]) ? $aligns[$i] : 'L';
                $this->Cell($w, 6, $row[$i], 1, 0, $align, $fill);
            }
            $this->Ln();
            $fill = !$fill; // Alternar color
        }
        $this->Cell(array_sum($widths), 0, '', 'T'); // Línea de cierre
        $this->Ln();
    }

    function GenerateRendicionContent($detalles) {
        // Configuración inicial
        $this->SetFont('DejaVu', '', 9);
        
        // Fecha en la parte superior derecha
        $this->Cell(120, 6, '', 0, 0);
        $this->Cell(70, 6, 'Fecha: ' . date('d/m/Y'), 0, 1, 'R');
        
        // Información del usuario que rinde
        $this->SetFont('DejaVu', 'B', 9);
        if (!empty($this->documento['tercero_id'])) {
            $this->Cell(35, 6, 'Rendido por:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['tercero_nombre'] . ' ' . $this->documento['tercero_apellido'], 0, 1, 'L');
            
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(35, 6, 'RUT:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['tercero_rut'], 0, 1, 'L');
            
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(35, 6, 'Cargo:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['tercero_cargo'], 0, 1, 'L');
            
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(35, 6, 'Ingresado por:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['nombre'] . ' ' . $this->documento['apellido'] . ' - ' . $this->documento['rut'], 0, 1, 'L');
        } else {
            $this->Cell(35, 6, 'Rendido por:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['nombre'] . ' ' . $this->documento['apellido'], 0, 1, 'L');
            
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(35, 6, 'RUT:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['rut'], 0, 1, 'L');
            
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(35, 6, 'Cargo:', 0, 0, 'L');
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(155, 6, $this->documento['cargo'], 0, 1, 'L');
        }
        
        $this->Ln(5);

        // Función auxiliar para truncar texto
        function truncateText($text, $maxLength) {
            if (mb_strlen($text) > $maxLength) {
                return mb_substr($text, 0, $maxLength-3) . '...';
            }
            return $text;
        }

        // Agrupar detalles por tipo de documento y categoría
        $grupos = [];
        $totales_tipo = [];
        
        foreach ($detalles as $detalle) {
            $tipo_doc = $detalle['tipo_documento'] ?? 'otro';
            $categoria = truncateText($detalle['categoria_nombre'], 25); // Truncar categoría
            $codigo_categoria = $detalle['codigo_categoria'] ?? '';
            
            $key = $categoria;
            
            if (!isset($grupos[$tipo_doc][$key])) {
                $grupos[$tipo_doc][$key] = [
                    'cantidad' => 0,
                    'monto' => 0,
                    'codigo' => $codigo_categoria
                ];
            }
            
            $grupos[$tipo_doc][$key]['cantidad']++;
            $grupos[$tipo_doc][$key]['monto'] += $detalle['monto'];
            
            if (!isset($totales_tipo[$tipo_doc])) {
                $totales_tipo[$tipo_doc] = 0;
            }
            $totales_tipo[$tipo_doc] += $detalle['monto'];
        }

        // Ordenar tipos de documento
        $orden_tipos = [
            'boleta' => 'BOLETAS',
            'factura' => 'FACTURAS',
            'recibo' => 'RECIBOS',
            'cotizacion' => 'COTIZACIONES',
            'otro' => 'OTROS DOCUMENTOS'
        ];

        $total_general = 0;

        foreach ($orden_tipos as $tipo => $titulo) {
            if (isset($grupos[$tipo])) {
                // Título de la sección
                $this->SetFont('DejaVu', 'B', 10);
                $this->SetFillColor(30, 136, 229);
                $this->SetTextColor(255);
                $this->Cell(0, 7, $titulo, 1, 1, 'C', true);
                $this->SetTextColor(0);
                $this->Ln(2);

                // Configurar anchos de columna específicos para facturas
                if ($tipo == 'factura') {
                    // Calcular el ancho disponible de la página
                    $page_width = $this->GetPageWidth();
                    $margins = 20; // 10mm en cada lado
                    $available_width = $page_width - $margins;

                    $this->SetFont('DejaVu', 'B', 6);
                    // Definir anchos proporcionales que sumen exactamente el ancho disponible
                    $w = array(
                        18,     // Fecha
                        47,     // Categoría
                        20,     // Folio
                        25,     // Monto
                        20,     // Fecha
                        25,     // RUT
                        35      // Proveedor
                    );

                    // Verificar que la suma de anchos no exceda el ancho disponible
                    $total_width = array_sum($w);
                    if ($total_width > $available_width) {
                        $scale = $available_width / $total_width;
                        $w = array_map(function($width) use ($scale) {
                            return floor($width * $scale);
                        }, $w);
                    }

                    $headers = array('Fecha', 'Categoría', 'Folio', 'Monto', 'Fecha', 'RUT', 'Proveedor');
                    $aligns = array('C', 'L', 'C', 'R', 'C', 'C', 'L');
                    
                    // Dibujar encabezados con fondo azul
                    $this->SetFillColor(30, 136, 229);
                    $this->SetTextColor(255);
                    $this->SetX(10); // Asegurar que empezamos desde el margen izquierdo
                    
                    for($i = 0; $i < count($headers); $i++) {
                        $this->Cell($w[$i], 6, $headers[$i], 1, 0, $aligns[$i], true);
                    }
                    $this->Ln();
                    
                    // Restaurar colores y fuente para los datos
                    $this->SetTextColor(0);
                    $this->SetFont('DejaVu', '', 6);
                    $this->SetFillColor(245, 245, 245);
                    
                    // Imprimir datos con filas alternadas
                    $fill = false;
                    foreach ($detalles as $detalle) {
                        if ($detalle['tipo_documento'] == 'factura') {
                            $this->SetX(10); // Asegurar alineación con el margen
                            $this->Cell($w[0], 6, date('d/m/Y', strtotime($detalle['fecha_documento'])), 1, 0, 'C', $fill);
                            $this->Cell($w[1], 6, ' '.truncateText($detalle['categoria_nombre'], 30), 1, 0, 'L', $fill);
                            $this->Cell($w[2], 6, $detalle['folio_factura'], 1, 0, 'C', $fill);
                            $this->Cell($w[3], 6, $this->FormatoMoneda($detalle['monto']), 1, 0, 'R', $fill);
                            $this->Cell($w[4], 6, date('d/m/Y', strtotime($detalle['fecha_documento'])), 1, 0, 'C', $fill);
                            $this->Cell($w[5], 6, $detalle['rut_proveedor'], 1, 0, 'C', $fill);
                            $this->Cell($w[6], 6, ' '.truncateText($detalle['proveedor_factura'], 20), 1, 1, 'L', $fill);
                            $fill = !$fill;
                        }
                    }
                    
                    // Agregar total de facturas
                    $this->SetFont('DejaVu', 'B', 7);
                    $this->SetX(10);
                    $this->Cell(array_sum($w) - 35, 7, 'Total FACTURAS', 1, 0, 'R', true);
                    $this->Cell(35, 7, $this->FormatoMoneda($totales_tipo[$tipo]), 1, 1, 'R', true);
                    
                } else {
                    // Configuración para otros tipos de documentos (boletas, etc.)
                    $this->SetFont('DejaVu', 'B', 6);
                    
                    // Calcular anchos disponibles
                    $page_width = $this->GetPageWidth();
                    $margins = 20;
                    $available_width = $page_width - $margins;
                    
                    // Ajustar anchos para coincidir con la imagen
                    $w = array(
                        35,     // Código
                        105,    // Categoría
                        20,     // Cantidad
                        30      // Monto
                    );
                    
                    // Verificar y ajustar anchos si es necesario
                    $total_width = array_sum($w);
                    if ($total_width > $available_width) {
                        $scale = $available_width / $total_width;
                        $w = array_map(function($width) use ($scale) {
                            return floor($width * $scale);
                        }, $w);
                    }
                    
                    // Encabezados alineados con la imagen
                    $headers = array('Código', 'Categoría', 'Cantidad', 'Monto');
                    $aligns = array('L', 'L', 'C', 'R');
                    
                    // Dibujar encabezados con fondo azul
                    $this->SetFillColor(30, 136, 229);
                    $this->SetTextColor(255);
                    $this->SetFont('DejaVu', 'B', 6);
                    $this->SetX(10);
                    
                    // Solo mostramos los encabezados de columna, no el título de la sección
                    for($i = 0; $i < count($headers); $i++) {
                        $this->Cell($w[$i], 6, $headers[$i], 1, 0, $aligns[$i], true);
                    }
                    $this->Ln();
                    
                    // Restaurar colores para los datos
                    $this->SetTextColor(0);
                    $this->SetFont('DejaVu', '', 6);
                    $this->SetFillColor(245, 245, 245);
                    
                    // Imprimir datos con filas alternadas
                    $fill = false;
                    foreach ($grupos[$tipo] as $categoria => $datos) {
                        $this->SetX(10);
                        $this->Cell($w[0], 6, $datos['codigo'] ?? '-', 1, 0, 'L', $fill);
                        $this->Cell($w[1], 6, ' ' . truncateText($categoria, 60), 1, 0, 'L', $fill);
                        $this->Cell($w[2], 6, $datos['cantidad'], 1, 0, 'C', $fill);
                        $this->Cell($w[3], 6, $this->FormatoMoneda($datos['monto']), 1, 1, 'R', $fill);
                        $fill = !$fill;
                    }
                    
                    // Agregar total con el mismo formato que en la imagen
                    $this->SetFont('DejaVu', 'B', 7);
                    $this->SetX(10);
                    $this->SetFillColor(220, 220, 220);
                    $total_width = array_sum($w);
                    $this->Cell($total_width - 30, 7, 'Total ' . $titulo, 1, 0, 'R', true);
                    $this->Cell(30, 7, $this->FormatoMoneda($totales_tipo[$tipo]), 1, 1, 'R', true);
                }
                
                $this->Ln(3);
            }
        }

        // Espacio antes del resumen final
        $this->Ln(10);

        // Resumen final alineado a la derecha
        $this->SetFont('DejaVu', 'B', 9);
        $ancho_total = 70;
        $pos_x = $this->GetPageWidth() - $ancho_total - 10;

        // Fondo gris claro para el resumen
        $this->SetFillColor(245, 245, 245);


        // FONDO A RENDIR
        $this->SetX($pos_x);
        $this->Cell($ancho_total/2, 8, 'FONDO A RENDIR', 1, 0, 'L', true);
        $this->Cell($ancho_total/2, 8, $this->FormatoMoneda($this->documento['fondo_a_rendir']), 1, 1, 'R', true);

        // TOTAL RENDIDO
        $saldo_pendiente = $this->documento['fondo_a_rendir'] - $total_general;
        $this->SetX($pos_x);
        $this->SetTextColor($saldo_pendiente >= 0 ? 40 : 220, $saldo_pendiente >= 0 ? 167 : 53, $saldo_pendiente >= 0 ? 69 : 69);
        $this->Cell($ancho_total/2, 8, 'TOTAL RENDIDO', 1, 0, 'L', true);
        $this->Cell($ancho_total/2, 8, $this->FormatoMoneda($saldo_pendiente), 1, 1, 'R', true);
        $this->SetTextColor(0, 0, 0);
        
        // SALDO PENDIENTE
        $this->SetX($pos_x);
        $this->Cell($ancho_total/2, 8, 'SALDO PENDIENTE', 1, 0, 'L', true);
        $this->Cell($ancho_total/2, 8, $this->FormatoMoneda($total_general), 1, 1, 'R', true);

        // Espacio antes de la información del aprobador
        $this->Ln(15);

        // Información del aprobador si está aprobada
        if ($this->documento['estado_id'] == 3 && !empty($this->documento['revisor_nombre'])) {
            $this->SetFont('DejaVu', '', 9);
            $this->Cell(0, 6, 'Aprobado por: ' . $this->documento['revisor_nombre'] . ' ' . $this->documento['revisor_apellido'], 0, 1, 'R');
            $this->Cell(0, 6, 'Fecha de aprobación: ' . date('d/m/Y H:i', strtotime($this->documento['fecha_revision'])), 0, 1, 'R');
        }
    }

    function GenerateDevolucionContent($detalles) {
        // Configuración inicial
        $this->SetFont('DejaVu', '', 9);
        $this->SetDrawColor(30, 136, 229);
        $this->SetFillColor(240, 248, 255);
        
        // Encabezado con información principal
        $this->Cell(0, 10, '', 0, 1); // Espacio superior
        
        // Primera fila
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'Solicitante:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(70, 6, $this->documento['nombre'] . ' ' . $this->documento['apellido'], 0, 0);
        
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'Fecha:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(60, 6, date('d/m/Y', strtotime($this->documento['fecha'])), 0, 1);
        
        // Segunda fila
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'RUT:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(70, 6, $this->documento['rut'], 0, 0);
        
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'Canal:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(60, 6, $this->documento['origen_nombre'], 0, 1);
        
        // Tercera fila
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'Cargo:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(70, 6, $this->documento['cargo'], 0, 0);
        
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(30, 6, 'Monto:', 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->Cell(60, 6, $this->FormatoMoneda($this->documento['monto_total']), 0, 1);
        
        $this->Ln(5);
        
        // Sección Motivo
        $this->SetFillColor(30, 136, 229);
        $this->SetTextColor(255, 255, 255);
        $this->SetFont('DejaVu', 'B', 9);
        $this->Cell(0, 6, ' Motivo:', 1, 1, 'L', true);
        
        $this->SetTextColor(0, 0, 0);
        $this->SetFont('DejaVu', '', 9);
        $this->MultiCell(0, 6, $this->documento['descripcion'], 1);
        
        $this->Ln(5);
        
        // Sección Datos Bancarios (antes Observaciones)
        if (!empty($this->documento['observaciones'])) {
            $this->SetFillColor(30, 136, 229);
            $this->SetTextColor(255, 255, 255);
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(0, 6, ' Datos Bancarios:', 1, 1, 'L', true);
            
            $this->SetTextColor(0, 0, 0);
            $this->SetFont('DejaVu', '', 9);
            $this->MultiCell(0, 6, $this->documento['observaciones'], 1);
            
            $this->Ln(5);
        }
        
        // Sección Documentos
        if (!empty($detalles)) {
            $this->SetFillColor(30, 136, 229);
            $this->SetTextColor(255, 255, 255);
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(0, 6, ' Documentos Adjuntos:', 1, 1, 'L', true);
            
            $this->SetTextColor(0, 0, 0);
            $this->SetFont('DejaVu', '', 9);
            
            // Encabezados de la tabla
            $this->SetFillColor(240, 248, 255);
            $w = array(15, 120, 55);
            $this->Cell($w[0], 6, '#', 1, 0, 'C', true);
            $this->Cell($w[1], 6, 'Archivo', 1, 0, 'L', true);
            $this->Cell($w[2], 6, 'Fecha', 1, 1, 'C', true);
            
            // Datos de la tabla
            $this->SetFont('DejaVu', '', 8);
            foreach ($detalles as $index => $soporte) {
                $this->Cell($w[0], 6, $index + 1, 1, 0, 'C');
                $this->Cell($w[1], 6, $soporte['nombre_archivo'], 1, 0, 'L');
                $this->Cell($w[2], 6, date('d/m/Y H:i', strtotime($soporte['fecha_subida'])), 1, 1, 'C');
            }
        }
        
        $this->Ln(5);
        
        // Sección de aprobación
        if ($this->documento['estado_id'] == 3 && !empty($this->documento['revisor_nombre'])) {
            $this->SetFillColor(30, 136, 229);
            $this->SetTextColor(255, 255, 255);
            $this->SetFont('DejaVu', 'B', 9);
            $this->Cell(0, 6, ' Aprobación:', 1, 1, 'L', true);
            
            $this->SetTextColor(0, 0, 0);
            $this->SetFont('DejaVu', '', 9);
            
            $this->Cell(40, 6, 'Aprobado por:', 1, 0);
            $this->Cell(0, 6, $this->documento['revisor_nombre'] . ' ' . $this->documento['revisor_apellido'], 1, 1);
            
            $this->Cell(40, 6, 'Fecha:', 1, 0);
            $this->Cell(0, 6, date('d/m/Y H:i', strtotime($this->documento['fecha_revision'])), 1, 1);
            
            if (!empty($this->documento['comentario_revision'])) {
                $this->Cell(40, 6, 'Comentario:', 1, 0);
                $this->MultiCell(0, 6, $this->documento['comentario_revision'], 1);
            }
        }
        
        $this->Ln(10);
        
        // Sección de firmas
        $this->Cell(95, 6, '_____________________', 0, 0, 'C');
        $this->Cell(95, 6, '_____________________', 0, 1, 'C');
        $this->Cell(95, 6, 'Firma Solicitante', 0, 0, 'C');
        $this->Cell(95, 6, 'Firma Aprobador', 0, 1, 'C');
    }

    // Función auxiliar para calcular número de líneas
    function NbLines($w, $txt) {
        $cw = &$this->CurrentFont['cw'];
        if($w==0)
            $w = $this->w-$this->rMargin-$this->x;
        $wmax = ($w-2*$this->cMargin)*1000/$this->FontSize;
        $s = str_replace("\r",'',$txt);
        $nb = strlen($s);
        if($nb>0 && $s[$nb-1]=="\n")
            $nb--;
        $sep = -1;
        $i = 0;
        $j = 0;
        $l = 0;
        $nl = 1;
        while($i<$nb) {
            $c = $s[$i];
            if($c=="\n") {
                $i++;
                $sep = -1;
                $j = $i;
                $l = 0;
                $nl++;
                continue;
            }
            if($c==' ')
                $sep = $i;
            $l += $cw[$c];
            if($l>$wmax) {
                if($sep==-1) {
                    if($i==$j)
                        $i++;
                }
                else
                    $i = $sep+1;
                $sep = -1;
                $j = $i;
                $l = 0;
                $nl++;
            }
            else
                $i++;
        }
        return $nl;
    }
}

try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Conexión a la base de datos establecida' . "\n", FILE_APPEND);

    if ($tipo == 'devolucion') {
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Intentando generar PDF para devolución ID=' . $id . "\n", FILE_APPEND);
        
        $stmt = $conn->prepare("
            SELECT d.*, 
                   u.nombre, u.apellido, u.rut, u.cargo,
                   COALESCE(o.nombre, 'No especificado') as origen_nombre,
                   e.nombre as estado_nombre,
                   rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
            FROM devoluciones d
            JOIN users u ON d.user_id = u.id
            LEFT JOIN origenes_devolucion o ON d.origen_id = o.id
            LEFT JOIN estados e ON d.estado_id = e.id
            LEFT JOIN users rev ON d.revisor_id = rev.id
            WHERE d.id = :id
        ");
        
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $devolucion = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$devolucion) {
            file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Devolución no encontrada para id=' . $id . "\n", FILE_APPEND);
            die('Documento no encontrado');
        }
        
        // Verificar permisos después de obtener el documento
        if (!$is_admin && $devolucion['user_id'] != $user_id) {
            file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: No tiene permisos para ver este documento. user_id=' . $user_id . ', documento_user_id=' . $devolucion['user_id'] . "\n", FILE_APPEND);
            die('No tiene permisos para ver este documento');
        }
        
        // Obtener archivos de soporte
        $stmt = $conn->prepare("SELECT * FROM devolucion_soportes WHERE devolucion_id = :id ORDER BY fecha_subida");
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $titulo_documento = 'SOLICITUD DE DEVOLUCIÓN';
        $codigo = generarCodigo('DEV', $id);
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Generando PDF con código: ' . $codigo . "\n", FILE_APPEND);
        
        // Asignar valores por defecto si son null
        $devolucion['estado_nombre'] = $devolucion['estado_nombre'] ?? 'Pendiente';
        $devolucion['origen_nombre'] = $devolucion['origen_nombre'] ?? 'No especificado';
        $devolucion['revisor_nombre'] = $devolucion['revisor_nombre'] ?? 'No asignado';
        $devolucion['revisor_apellido'] = $devolucion['revisor_apellido'] ?? '';
        
        $pdf = new EnhancedPDF($devolucion, $titulo_documento, $codigo);
        $pdf->AliasNbPages();
        $pdf->SetFont('DejaVu', '', 9);
        $pdf->GenerateDevolucionContent($detalles);

        $file_name = $codigo . '.pdf';
        if ($save) {
            $file_path = 'temp/' . $file_name;
            $pdf->Output($file_path, 'F');
            if (!empty($redirect)) {
                header('Location: ' . $redirect);
                exit;
            }
            echo "PDF guardado exitosamente en: " . $file_path;
        } else {
            $pdf->Output($file_name, 'I');
        }
        exit;
    } elseif ($tipo == 'rendicion') {
        $stmt = $conn->prepare("
            SELECT r.*, 
                   u.nombre, u.apellido, u.rut, u.cargo,
                   t.nombre as tercero_nombre, t.apellido as tercero_apellido, t.rut as tercero_rut, t.cargo as tercero_cargo,
                   rev.nombre as revisor_nombre, rev.apellido as revisor_apellido
            FROM rendiciones r
            JOIN users u ON r.user_id = u.id
            LEFT JOIN users t ON r.tercero_id = t.id
            LEFT JOIN users rev ON r.revisor_id = rev.id
            WHERE r.id = :id " . (!$is_admin ? "AND (r.user_id = :user_id OR r.revisor_id = :user_id2)" : "")
        );
        
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        if (!$is_admin) {
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->bindParam(':user_id2', $user_id, PDO::PARAM_INT);
        }
        
        $stmt->execute();
        $documento = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$documento) {
            file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Rendición no encontrada para id=' . $id . "\n", FILE_APPEND);
            die('Documento no encontrado');
        }
        
        // Verificar permisos después de obtener el documento
        if (!$is_admin && $documento['user_id'] != $user_id && $documento['revisor_id'] != $user_id) {
            file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: No tiene permisos para ver este documento. user_id=' . $user_id . ', documento_user_id=' . $documento['user_id'] . "\n", FILE_APPEND);
            die('No tiene permisos para ver este documento');
        }
        
        // Obtener detalles de la rendición
        $stmt = $conn->prepare("
            SELECT rc.*, cg.nombre as categoria_nombre, cg.codigo as codigo_categoria
            FROM rendicion_categoria rc
            LEFT JOIN categorias_gastos cg ON rc.categoria_id = cg.id
            WHERE rc.rendicion_id = :id
        ");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $titulo_documento = 'RENDICIÓN DE GASTOS';
        $codigo = generarCodigo('REN', $id);
        
        $pdf = new EnhancedPDF($documento, $titulo_documento, $codigo);
        $pdf->AliasNbPages();
        $pdf->SetFont('DejaVu', '', 9);
        $pdf->GenerateRendicionContent($detalles);
        
        $file_name = $codigo . '.pdf';
        if ($save) {
            $file_path = 'temp/' . $file_name;
            $pdf->Output($file_path, 'F');
            if (!empty($redirect)) {
                header('Location: ' . $redirect);
                exit;
            }
            echo "PDF guardado exitosamente en: " . $file_path;
        } else {
            $pdf->Output($file_name, 'I');
        }
        exit;
    } elseif ($tipo == 'orden_compra') {
        $stmt = $conn->prepare("
            SELECT oc.*, u.nombre, u.apellido, u.rut, u.cargo,
                   p.direccion, p.telefono, p.email, p.contacto, p.giro
            FROM ordenes_compra oc
            JOIN users u ON oc.user_id = u.id
            LEFT JOIN proveedores p ON oc.proveedor_id = p.id
            WHERE oc.id = :id
        ");
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Ejecutando consulta para orden_compra con id=' . $id . "\n", FILE_APPEND);
        $tables = ['main' => 'ordenes_compra', 'detail' => 'detalle_orden_compra', 'join_field' => 'orden_compra_id'];
        $titulo_documento = 'ORDEN DE COMPRA';
    } else {
        $stmt = $conn->prepare("
            SELECT ge.*, u.nombre, u.apellido, u.rut as user_rut, u.cargo,
                   e.nombre as estado_nombre
            FROM guias_entrada ge
            LEFT JOIN users u ON ge.user_id = u.id
            LEFT JOIN estados e ON ge.estado_id = e.id
            WHERE ge.id = :id
        ");
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Ejecutando consulta para guia_entrada con id=' . $id . "\n", FILE_APPEND);
        $tables = ['main' => 'guias_entrada', 'detail' => 'detalle_guia_entrada', 'join_field' => 'guia_entrada_id'];
        $titulo_documento = 'GUÍA DE ENTRADA';
    }

    $stmt->execute();
    $documento = $stmt->fetch(PDO::FETCH_ASSOC);
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Documento obtenido: ' . print_r($documento, true) . "\n", FILE_APPEND);

    if (!$documento) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: Documento no encontrado para id=' . $id . "\n", FILE_APPEND);
        die('Documento no encontrado');
    }

    if (!$is_admin && $documento['user_id'] != $user_id) {
        file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Error: No tiene permisos para ver este documento. user_id=' . $user_id . ', documento_user_id=' . $documento['user_id'] . "\n", FILE_APPEND);
        die('No tiene permisos para ver este documento');
    }

    $stmt = $conn->prepare("
        SELECT * FROM {$tables['detail']} 
        WHERE {$tables['join_field']} = :id
    ");
    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $detalles = $stmt->fetchAll(PDO::FETCH_ASSOC);
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Detalles obtenidos para ' . $tipo . ': ' . print_r($detalles, true) . "\n", FILE_APPEND);

    if ($tipo == 'rendicion') {
        foreach ($detalles as &$detalle) {
            $stmt = $conn->prepare("SELECT nombre, codigo FROM categorias_gastos WHERE id = :id");
            $stmt->bindParam(':id', $detalle['categoria_id']);
            $stmt->execute();
            $categoria = $stmt->fetch(PDO::FETCH_ASSOC);
            $detalle['categoria_nombre'] = $categoria ? $categoria['nombre'] : 'Desconocida';
            $detalle['codigo_categoria'] = $categoria ? $categoria['codigo'] : '';
            $detalle['detalle'] = !empty($detalle['detalle']) ? $detalle['detalle'] : 'No especificado';
        }
        unset($detalle); // Liberar la referencia
    }

    $prefijo = ($tipo == 'rendicion') ? 'REN' : (($tipo == 'orden_compra') ? 'OC' : 'GE');
    $codigo = generarCodigo($prefijo, $id);
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - Código generado: ' . $codigo . "\n", FILE_APPEND);

    $pdf = new EnhancedPDF($documento, $titulo_documento, $codigo);
    $pdf->AliasNbPages();
    $pdf->SetFont('DejaVu', '', 9);

    if ($tipo == 'rendicion') {
        $pdf->AddSection('DESCRIPCIÓN', 0, true);
        $pdf->SetFont('DejaVu', '', 9);
        $pdf->MultiCell(0, 5, $documento['descripcion'] ?? 'No especificada');
        $pdf->Ln(3);
        
        $pdf->GenerateRendicionContent($detalles);
    } elseif ($tipo == 'orden_compra') {
        // Configuración inicial más compacta
        $pdf->SetFont('DejaVu', '', 7);
        $pdf->SetMargins(10, 10, 10);
        
        // Información en dos columnas más compactas
        $current_y = $pdf->GetY();
        $left_x = $pdf->GetX();
        $page_width = $pdf->GetPageWidth();
        $col_width = ($page_width - 20) / 2 - 1;
        
        // Cuadro izquierdo - Proveedor
        $pdf->SetFillColor(240, 248, 255);
        $pdf->SetDrawColor(30, 136, 229);
        $pdf->SetFont('DejaVu', 'B', 6); // Reducido a 6
        
        // Altura inicial para ambas columnas
        $initial_y = $pdf->GetY();
        
        // Calcular altura máxima necesaria para ambas columnas
        $left_height = 0;
        $right_height = 0;
        
        // Función auxiliar para truncar texto
        function truncateText($text, $maxLength = 35) {
            return mb_strlen($text) > $maxLength ? mb_substr($text, 0, $maxLength-3) . '...' : $text;
        }
        
        // Datos del proveedor
        $info_proveedor = [
            ['Razón Social:', truncateText($documento['proveedor'], 40)],
            ['RUT:', $documento['rut_proveedor']],
            ['Dirección:', truncateText($documento['direccion'] ?? '', 40)],
            ['Contacto:', truncateText($documento['contacto'] ?? '', 30)],
            ['Email:', truncateText($documento['email'] ?? '', 30)],
            ['Teléfono:', truncateText($documento['telefono'] ?? '', 20)]
        ];
        
        // Datos de la orden
        $info_orden = [
            ['N° Orden:', $documento['numero_documento']],
            ['Fecha:', date('d/m/Y', strtotime($documento['fecha']))],
            ['Solicitante:', truncateText($documento['nombre'] . ' ' . $documento['apellido'], 30)],
            ['Área:', truncateText($documento['area_solicitud'] ?? '', 30)],
            ['Despacho:', truncateText($documento['direccion_despacho'] ?? '', 40)],
            ['Entrega:', truncateText($documento['tiempo_entrega'] ?? '', 30)]
        ];
        
        // Dibujar los encabezados con bordes completos
        $pdf->Cell($col_width, 4, 'INFORMACIÓN DEL PROVEEDOR', 1, 0, 'L', true);
        $pdf->SetX($left_x + $col_width + 2);
        $pdf->Cell($col_width, 4, 'INFORMACIÓN DE LA ORDEN', 1, 1, 'L', true);
        
        // Guardar posición después de los encabezados
        $content_y = $pdf->GetY();
        
        // Imprimir datos del proveedor
        $pdf->SetFont('DejaVu', '', 6); // Reducido a 6
        foreach ($info_proveedor as $info) {
            if (!empty($info[1])) {
                $pdf->SetX($left_x);
                $pdf->Cell(20, 4, $info[0], 'L', 0);
                $pdf->Cell($col_width - 20, 4, '  ' . $info[1], 'R', 1);
            }
        }
        
        // Volver a la posición inicial para datos de la orden
        $pdf->SetY($content_y);
        foreach ($info_orden as $info) {
            if (!empty($info[1])) {
                $pdf->SetX($left_x + $col_width + 2);
                $pdf->Cell(20, 4, $info[0], 'L', 0);
                $pdf->Cell($col_width - 20, 4, '  ' . $info[1], 'R', 1);
            }
        }
        
        // Dibujar los bordes exteriores
        $pdf->Rect($left_x, $content_y, $col_width, $max_height);
        $pdf->Rect($left_x + $col_width + 2, $content_y, $col_width, $max_height);
        
        // Ajustar la posición Y al final del contenido más alto
        $pdf->SetY($content_y + $max_height);
        
        $pdf->Ln(4);
        
        // Tabla de productos
        $pdf->SetFillColor(30, 136, 229);
        $pdf->SetTextColor(255, 255, 255);
        $pdf->SetFont('DejaVu', 'B', 6); // Reducido tamaño de fuente
        $pdf->Cell(0, 4, 'DETALLE DE PRODUCTOS', 1, 1, 'L', true);
        
        // Restablecer colores
        $pdf->SetTextColor(0, 0, 0);
        $pdf->SetFillColor(240, 248, 255);
        
        // Anchos de columna reajustados y reducidos
        $w = array(5, 55, 10, 20, 10, 20, 20, 20, 20);
        
        // Encabezados con padding interno y fuente más pequeña
        $headers = array('#', 'Producto', 'Cant.', 'Precio', 'Desc.', 'P.Final', 'Total', 'IVA', 'T.Bruto');
        $pdf->SetFont('DejaVu', 'B', 6);
        
        // Dibujar encabezados con bordes completos y más espacio
        foreach(array_combine($w, $headers) as $width => $header) {
            $pdf->Cell($width, 6, '  ' . $header . '  ', 1, 0, 'C', true);
        }
        $pdf->Ln();
        
        // Datos de productos con padding y control de longitud
        $pdf->SetFont('DejaVu', '', 6); // Fuente más pequeña para los datos
        
        $iva = 0.19;
        $subtotal_neto = 0;
        $total_iva = 0;
        $total_bruto = 0;
        
        foreach ($detalles as $index => $detalle) {
            $precio_unitario = $detalle['precio_unitario'];
            $cantidad = $detalle['cantidad'];
            $descuento = isset($detalle['descuento']) ? $detalle['descuento'] : 0;
            $precio_final = $precio_unitario * (1 - ($descuento / 100));
            $total_linea = $precio_final * $cantidad;
            $iva_linea = $total_linea * $iva;
            $total_bruto_linea = $total_linea + $iva_linea;
            
            $subtotal_neto += $total_linea;
            $total_iva += $iva_linea;
            $total_bruto += $total_bruto_linea;
            
            $fill = ($index % 2 == 0);
            
            // Truncar y añadir puntos suspensivos si el texto es muy largo
            $producto_texto = mb_strlen($detalle['producto']) > 25 ? 
                            mb_substr($detalle['producto'], 0, 22) . '...' : 
                            $detalle['producto'];
            
            // Añadir más padding y reducir contenido
            $pdf->Cell($w[0], 6, ' ' . ($index + 1) . ' ', 1, 0, 'C', $fill);
            $pdf->Cell($w[1], 6, '  ' . $producto_texto . '  ', 1, 0, 'L', $fill);
            $pdf->Cell($w[2], 6, ' ' . $cantidad . ' ', 1, 0, 'R', $fill);
            $pdf->Cell($w[3], 6, '  ' . $pdf->FormatoMoneda($precio_unitario) . '  ', 1, 0, 'R', $fill);
            $pdf->Cell($w[4], 6, ' ' . $descuento . ' ', 1, 0, 'R', $fill);
            $pdf->Cell($w[5], 6, '  ' . $pdf->FormatoMoneda($precio_final) . '  ', 1, 0, 'R', $fill);
            $pdf->Cell($w[6], 6, '  ' . $pdf->FormatoMoneda($total_linea) . '  ', 1, 0, 'R', $fill);
            $pdf->Cell($w[7], 6, '  ' . $pdf->FormatoMoneda($iva_linea) . '  ', 1, 0, 'R', $fill);
            $pdf->Cell($w[8], 6, '  ' . $pdf->FormatoMoneda($total_bruto_linea) . '  ', 1, 0, 'R', $fill);
            $pdf->Ln();
        }

        // Sección de totales con fuente más pequeña y más padding
        $pdf->SetFillColor(245, 245, 245);
        $width_before_totals = array_sum(array_slice($w, 0, 6));
        
        // Subtotal Neto
        $pdf->Cell($width_before_totals, 6, '', 0, 0);
        $pdf->Cell($w[6], 6, '  Subtotal Neto:  ', 1, 0, 'R', true);
        $pdf->Cell($w[7], 6, '', 1, 0, 'R', true);
        $pdf->Cell($w[8], 6, '  ' . $pdf->FormatoMoneda($subtotal_neto) . '  ', 1, 1, 'R', true);
        
        // IVA
        $pdf->Cell($width_before_totals, 6, '', 0, 0);
        $pdf->Cell($w[6], 6, '  IVA (19%):  ', 1, 0, 'R', true);
        $pdf->Cell($w[7], 6, '  ' . $pdf->FormatoMoneda($total_iva) . '  ', 1, 0, 'R', true);
        $pdf->Cell($w[8], 6, '', 1, 1, 'R', true);
        
        // Total Final con fuente un poco más grande para destacar
        $pdf->SetFont('DejaVu', 'B', 7);
        $pdf->Cell($width_before_totals, 6, '', 0, 0);
        $pdf->Cell($w[6], 6, '  TOTAL:  ', 1, 0, 'R', true);
        $pdf->Cell($w[7], 6, '', 1, 0, 'R', true);
        $pdf->Cell($w[8], 6, '  ' . $pdf->FormatoMoneda($total_bruto) . '  ', 1, 1, 'R', true);
        
        // Información adicional en dos columnas más compacta
        $pdf->Ln(3);
        $pdf->SetFont('DejaVu', '', 7);
        
        // Primera columna
        $y_start = $pdf->GetY();
        $x_start = $pdf->GetX();
        
        if (!empty($documento['condiciones_pago'])) {
            $pdf->SetX($x_start);
            $pdf->Cell(20, 4, 'Condiciones:', 0, 0);
            $pdf->MultiCell($col_width - 20, 4, $documento['condiciones_pago'], 0, 'L');
        }
        
        // Segunda columna
        if (!empty($documento['descripcion'])) {
            $pdf->SetXY($x_start + $col_width + 2, $y_start);
            $pdf->Cell(25, 4, 'Observaciones:', 0, 0);
            $pdf->MultiCell($col_width - 25, 4, $documento['descripcion'], 0, 'L');
        }
        
        // Firmas al final de la página
        $pdf->SetY(-40);
        $pdf->SetFont('DejaVu', '', 7);
        
        // Líneas de firma
        $pdf->Cell($col_width, 0.2, '', 'T', 0, 'C');
        $pdf->Cell(2);
        $pdf->Cell($col_width, 0.2, '', 'T', 1, 'C');
        
        // Nombres
        $pdf->Cell($col_width, 4, $documento['nombre'] . ' ' . $documento['apellido'], 0, 0, 'C');
        $pdf->Cell(2);
        $pdf->Cell($col_width, 4, ($documento['estado_id'] == 3 || $documento['estado_id'] == 5) ? 
            $documento['revisor_nombre'] ?? 'Aprobador' : 'Aprobador', 0, 1, 'C');
        
        // Cargos
        $pdf->SetFont('DejaVu', '', 6);
        $pdf->Cell($col_width, 3, $documento['cargo'], 0, 0, 'C');
        $pdf->Cell(2);
        $pdf->Cell($col_width, 3, 'Firma Autorizada', 0, 1, 'C');
    } else {
        // Código para guía de entrada (similar a orden de compra, no detallado aquí)
    }

    $file_name = $codigo . '.pdf';
    if ($save) {
        $file_path = 'temp/' . $file_name;
        $pdf->Output($file_path, 'F');
        if (!empty($redirect)) {
            header('Location: ' . $redirect);
            exit;
        }
        echo "PDF guardado exitosamente en: " . $file_path;
    } else {
        $pdf->Output($file_name, 'I');
    }
} catch (PDOException $e) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - ERROR PDO: ' . $e->getMessage() . "\n", FILE_APPEND);
    die('Error de conexión a la base de datos: ' . $e->getMessage());
} catch (Exception $e) {
    file_put_contents($log_file, date('Y-m-d H:i:s') . ' - ERROR General: ' . $e->getMessage() . "\n", FILE_APPEND);
    die('Error general: ' . $e->getMessage());
}