<?php
/**
 * Script de diagnóstico para identificar problemas con la actualización de stock
 * 
 * Este script verifica paso a paso cada componente necesario para que
 * el actualizador de stock funcione correctamente.
 */

// Mostrar todos los errores para diagnóstico
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<pre style='font-family: monospace; background-color: #f5f5f5; padding: 15px; border: 1px solid #ddd;'>";
echo "=== DIAGNÓSTICO DEL SISTEMA DE ACTUALIZACIÓN DE STOCK ===\n\n";

// 1. Verificar versión de PHP
echo "1. Verificando versión de PHP...\n";
echo "   Versión actual: " . phpversion() . "\n";
if (version_compare(phpversion(), '7.0.0', '<')) {
    echo "   ❌ ERROR: Se requiere PHP 7.0 o superior\n";
} else {
    echo "   ✅ OK: Versión de PHP compatible\n";
}
echo "\n";

// 2. Verificar extensiones requeridas
echo "2. Verificando extensiones requeridas...\n";
$required_extensions = ['curl', 'simplexml', 'mysqli', 'pdo', 'pdo_mysql'];
foreach ($required_extensions as $ext) {
    if (extension_loaded($ext)) {
        echo "   ✅ Extensión $ext: Cargada\n";
    } else {
        echo "   ❌ ERROR: Extensión $ext: No disponible\n";
    }
}
echo "\n";

// 3. Verificar permisos de directorio y archivo
echo "3. Verificando permisos de archivos y directorios...\n";

// Directorio actual
$current_dir = __DIR__;
echo "   Directorio de trabajo: $current_dir\n";

if (is_writable($current_dir)) {
    echo "   ✅ OK: El directorio actual tiene permisos de escritura\n";
} else {
    echo "   ❌ ERROR: El directorio actual NO tiene permisos de escritura\n";
}

// Verificar o crear directorio de logs
$logs_dir = $current_dir . "/logs";
if (!file_exists($logs_dir)) {
    echo "   Directorio de logs no existe, intentando crear...\n";
    if (mkdir($logs_dir, 0755, true)) {
        echo "   ✅ OK: Directorio de logs creado correctamente\n";
    } else {
        echo "   ❌ ERROR: No se pudo crear el directorio de logs\n";
    }
} else {
    echo "   Directorio de logs ya existe\n";
    if (is_writable($logs_dir)) {
        echo "   ✅ OK: El directorio de logs tiene permisos de escritura\n";
    } else {
        echo "   ❌ ERROR: El directorio de logs NO tiene permisos de escritura\n";
    }
}

// Verificar archivos principales
$main_script = $current_dir . "/actualizar_stock.php";
if (file_exists($main_script)) {
    echo "   ✅ Script principal encontrado: $main_script\n";
    
    if (is_readable($main_script)) {
        echo "   ✅ OK: El script principal tiene permisos de lectura\n";
    } else {
        echo "   ❌ ERROR: El script principal NO tiene permisos de lectura\n";
    }
} else {
    echo "   ❌ ERROR: Script principal NO encontrado\n";
    echo "   ⚠️ Archivos en el directorio actual:\n";
    $files = scandir($current_dir);
    foreach ($files as $file) {
        if ($file != "." && $file != ".." && is_file($current_dir . "/" . $file)) {
            echo "     - $file\n";
        }
    }
}
echo "\n";

// 4. Intentar crear un archivo de log de prueba
echo "4. Intentando crear un archivo de log de prueba...\n";
$test_log = $logs_dir . "/diagnostico_" . date("Y-m-d_H-i-s") . ".log";
if (file_put_contents($test_log, "Prueba de escritura: " . date("Y-m-d H:i:s") . "\n")) {
    echo "   ✅ OK: Archivo de log creado exitosamente: $test_log\n";
} else {
    echo "   ❌ ERROR: No se pudo crear el archivo de log de prueba\n";
}
echo "\n";

// 5. Verificar conexión a la base de datos
echo "5. Verificando conexión a la base de datos...\n";

// NOTA: Establecer las credenciales correctas aquí
$db_host = 'localhost';
$db_user = 'santavictoria_s3rv1c10';
$db_pass = 'd.=[YeZkc}]h';
$db_name = 'santavictoria_t0l1g3';

echo "   Intentando conectar a MySQL ($db_host)...\n";
try {
    $conn = new mysqli($db_host, $db_user, $db_pass);
    
    if ($conn->connect_error) {
        echo "   ❌ ERROR: Fallo en la conexión MySQL: " . $conn->connect_error . "\n";
    } else {
        echo "   ✅ OK: Conexión MySQL establecida\n";
        
        // Verificar la base de datos
        echo "   Verificando base de datos '$db_name'...\n";
        if ($conn->select_db($db_name)) {
            echo "   ✅ OK: Base de datos seleccionada correctamente\n";
            
            // Verificar tabla de repuestos
            $result = $conn->query("SHOW TABLES LIKE 'repuestos'");
            if ($result->num_rows > 0) {
                echo "   ✅ OK: Tabla 'repuestos' encontrada\n";
                
                // Verificar estructura de la tabla
                $result = $conn->query("DESCRIBE repuestos");
                if ($result) {
                    $fields = [];
                    while ($row = $result->fetch_assoc()) {
                        $fields[] = $row['Field'];
                    }
                    
                    echo "   Campos encontrados en la tabla 'repuestos':\n";
                    echo "     " . implode(", ", $fields) . "\n";
                    
                    // Verificar campos requeridos
                    $required_fields = ['id', 'codigo', 'stock', 'ubicacion', 'activo'];
                    $missing_fields = [];
                    
                    foreach ($required_fields as $field) {
                        if (!in_array($field, $fields)) {
                            $missing_fields[] = $field;
                        }
                    }
                    
                    if (count($missing_fields) > 0) {
                        echo "   ❌ ERROR: Campos faltantes en la tabla 'repuestos': " . implode(", ", $missing_fields) . "\n";
                        
                        // Si falta el campo ubicacion, mostrar cómo crearlo
                        if (in_array('ubicacion', $missing_fields)) {
                            echo "   Para crear el campo ubicacion, ejecuta:\n";
                            echo "   ALTER TABLE repuestos ADD COLUMN ubicacion VARCHAR(100) DEFAULT NULL;\n";
                        }
                    } else {
                        echo "   ✅ OK: Todos los campos requeridos están presentes\n";
                    }
                    
                    // Contar registros en la tabla
                    $result = $conn->query("SELECT COUNT(*) AS total FROM repuestos WHERE activo = 1");
                    if ($result) {
                        $row = $result->fetch_assoc();
                        echo "   Registros activos en la tabla: " . $row['total'] . "\n";
                    }
                } else {
                    echo "   ❌ ERROR: No se pudo obtener la estructura de la tabla 'repuestos'\n";
                }
            } else {
                echo "   ❌ ERROR: Tabla 'repuestos' NO encontrada\n";
            }
        } else {
            echo "   ❌ ERROR: Base de datos '$db_name' NO encontrada\n";
        }
        
        // Cerrar conexión
        $conn->close();
    }
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n";
}
echo "\n";

// 6. Verificar conexión a Softland
echo "6. Verificando conexión a Softland...\n";

// Configuración de conexión a Softland
$softland_api_url = 'http://201.236.190.215/ECOMMERCE/wsproducto.asmx';
$softland_username = 'STORE';
$softland_password = 'softland';
$softland_empresa = 'FIGARO';

echo "   URL del API: $softland_api_url\n";
echo "   Realizando prueba de conexión...\n";

// Crear solicitud SOAP simple para probar la conexión
$xml_request = '<?xml version="1.0" encoding="utf-8"?>
<soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
               xmlns:xsd="http://www.w3.org/2001/XMLSchema"
               xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
  <soap:Header>
    <AuthHeader xmlns="http://softland.cl/">
      <Username>' . $softland_username . '</Username>
      <Password>' . $softland_password . '</Password>
    </AuthHeader>
  </soap:Header>
  <soap:Body>
    <ObtieneBodegas xmlns="http://softland.cl/">
      <Empresa>' . $softland_empresa . '</Empresa>
      <Token></Token>
    </ObtieneBodegas>
  </soap:Body>
</soap:Envelope>';

try {
    // Configurar cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $softland_api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $xml_request);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: text/xml; charset=utf-8',
        'SOAPAction: "http://softland.cl/ObtieneBodegas"',
        'Content-Length: ' . strlen($xml_request)
    ));
    
    // Ejecutar
    $response = curl_exec($ch);
    
    // Verificar errores
    if (curl_errno($ch)) {
        echo "   ❌ ERROR: Error en la solicitud cURL: " . curl_error($ch) . "\n";
    } else {
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ($http_code == 200) {
            echo "   ✅ OK: Conexión a Softland establecida (HTTP 200)\n";
            
            // Intentar parsear la respuesta XML
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($response);
            
            if ($xml === false) {
                echo "   ⚠️ ADVERTENCIA: La respuesta no es un XML válido\n";
            } else {
                echo "   ✅ OK: Respuesta XML válida recibida\n";
                
                // Verificar si hay datos de bodega en la respuesta
                $soap_ns = 'http://schemas.xmlsoap.org/soap/envelope/';
                $softland_ns = 'http://softland.cl/';
                
                try {
                    $body = $xml->children($soap_ns)->Body;
                    $response_node = $body->children($softland_ns)->ObtieneBodegasResponse;
                    
                    if (isset($response_node->ObtieneBodegasResult->Bodega)) {
                        echo "   ✅ OK: Se recibieron datos de bodegas correctamente\n";
                    } else {
                        echo "   ⚠️ ADVERTENCIA: No se encontraron datos de bodegas en la respuesta\n";
                    }
                } catch (Exception $e) {
                    echo "   ⚠️ ADVERTENCIA: Error al extraer datos de la respuesta: " . $e->getMessage() . "\n";
                }
            }
        } else {
            echo "   ❌ ERROR: Error HTTP " . $http_code . " al conectar con Softland\n";
        }
    }
    
    curl_close($ch);
} catch (Exception $e) {
    echo "   ❌ ERROR: " . $e->getMessage() . "\n";
}
echo "\n";

// 7. Recomendaciones finales
echo "7. Recomendaciones finales basadas en el diagnóstico\n\n";

echo "Para solucionar problemas de scripts que no ejecutan correctamente:\n";
echo "1. Verifica que las credenciales de la base de datos en el script sean correctas\n";
echo "2. Asegúrate de que las credenciales de Softland sean válidas\n";
echo "3. Comprueba que el campo 'ubicacion' exista en la tabla 'repuestos'\n";
echo "4. Revisa los permisos de la carpeta logs/ y asegúrate de que sea escribible\n";
echo "5. Ejecuta el script con error_reporting activado para ver errores detallados\n";
echo "6. Verifica que el CRON esté configurado correctamente (si usas ejecución automática)\n\n";

echo "=== FIN DEL DIAGNÓSTICO ===\n";
echo "</pre>";
?>