<?php
/**
 * Gestión de usuarios del sistema
 * CRUD para usuarios y gestión de permisos
 * Versión optimizada con mejoras de rendimiento y seguridad
 */

// Definir el título de la página
$page_title = 'Gestión de Usuarios';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';
require_once 'includes/permisos.php';

// Verificar sesión y permisos
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');
$is_super_admin = ($user_rol === 'super_admin');

// Si no es admin, redirigir al dashboard
if (!$is_admin) {
    redirigirConMensaje('dashboard.php', "No tienes permisos para acceder a esta sección", 'danger');
    exit;
}

// Determinar la acción a realizar
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Conexión a la base de datos
try {
    $conn = getDatabaseConnection();
    
    // Obtener permisos agrupados por módulo
    $stmt = $conn->prepare("
        SELECT p.*, 
               CASE 
                   WHEN p.nombre LIKE '%rendicion%' THEN 'Rendiciones'
                   WHEN p.nombre LIKE '%orden_compra%' THEN 'Órdenes de Compra'
                   WHEN p.nombre LIKE '%maquina%' THEN 'Máquinas'
                   WHEN p.nombre LIKE '%repuesto%' THEN 'Repuestos'
                   WHEN p.nombre LIKE '%usuario%' THEN 'Usuarios'
                   WHEN p.nombre LIKE '%devolucion%' THEN 'Devoluciones'
                   WHEN p.nombre LIKE '%sistema%' OR p.nombre LIKE '%cache%' OR p.nombre LIKE '%backup%' THEN 'Sistema'
                   ELSE 'General'
               END as modulo,
               CASE 
                   WHEN p.nombre LIKE '%crear%' THEN 1
                   WHEN p.nombre LIKE '%editar%' OR p.nombre LIKE '%modificar%' THEN 2
                   WHEN p.nombre LIKE '%ver%' OR p.nombre LIKE '%listar%' THEN 3
                   WHEN p.nombre LIKE '%aprobar%' OR p.nombre LIKE '%rechazar%' THEN 4
                   WHEN p.nombre LIKE '%eliminar%' THEN 5
                   ELSE 6
               END as orden
        FROM permisos p
        WHERE p.nombre NOT LIKE '%guia_entrada%' 
        AND p.nombre NOT LIKE '%nota_venta%'
        ORDER BY modulo, orden, nombre
    ");
    $stmt->execute();
    $todos_permisos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Agrupar permisos por módulo
    $permisos_por_modulo = [];
    foreach ($todos_permisos as $permiso) {
        $permisos_por_modulo[$permiso['modulo']][] = $permiso;
    }
    
    // Procesar formulario de creación o edición
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        // Iniciar transacción
        $conn->beginTransaction();
        
        try {
            // Validar datos del usuario
            $datos = [
                'rut' => trim($_POST['rut']),
                'nombre' => trim($_POST['nombre']),
                'apellido' => trim($_POST['apellido']),
                'email' => trim($_POST['email']),
                'telefono' => trim($_POST['telefono'] ?? ''),
                'cargo' => trim($_POST['cargo']),
                'rol' => trim($_POST['rol']),
                'password' => trim($_POST['password'] ?? ''),
                'password_confirm' => trim($_POST['password_confirm'] ?? '')
            ];
            
            $errores = validarDatosUsuario($datos, $action === 'new');
            
            if (empty($errores)) {
                // Verificar duplicados (RUT y email)
                if (!verificarDuplicados($conn, $datos['rut'], $datos['email'], $action === 'edit' ? $id : null)) {
                    if ($action === 'edit') {
                        actualizarUsuario($conn, $id, $datos);
                    } else {
                        $id = crearUsuario($conn, $datos);
                    }
                    
                    // Gestionar permisos
                    if ($id != 1 || $is_super_admin) {
                        actualizarPermisosUsuario($conn, $id, $_POST['permisos'] ?? []);
                    }
                    
                    $conn->commit();
                    redirigirConMensaje('usuarios.php', 
                        $action === 'edit' ? "Usuario actualizado correctamente" : "Usuario creado correctamente", 
                        'success');
                }
            }
            
            $error_message = implode("<br>", $errores);
            $conn->rollBack();
            
        } catch (Exception $e) {
            $conn->rollBack();
            $error_message = "Error al " . ($action === 'edit' ? "actualizar" : "crear") . " usuario: " . $e->getMessage();
        }
    }
    
    // Acción para activar/desactivar usuarios
    if ($action === 'toggle' && $id > 0) {
        if ($id == 1 || $id == $user_id) {
            redirigirConMensaje('usuarios.php', "No puedes desactivar esta cuenta", 'danger');
        }
        
        try {
            $stmt = $conn->prepare("SELECT activo, nombre, apellido FROM users WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($usuario) {
                $nuevo_estado = $usuario['activo'] ? 0 : 1;
                $stmt = $conn->prepare("UPDATE users SET activo = :activo, updated_at = NOW() WHERE id = :id");
                $stmt->bindParam(':activo', $nuevo_estado);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                $accion = $nuevo_estado ? 'activar' : 'desactivar';
                registrarActividad($conn, $user_id, $accion, 
                    ucfirst($accion) . ' usuario: ' . $usuario['nombre'] . ' ' . $usuario['apellido'], 
                    $id, 'usuario');
                
                redirigirConMensaje('usuarios.php', "Estado del usuario actualizado correctamente", 'success');
            }
        } catch (Exception $e) {
            redirigirConMensaje('usuarios.php', "Error al cambiar estado: " . $e->getMessage(), 'danger');
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

/**
 * Valida los datos del usuario
 * 
 * @param array $datos Datos del usuario
 * @param bool $es_nuevo Si es un nuevo usuario
 * @return array Array de errores
 */
function validarDatosUsuario($datos, $es_nuevo = true) {
    $errores = [];
    
    if (empty($datos['rut'])) {
        $errores[] = "El RUT es obligatorio";
    } else if (!validarRut($datos['rut'])) {
        $errores[] = "El RUT no es válido";
    }
    
    if (empty($datos['nombre'])) {
        $errores[] = "El nombre es obligatorio";
    }
    
    if (empty($datos['apellido'])) {
        $errores[] = "El apellido es obligatorio";
    }
    
    if (empty($datos['email'])) {
        $errores[] = "El email es obligatorio";
    } else if (!filter_var($datos['email'], FILTER_VALIDATE_EMAIL)) {
        $errores[] = "El email no es válido";
    }
    
    if (empty($datos['cargo'])) {
        $errores[] = "El cargo es obligatorio";
    }
    
    if (empty($datos['rol'])) {
        $errores[] = "El rol es obligatorio";
    }
    
    if ($es_nuevo || !empty($datos['password'])) {
        if (empty($datos['password'])) {
            $errores[] = "La contraseña es obligatoria para nuevos usuarios";
        } else if (strlen($datos['password']) < 6) {
            $errores[] = "La contraseña debe tener al menos 6 caracteres";
        } else if ($datos['password'] !== $datos['password_confirm']) {
            $errores[] = "Las contraseñas no coinciden";
        }
    }
    
    return $errores;
}

/**
 * Verifica si existe un usuario con el mismo RUT o email
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param string $rut RUT a verificar
 * @param string $email Email a verificar
 * @param int|null $id ID del usuario en caso de edición
 * @return bool True si existe duplicado, false en caso contrario
 */
function verificarDuplicados($conn, $rut, $email, $id = null) {
    // Verificar RUT
    $sql = "SELECT id FROM users WHERE rut = :rut";
    if ($id !== null) {
        $sql .= " AND id != :id";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bindParam(':rut', $rut);
    if ($id !== null) {
        $stmt->bindParam(':id', $id);
    }
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        return true;
    }
    
    // Verificar email
    $sql = "SELECT id FROM users WHERE email = :email";
    if ($id !== null) {
        $sql .= " AND id != :id";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bindParam(':email', $email);
    if ($id !== null) {
        $stmt->bindParam(':id', $id);
    }
    $stmt->execute();
    
    return $stmt->rowCount() > 0;
}

/**
 * Crea un nuevo usuario
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param array $datos Datos del usuario
 * @return int ID del usuario creado
 */
function crearUsuario($conn, $datos) {
    $stmt = $conn->prepare("
        INSERT INTO users 
        (rut, nombre, apellido, email, password, telefono, cargo, rol, activo, created_at, updated_at) 
        VALUES (:rut, :nombre, :apellido, :email, :password, :telefono, :cargo, :rol, 1, NOW(), NOW())
    ");
    
    $hashed_password = password_hash($datos['password'], PASSWORD_DEFAULT);
    
    $stmt->bindParam(':rut', $datos['rut']);
    $stmt->bindParam(':nombre', $datos['nombre']);
    $stmt->bindParam(':apellido', $datos['apellido']);
    $stmt->bindParam(':email', $datos['email']);
    $stmt->bindParam(':password', $hashed_password);
    $stmt->bindParam(':telefono', $datos['telefono']);
    $stmt->bindParam(':cargo', $datos['cargo']);
    $stmt->bindParam(':rol', $datos['rol']);
    $stmt->execute();
    
    $id = $conn->lastInsertId();
    registrarActividad($conn, $_SESSION['user_id'], 'crear', 
        'Creación de usuario: ' . $datos['nombre'] . ' ' . $datos['apellido'], 
        $id, 'usuario');
    
    return $id;
}

/**
 * Actualiza un usuario existente
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param int $id ID del usuario
 * @param array $datos Datos del usuario
 */
function actualizarUsuario($conn, $id, $datos) {
    if (!empty($datos['password'])) {
        $sql = "
            UPDATE users 
            SET rut = :rut, 
                nombre = :nombre, 
                apellido = :apellido, 
                email = :email, 
                telefono = :telefono, 
                cargo = :cargo, 
                rol = :rol, 
                password = :password,
                updated_at = NOW()
            WHERE id = :id
        ";
        
        $stmt = $conn->prepare($sql);
        $hashed_password = password_hash($datos['password'], PASSWORD_DEFAULT);
        $stmt->bindParam(':password', $hashed_password);
    } else {
        $sql = "
            UPDATE users 
            SET rut = :rut, 
                nombre = :nombre, 
                apellido = :apellido, 
                email = :email, 
                telefono = :telefono, 
                cargo = :cargo, 
                rol = :rol, 
                updated_at = NOW()
            WHERE id = :id
        ";
        
        $stmt = $conn->prepare($sql);
    }
    
    $stmt->bindParam(':rut', $datos['rut']);
    $stmt->bindParam(':nombre', $datos['nombre']);
    $stmt->bindParam(':apellido', $datos['apellido']);
    $stmt->bindParam(':email', $datos['email']);
    $stmt->bindParam(':telefono', $datos['telefono']);
    $stmt->bindParam(':cargo', $datos['cargo']);
    $stmt->bindParam(':rol', $datos['rol']);
    $stmt->bindParam(':id', $id);
    $stmt->execute();
    
    registrarActividad($conn, $_SESSION['user_id'], 'editar', 
        'Edición de usuario: ' . $datos['nombre'] . ' ' . $datos['apellido'], 
        $id, 'usuario');
}

/**
 * Actualiza los permisos de un usuario
 * 
 * @param PDO $conn Conexión a la base de datos
 * @param int $user_id ID del usuario
 * @param array $permisos_seleccionados Array de IDs de permisos
 */
function actualizarPermisosUsuario($conn, $user_id, $permisos_seleccionados) {
    // Eliminar permisos anteriores
    $stmt = $conn->prepare("DELETE FROM user_permisos WHERE user_id = :user_id");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    
    // Insertar nuevos permisos
    if (!empty($permisos_seleccionados)) {
        $permisos_seleccionados = array_unique($permisos_seleccionados);
        $stmt = $conn->prepare("
            INSERT INTO user_permisos (user_id, permiso_id) 
            VALUES (:user_id, :permiso_id)
        ");
        
        foreach ($permisos_seleccionados as $permiso_id) {
            $stmt->bindParam(':user_id', $user_id);
            $stmt->bindParam(':permiso_id', $permiso_id);
            $stmt->execute();
        }
    }
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    /* Estilos personalizados para la gestión de usuarios */
    .usuarios-table {
        width: 100%;
        border-collapse: collapse;
    }
    
    .usuarios-table td, .usuarios-table th {
        padding: 10px;
        border-bottom: 1px solid #eee;
    }
    
    .usuarios-table th {
        background-color: #f5f5f5;
        font-weight: 600;
        color: #333;
        text-align: left;
    }
    
    .usuarios-table tr:hover {
        background-color: rgba(0,0,0,0.03);
    }
    
    .action-icon {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 32px;
        height: 32px;
        border-radius: 50%;
        margin-right: 5px;
        transition: all 0.2s;
        color: #555 !important;
        background-color: #f0f0f0;
    }
    
    .action-icon:hover {
        background-color: #e0e0e0;
        transform: scale(1.1);
    }
    
    .action-icon.view-icon:hover {
        color: #1e88e5 !important;
    }
    
    .action-icon.edit-icon:hover {
        color: #ffa726 !important;
    }
    
    .action-icon.toggle-icon:hover {
        color: #66bb6a !important;
    }
    
    .action-icon.toggle-off-icon:hover {
        color: #ef5350 !important;
    }
    
    .action-icon i {
        font-size: 18px;
    }
    
    /* Estilos para indicador de usuarios inactivos */
    .inactive-user {
        opacity: 0.7;
        font-style: italic;
    }
    
    /* Estilos para badges de roles */
    .badge-rol {
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 500;
        display: inline-block;
        text-align: center;
    }
    
    .badge-super_admin {
        background-color: rgba(239, 83, 80, 0.2);
        color: #ef5350;
    }
    
    .badge-admin {
        background-color: rgba(30, 136, 229, 0.2);
        color: #1e88e5;
    }
    
    .badge-usuario {
        background-color: rgba(96, 125, 139, 0.2);
        color: #607d8b;
    }
    
    /* Estilos para formularios */
    .form-row {
        display: flex;
        flex-wrap: wrap;
        margin: 0 -10px;
    }
    
    .form-col {
        flex: 1;
        padding: 0 10px;
        min-width: 250px;
    }
    
    .form-control:focus {
        border-color: #1e88e5;
        box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
    }
    
    .form-control:required {
        border-left: 3px solid #1e88e5;
    }
    
    /* Estilos para permisos */
    .permisos-container {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
        gap: 20px;
        padding: 20px;
    }
    
    .permiso-section {
        background: #fff;
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        overflow: hidden;
    }
    
    .permiso-section-header {
        display: flex;
        align-items: center;
        padding: 15px;
        background: #f8f9fa;
        border-bottom: 1px solid #e9ecef;
    }
    
    .permiso-section-header i {
        margin-right: 10px;
        color: #1e88e5;
    }
    
    .permiso-section-header h4 {
        margin: 0;
        font-size: 16px;
        color: #333;
    }
    
    .permiso-items {
        padding: 15px;
    }
    
    .permiso-item {
        margin-bottom: 10px;
    }
    
    .checkbox-custom {
        display: flex;
        align-items: center;
        padding: 8px 12px;
        border-radius: 6px;
        transition: all 0.2s;
        cursor: pointer;
        background-color: #f8f9fa;
        border: 1px solid #e9ecef;
    }
    
    .checkbox-custom:hover {
        background-color: #e9ecef;
    }
    
    .checkbox-custom input[type="checkbox"] {
        margin-right: 10px;
        width: 18px;
        height: 18px;
    }
    
    .checkbox-label {
        font-size: 14px;
        color: #495057;
    }
    
    .alert {
        display: flex;
        align-items: center;
        padding: 10px 15px;
        margin: 10px 0;
        border-radius: 4px;
    }
    
    .alert i {
        margin-right: 10px;
    }
    
    .alert-info {
        background-color: #e3f2fd;
        color: #1e88e5;
        border: 1px solid #bbdefb;
    }
    
    /* Filtros de búsqueda */
    .filtros-container {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 20px;
    }
    
    .filtro-item {
        flex: 1;
        min-width: 200px;
    }
    
    /* Responsividad */
    @media (max-width: 992px) {
        .permiso-item {
            flex: 0 0 50%;
            max-width: 50%;
        }
    }
    
    @media (max-width: 768px) {
        .permiso-item {
            flex: 0 0 100%;
            max-width: 100%;
        }
        
        .form-col {
            flex: 0 0 100%;
        }
        
        .action-buttons {
            display: flex;
            flex-direction: column;
            gap: 5px;
        }
        
        .permisos-container {
            grid-template-columns: 1fr;
        }
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<?php
// Mostrar contenido según la acción
switch ($action) {
    case 'new':
    case 'edit':
        // Cargar datos del usuario si es edición
        $usuario = null;
        $usuario_permisos = [];
        
        if ($action === 'edit' && $id > 0) {
            $stmt = $conn->prepare("SELECT * FROM users WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$usuario) {
                echo '<div class="alert alert-danger">Usuario no encontrado</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="usuarios.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
            
            // Cargar permisos del usuario
            $stmt = $conn->prepare("SELECT permiso_id FROM user_permisos WHERE user_id = :user_id");
            $stmt->bindParam(':user_id', $id);
            $stmt->execute();
            $usuario_permisos = $stmt->fetchAll(PDO::FETCH_COLUMN);
        }
        
        // No permitir editar usuario principal a menos que sea super_admin
        $readonly_principal = ($id == 1 && !$is_super_admin) ? 'readonly' : '';
        $disabled_permisos = ($id == 1 && !$is_super_admin) ? 'disabled' : '';
        
        $titulo = ($action === 'new') ? 'Crear Nuevo Usuario' : 'Editar Usuario';
        ?>
        
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><?php echo $titulo; ?></h2>
                <a href="usuarios.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="usuarios.php?action=<?php echo $action . ($action === 'edit' ? '&id=' . $id : ''); ?>">
                <div class="form-row">
                    <div class="form-col">
                        <div class="form-group">
                            <label for="rut" class="form-label">RUT <span class="text-danger">*</span></label>
                            <input type="text" id="rut" name="rut" class="form-control" 
                                   placeholder="Ej: 12.345.678-9" required autocomplete="off" <?php echo $readonly_principal; ?>
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['rut']) : ''; ?>">
                            <small class="form-text text-muted">Formato: 12.345.678-9</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="nombre" class="form-label">Nombre <span class="text-danger">*</span></label>
                            <input type="text" id="nombre" name="nombre" class="form-control" 
                                   placeholder="Nombre del usuario" required autocomplete="off" <?php echo $readonly_principal; ?>
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['nombre']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="apellido" class="form-label">Apellido <span class="text-danger">*</span></label>
                            <input type="text" id="apellido" name="apellido" class="form-control" 
                                   placeholder="Apellido del usuario" required autocomplete="off" <?php echo $readonly_principal; ?>
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['apellido']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                            <input type="email" id="email" name="email" class="form-control" 
                                   placeholder="ejemplo@empresa.cl" required autocomplete="off" <?php echo $readonly_principal; ?>
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['email']) : ''; ?>">
                        </div>
                    </div>
                    
                    <div class="form-col">
                        <div class="form-group">
                            <label for="telefono" class="form-label">Teléfono</label>
                            <input type="text" id="telefono" name="telefono" class="form-control" 
                                   placeholder="Ej: +56912345678" autocomplete="off"
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['telefono']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="cargo" class="form-label">Cargo <span class="text-danger">*</span></label>
                            <input type="text" id="cargo" name="cargo" class="form-control" 
                                   placeholder="Cargo del usuario" required autocomplete="off"
                                   value="<?php echo isset($usuario) ? htmlspecialchars($usuario['cargo']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="rol" class="form-label">Rol <span class="text-danger">*</span></label>
                            <select id="rol" name="rol" class="form-control" required <?php echo $readonly_principal ? 'disabled' : ''; ?>>
                                <option value="">Seleccione un rol</option>
                                <?php if ($is_super_admin): ?>
                                <option value="super_admin" <?php echo (isset($usuario) && $usuario['rol'] === 'super_admin') ? 'selected' : ''; ?>>
                                    Super Administrador
                                </option>
                                <?php endif; ?>
                                <option value="admin" <?php echo (isset($usuario) && $usuario['rol'] === 'admin') ? 'selected' : ''; ?>>
                                    Administrador
                                </option>
                                <option value="usuario" <?php echo (isset($usuario) && $usuario['rol'] === 'usuario') ? 'selected' : ''; ?>>
                                    Usuario
                                </option>
                            </select>
                            <?php if ($readonly_principal): ?>
                            <input type="hidden" name="rol" value="<?php echo $usuario['rol']; ?>">
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($action === 'new' || $id != 1 || $is_super_admin): ?>
                        <div class="form-group">
                            <label for="password" class="form-label">
                                Contraseña <?php echo ($action === 'new') ? '<span class="text-danger">*</span>' : ''; ?>
                            </label>
                            <input type="password" id="password" name="password" class="form-control" 
                                   placeholder="<?php echo ($action === 'new') ? 'Contraseña' : 'Dejar en blanco para no cambiar'; ?>" 
                                   <?php echo ($action === 'new') ? 'required' : ''; ?>>
                            <small class="form-text text-muted">Mínimo 6 caracteres</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="password_confirm" class="form-label">
                                Confirmar Contraseña <?php echo ($action === 'new') ? '<span class="text-danger">*</span>' : ''; ?>
                            </label>
                            <input type="password" id="password_confirm" name="password_confirm" class="form-control" 
                                   placeholder="Confirme la contraseña" <?php echo ($action === 'new') ? 'required' : ''; ?>>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Sección de Permisos Mejorada -->
                <div class="card" style="margin-top: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">
                            <i class="material-icons">security</i> 
                            Permisos del Usuario
                        </h3>
                        <?php if ($action === 'edit' && $id == 1 && !$is_super_admin): ?>
                        <div class="alert alert-info">
                            <i class="material-icons">info</i>
                            Los permisos del usuario principal no pueden ser modificados
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="permisos-container">
                        <?php foreach ($permisos_por_modulo as $modulo => $permisos_modulo): ?>
                        <div class="permiso-section">
                            <div class="permiso-section-header">
                                <i class="material-icons">
                                    <?php
                                    $icon = 'folder';
                                    switch ($modulo) {
                                        case 'Rendiciones': $icon = 'description'; break;
                                        case 'Órdenes de Compra': $icon = 'shopping_cart'; break;
                                        case 'Máquinas': $icon = 'precision_manufacturing'; break;
                                        case 'Repuestos': $icon = 'build'; break;
                                        case 'Usuarios': $icon = 'people'; break;
                                        case 'Devoluciones': $icon = 'assignment_return'; break;
                                        case 'Sistema': $icon = 'settings'; break;
                                    }
                                    echo $icon;
                                    ?>
                                </i>
                                <h4><?php echo $modulo; ?></h4>
                            </div>
                            <div class="permiso-items">
                                <?php foreach ($permisos_modulo as $permiso): ?>
                                <div class="permiso-item">
                                    <label class="checkbox-custom">
                                        <input type="checkbox" 
                                               name="permisos[]" 
                                               value="<?php echo $permiso['id']; ?>" 
                                               <?php echo in_array($permiso['id'], $usuario_permisos) ? 'checked' : ''; ?> 
                                               <?php echo $disabled_permisos; ?>>
                                        <span class="checkbox-label">
                                            <?php echo htmlspecialchars($permiso['descripcion']); ?>
                                        </span>
                                    </label>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> <?php echo ($action === 'new') ? 'Crear Usuario' : 'Guardar Cambios'; ?>
                    </button>
                    
                    <a href="usuarios.php" class="btn btn-danger" style="margin-left: 10px;">
                        <i class="material-icons">cancel</i> Cancelar
                    </a>
                </div>
            </form>
        </div>
        <?php
        break;
        
    default:
        // Lista de usuarios
        try {
            // Filtros de búsqueda
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            $rol = isset($_GET['rol']) ? trim($_GET['rol']) : '';
            $activo = isset($_GET['activo']) ? $_GET['activo'] : 'all';
            
            // Construir consulta
            $sql = "SELECT * FROM users WHERE 1=1";
            $params = [];
            
            if (!empty($busqueda)) {
                $sql .= " AND (nombre LIKE :busqueda 
                               OR apellido LIKE :busqueda 
                               OR email LIKE :busqueda 
                               OR rut LIKE :busqueda
                               OR cargo LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            
            if (!empty($rol)) {
                $sql .= " AND rol = :rol";
                $params[':rol'] = $rol;
            }
            
            if ($activo !== 'all') {
                $activo_value = ($activo === 'active') ? 1 : 0;
                $sql .= " AND activo = :activo";
                $params[':activo'] = $activo_value;
            }
            
            $sql .= " ORDER BY nombre, apellido";
            
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
            ?>
            
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Gestión de Usuarios</h2>
                    <?php if (verificarPermisosAccion('usuarios', 'crear', false)): ?>
                    <a href="usuarios.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nuevo Usuario
                    </a>
                    <?php endif; ?>
                </div>
                
                <!-- Filtros de búsqueda -->
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="usuarios.php" class="filtros-container">
                        <div class="filtro-item">
                            <label for="rol" class="form-label">Rol</label>
                            <select id="rol" name="rol" class="form-control">
                                <option value="">Todos</option>
                                <option value="super_admin" <?php echo $rol === 'super_admin' ? 'selected' : ''; ?>>Super Administrador</option>
                                <option value="admin" <?php echo $rol === 'admin' ? 'selected' : ''; ?>>Administrador</option>
                                <option value="usuario" <?php echo $rol === 'usuario' ? 'selected' : ''; ?>>Usuario</option>
                            </select>
                        </div>
                        
                        <div class="filtro-item">
                            <label for="activo" class="form-label">Estado</label>
                            <select id="activo" name="activo" class="form-control">
                                <option value="all" <?php echo $activo === 'all' ? 'selected' : ''; ?>>Todos</option>
                                <option value="active" <?php echo $activo === 'active' ? 'selected' : ''; ?>>Activos</option>
                                <option value="inactive" <?php echo $activo === 'inactive' ? 'selected' : ''; ?>>Inactivos</option>
                            </select>
                        </div>
                        
                        <div class="filtro-item">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" 
                                   placeholder="Nombre, RUT, Email, Cargo..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div class="filtro-item" style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="usuarios.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <div class="table-responsive">
                    <table class="usuarios-table">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>RUT</th>
                                <th>Email</th>
                                <th>Cargo</th>
                                <th>Rol</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($usuarios)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center;">
                                    No se encontraron usuarios
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($usuarios as $usuario): ?>
                                <tr class="<?php echo $usuario['activo'] ? '' : 'inactive-user'; ?>">
                                    <td><?php echo htmlspecialchars($usuario['nombre'] . ' ' . $usuario['apellido']); ?></td>
                                    <td><?php echo htmlspecialchars($usuario['rut']); ?></td>
                                    <td><?php echo htmlspecialchars($usuario['email']); ?></td>
                                    <td><?php echo htmlspecialchars($usuario['cargo']); ?></td>
                                    <td>
                                        <span class="badge-rol badge-<?php echo $usuario['rol']; ?>">
                                            <?php 
                                                switch ($usuario['rol']) {
                                                    case 'super_admin':
                                                        echo "Super Admin";
                                                        break;
                                                    case 'admin':
                                                        echo "Administrador";
                                                        break;
                                                    default:
                                                        echo "Usuario";
                                                }
                                            ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($usuario['activo']): ?>
                                            <span class="badge-rol" style="background-color: rgba(102, 187, 106, 0.2); color: #66bb6a;">
                                                Activo
                                            </span>
                                        <?php else: ?>
                                            <span class="badge-rol" style="background-color: rgba(239, 83, 80, 0.2); color: #ef5350;">
                                                Inactivo
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <?php if (verificarPermisosAccion('usuarios', 'editar', false)): ?>
                                            <a href="usuarios.php?action=edit&id=<?php echo $usuario['id']; ?>" 
                                               title="Editar" class="action-icon edit-icon">
                                                <i class="material-icons">edit</i>
                                            </a>
                                            <?php endif; ?>
                                            
                                            <?php if ($usuario['id'] != 1 && $usuario['id'] != $user_id && verificarPermisosAccion('usuarios', 'editar', false)): ?>
                                                <?php if ($usuario['activo']): ?>
                                                <a href="usuarios.php?action=toggle&id=<?php echo $usuario['id']; ?>" 
                                                   onclick="return confirm('¿Está seguro de desactivar a este usuario?')"
                                                   title="Desactivar" class="action-icon toggle-off-icon">
                                                    <i class="material-icons">toggle_off</i>
                                                </a>
                                                <?php else: ?>
                                                <a href="usuarios.php?action=toggle&id=<?php echo $usuario['id']; ?>" 
                                                   onclick="return confirm('¿Está seguro de activar a este usuario?')"
                                                   title="Activar" class="action-icon toggle-icon">
                                                    <i class="material-icons">toggle_on</i>
                                                </a>
                                                <?php endif; ?>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar usuarios: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<script>
    // Función para formatear RUT mientras se escribe
    document.addEventListener('DOMContentLoaded', function() {
        const rutInput = document.getElementById('rut');
        if (rutInput) {
            rutInput.addEventListener('input', function(e) {
                let rut = e.target.value.replace(/[^0-9kK]/g, '');
                
                if (rut.length > 1) {
                    // Separar dígito verificador
                    const dv = rut.slice(-1);
                    const rutCuerpo = rut.slice(0, -1);
                    
                    // Formatear con puntos y guión
                    let rutFormateado = '';
                    for (let i = rutCuerpo.length - 1, j = 0; i >= 0; i--, j++) {
                        rutFormateado = rutCuerpo.charAt(i) + rutFormateado;
                        if ((j + 1) % 3 === 0 && i !== 0) {
                            rutFormateado = '.' + rutFormateado;
                        }
                    }
                    
                    // Agregar guión y dígito verificador
                    rutFormateado = rutFormateado + '-' + dv;
                    
                    e.target.value = rutFormateado;
                }
            });
        }
        
        // Evitar el envío del formulario al presionar Enter en los inputs
        document.querySelectorAll('input:not([type="submit"])').forEach(input => {
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    const form = this.closest('form');
                    const inputs = Array.from(form.elements);
                    const index = inputs.indexOf(this);
                    if (index < inputs.length - 1) {
                        inputs[index + 1].focus();
                    }
                }
            });
        });
    });
</script>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>