<?php
/**
 * Gestión de proveedores
 * CRUD completo para proveedores
 */

// Definir el título de la página
$page_title = 'Gestión de Proveedores';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];
$user_rol = $_SESSION['user_rol'] ?? '';
$user_nombre = $_SESSION['user_nombre'] ?? '';
$user_apellido = $_SESSION['user_apellido'] ?? '';
$is_admin = ($user_rol === 'admin' || $user_rol === 'super_admin');

// Si no es admin, redirigir al dashboard
if (!$is_admin) {
    $_SESSION['flash_message'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['flash_type'] = 'danger';
    header('Location: dashboard.php');
    exit;
}

// Determinar la acción a realizar
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? intval($_GET['id']) : 0;

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Procesar formulario de creación o edición
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'new' || $action === 'edit')) {
        // Datos generales del proveedor
        $rut = trim($_POST['rut']);
        $nombre = trim($_POST['nombre']);
        $direccion = trim($_POST['direccion']);
        $telefono = trim($_POST['telefono']);
        $email = trim($_POST['email']);
        $contacto = trim($_POST['contacto']);
        $giro = trim($_POST['giro']);
        $activo = isset($_POST['activo']) ? 1 : 0;
        
        // Validación básica
        $errores = [];
        
        if (empty($nombre)) {
            $errores[] = "El nombre del proveedor es obligatorio";
        }
        
        if (empty($rut)) {
            $errores[] = "El RUT es obligatorio";
        } elseif (!validarRut($rut)) {
            $errores[] = "El RUT no es válido";
        }
        
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errores[] = "El email no es válido";
        }
        
        // Verificar que no exista otro proveedor con el mismo RUT
        $stmt = $conn->prepare("SELECT id FROM proveedores WHERE rut = :rut" . ($action === 'edit' ? " AND id != :id" : ""));
        $stmt->bindParam(":rut", $rut);
        if ($action === 'edit') {
            $stmt->bindParam(":id", $id);
        }
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $errores[] = "Ya existe un proveedor con este RUT";
        }
        
        // Si no hay errores, proceder a guardar
        if (empty($errores)) {
            $conn->beginTransaction();
            
            try {
                if ($action === 'edit') {
                    // Actualizar proveedor existente
                    $stmt = $conn->prepare("
                        UPDATE proveedores 
                        SET rut = :rut, 
                            nombre = :nombre, 
                            direccion = :direccion,
                            telefono = :telefono,
                            email = :email,
                            contacto = :contacto,
                            giro = :giro,
                            activo = :activo,
                            updated_at = NOW()
                        WHERE id = :id
                    ");
                    
                    $stmt->bindParam(':id', $id);
                    $stmt->bindParam(':rut', $rut);
                    $stmt->bindParam(':nombre', $nombre);
                    $stmt->bindParam(':direccion', $direccion);
                    $stmt->bindParam(':telefono', $telefono);
                    $stmt->bindParam(':email', $email);
                    $stmt->bindParam(':contacto', $contacto);
                    $stmt->bindParam(':giro', $giro);
                    $stmt->bindParam(':activo', $activo);
                    $stmt->execute();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'editar', 'Edición de proveedor: ' . $nombre, $id, 'proveedor');
                    
                    $mensaje = "Proveedor actualizado correctamente";
                } else {
                    // Crear nuevo proveedor
                    $stmt = $conn->prepare("
                        INSERT INTO proveedores 
                        (rut, nombre, direccion, telefono, email, contacto, giro, activo, created_at, updated_at) 
                        VALUES (:rut, :nombre, :direccion, :telefono, :email, :contacto, :giro, :activo, NOW(), NOW())
                    ");
                    
                    $stmt->bindParam(':rut', $rut);
                    $stmt->bindParam(':nombre', $nombre);
                    $stmt->bindParam(':direccion', $direccion);
                    $stmt->bindParam(':telefono', $telefono);
                    $stmt->bindParam(':email', $email);
                    $stmt->bindParam(':contacto', $contacto);
                    $stmt->bindParam(':giro', $giro);
                    $stmt->bindParam(':activo', $activo);
                    $stmt->execute();
                    
                    $id = $conn->lastInsertId();
                    
                    // Registrar actividad
                    registrarActividad($conn, $user_id, 'crear', 'Creación de proveedor: ' . $nombre, $id, 'proveedor');
                    
                    $mensaje = "Proveedor creado correctamente";
                }
                
                $conn->commit();
                
                $_SESSION['flash_message'] = $mensaje;
                $_SESSION['flash_type'] = 'success';
                header('Location: proveedores.php');
                exit;
                
            } catch (Exception $e) {
                $conn->rollBack();
                $error_message = $e->getMessage();
            }
        } else {
            $error_message = implode("<br>", $errores);
        }
    }
    
    // Acción para activar/desactivar proveedor
    if ($action === 'toggle' && $id > 0) {
        try {
            // Obtener estado actual
            $stmt = $conn->prepare("SELECT activo, nombre FROM proveedores WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $proveedor = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$proveedor) {
                throw new Exception("Proveedor no encontrado");
            }
            
            // Cambiar estado
            $nuevo_estado = $proveedor['activo'] ? 0 : 1;
            
            $stmt = $conn->prepare("UPDATE proveedores SET activo = :activo, updated_at = NOW() WHERE id = :id");
            $stmt->bindParam(':activo', $nuevo_estado);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Registrar actividad
            $accion = $nuevo_estado ? 'activar' : 'desactivar';
            registrarActividad($conn, $user_id, $accion, ucfirst($accion) . ' proveedor: ' . $proveedor['nombre'], $id, 'proveedor');
            
            $_SESSION['flash_message'] = "Estado del proveedor actualizado correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: proveedores.php');
            exit;
        } catch (Exception $e) {
            $_SESSION['flash_message'] = "Error al cambiar estado: " . $e->getMessage();
            $_SESSION['flash_type'] = 'danger';
            header('Location: proveedores.php');
            exit;
        }
    }
    
    // Acción para eliminar proveedor
    if ($action === 'delete' && $id > 0) {
        try {
            // Verificar si el proveedor está siendo utilizado en órdenes de compra
            $stmt = $conn->prepare("SELECT COUNT(*) as total FROM ordenes_compra WHERE proveedor_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $count = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
            
            if ($count > 0) {
                throw new Exception("No se puede eliminar el proveedor porque está siendo utilizado en órdenes de compra. Considere desactivarlo en su lugar.");
            }
            
            // Obtener el nombre para el registro de actividad
            $stmt = $conn->prepare("SELECT nombre FROM proveedores WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $proveedor = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$proveedor) {
                throw new Exception("Proveedor no encontrado");
            }
            
            // Eliminar proveedor
            $stmt = $conn->prepare("DELETE FROM proveedores WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'eliminar', 'Eliminación de proveedor: ' . $proveedor['nombre'], $id, 'proveedor');
            
            $_SESSION['flash_message'] = "Proveedor eliminado correctamente";
            $_SESSION['flash_type'] = 'success';
            header('Location: proveedores.php');
            exit;
        } catch (Exception $e) {
            $_SESSION['flash_message'] = "Error al eliminar el proveedor: " . $e->getMessage();
            $_SESSION['flash_type'] = 'danger';
            header('Location: proveedores.php');
            exit;
        }
    }
    
} catch (PDOException $e) {
    $error_message = 'Error de conexión a la base de datos: ' . $e->getMessage();
}

// Incluir el header
require_once 'includes/header.php';
?>

<style>
    /* Estilos personalizados para la tabla de proveedores */
    .proveedores-table td, .proveedores-table th {
        padding: 8px 10px; /* Reducimos el padding para filas menos gruesas */
    }
    
    .proveedores-table tr:hover {
        background-color: rgba(0,0,0,0.03);
    }
    
    .action-icon {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 28px;
        height: 28px;
        border-radius: 50%;
        margin-right: 5px;
        transition: all 0.2s;
        color: #555 !important; /* Forzamos color oscuro para los iconos */
        background-color: #f0f0f0;
    }
    
    .action-icon:hover {
        background-color: #e0e0e0;
        transform: scale(1.1);
    }
    
    .action-icon.view-icon:hover {
        color: #1e88e5 !important;
    }
    
    .action-icon.edit-icon:hover {
        color: #ffa726 !important;
    }
    
    .action-icon.toggle-icon:hover {
        color: #66bb6a !important;
    }
    
    .action-icon.toggle-off-icon:hover {
        color: #ef5350 !important;
    }
    
    .action-icon.delete-icon:hover {
        color: #ef5350 !important;
    }
    
    .action-icon i {
        font-size: 18px;
    }
    
    /* Estado activo/inactivo */
    .inactive-row {
        opacity: 0.65;
        font-style: italic;
    }
    
    /* Mejorar aspecto de los inputs */
    .form-control:focus {
        border-color: #1e88e5;
        box-shadow: 0 0 0 0.15rem rgba(30, 136, 229, 0.25);
    }
    
    /* Destacar campos requeridos */
    .form-control:required {
        border-left: 3px solid #1e88e5;
    }
    
    /* Reducir padding para inputs de texto en formularios */
    .input-sm {
        padding: 8px 10px;
        height: auto;
    }
</style>

<!-- Mensaje de error si existe -->
<?php if (isset($error_message)): ?>
<div class="alert alert-danger">
    <?php echo $error_message; ?>
</div>
<?php endif; ?>

<?php
// Mostrar contenido según la acción
switch ($action) {
    case 'new':
    case 'edit':
        // Formulario para crear o editar proveedor
        $proveedor = null;
        
        if ($action === 'edit' && $id > 0) {
            // Cargar datos del proveedor para edición
            $stmt = $conn->prepare("SELECT * FROM proveedores WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $proveedor = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$proveedor) {
                echo '<div class="alert alert-danger">Proveedor no encontrado</div>';
                echo '<div style="text-align: center; margin-top: 20px;">
                          <a href="proveedores.php" class="btn btn-primary">
                              <i class="material-icons">arrow_back</i> Volver a la Lista
                          </a>
                      </div>';
                break;
            }
        }
        
        $titulo = ($action === 'new') ? 'Crear Nuevo Proveedor' : 'Editar Proveedor';
        ?>
        
        <div class="card">
            <div class="panel-header">
                <h2 class="panel-title"><?php echo $titulo; ?></h2>
                <a href="proveedores.php" class="btn btn-primary btn-sm">
                    <i class="material-icons">arrow_back</i> Volver a la Lista
                </a>
            </div>
            
            <form method="POST" action="proveedores.php?action=<?php echo $action . ($action === 'edit' ? '&id=' . $id : ''); ?>">
                <div class="row" style="display: flex; flex-wrap: wrap; margin: 0 -10px;">
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                        <div class="form-group">
                            <label for="rut" class="form-label">RUT <span class="text-danger">*</span></label>
                            <input type="text" id="rut" name="rut" class="form-control input-sm" 
                                   placeholder="Ej: 76.123.456-7" required autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['rut']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="nombre" class="form-label">Nombre <span class="text-danger">*</span></label>
                            <input type="text" id="nombre" name="nombre" class="form-control input-sm" 
                                   placeholder="Nombre del proveedor" required autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['nombre']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="direccion" class="form-label">Dirección</label>
                            <input type="text" id="direccion" name="direccion" class="form-control input-sm" 
                                   placeholder="Dirección del proveedor" autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['direccion']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="giro" class="form-label">Giro</label>
                            <input type="text" id="giro" name="giro" class="form-control input-sm" 
                                   placeholder="Giro o actividad comercial" autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['giro']) : ''; ?>">
                        </div>
                    </div>
                    
                    <div class="col" style="flex: 1; padding: 0 10px; min-width: 250px;">
                        <div class="form-group">
                            <label for="telefono" class="form-label">Teléfono</label>
                            <input type="text" id="telefono" name="telefono" class="form-control input-sm" 
                                   placeholder="Teléfono de contacto" autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['telefono']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" id="email" name="email" class="form-control input-sm" 
                                   placeholder="Email de contacto" autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['email']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="contacto" class="form-label">Persona de Contacto</label>
                            <input type="text" id="contacto" name="contacto" class="form-control input-sm" 
                                   placeholder="Nombre del contacto" autocomplete="off"
                                   value="<?php echo isset($proveedor) ? htmlspecialchars($proveedor['contacto']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <div class="form-check" style="padding-top: 20px;">
                                <input type="checkbox" class="form-check-input" id="activo" name="activo" value="1" 
                                       <?php echo (!isset($proveedor) || $proveedor['activo'] == 1) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="activo">Activo</label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> <?php echo ($action === 'new') ? 'Crear Proveedor' : 'Guardar Cambios'; ?>
                    </button>
                </div>
            </form>
        </div>
        <?php
        break;
        
    default:
        // Lista de proveedores
        try {
            // Filtros de búsqueda
            $estado = isset($_GET['estado']) ? $_GET['estado'] : 'all';
            $busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : '';
            
            // Construir consulta base
            $sql = "SELECT * FROM proveedores WHERE 1=1";
            $params = [];
            
            // Aplicar filtros
            if ($estado === 'active') {
                $sql .= " AND activo = 1";
            } elseif ($estado === 'inactive') {
                $sql .= " AND activo = 0";
            }
            
            if (!empty($busqueda)) {
                $sql .= " AND (nombre LIKE :busqueda OR rut LIKE :busqueda OR direccion LIKE :busqueda 
                              OR email LIKE :busqueda OR contacto LIKE :busqueda OR giro LIKE :busqueda)";
                $params[':busqueda'] = '%' . $busqueda . '%';
            }
            
            // Ordenar por nombre
            $sql .= " ORDER BY nombre";
            
            // Preparar y ejecutar consulta
            $stmt = $conn->prepare($sql);
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $stmt->execute();
            $proveedores = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            ?>
            <div class="card">
                <div class="panel-header">
                    <h2 class="panel-title">Gestión de Proveedores</h2>
                    <a href="proveedores.php?action=new" class="btn btn-primary btn-sm">
                        <i class="material-icons">add</i> Nuevo Proveedor
                    </a>
                </div>
                
                <!-- Filtros de búsqueda -->
                <div class="card" style="margin-bottom: 20px;">
                    <div class="panel-header">
                        <h3 class="panel-title">Filtros de Búsqueda</h3>
                    </div>
                    
                    <form method="GET" action="proveedores.php" class="form-inline" style="display: flex; flex-wrap: wrap; gap: 10px;">
                        <div style="flex: 1; min-width: 200px;">
                            <label for="estado" class="form-label">Estado</label>
                            <select id="estado" name="estado" class="form-control">
                                <option value="all" <?php echo $estado === 'all' ? 'selected' : ''; ?>>Todos</option>
                                <option value="active" <?php echo $estado === 'active' ? 'selected' : ''; ?>>Activos</option>
                                <option value="inactive" <?php echo $estado === 'inactive' ? 'selected' : ''; ?>>Inactivos</option>
                            </select>
                        </div>
                        
                        <div style="flex: 2; min-width: 300px;">
                            <label for="busqueda" class="form-label">Búsqueda</label>
                            <input type="text" id="busqueda" name="busqueda" class="form-control" 
                                   placeholder="Nombre, RUT, Contacto..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <div style="display: flex; align-items: flex-end; gap: 10px;">
                            <button type="submit" class="btn btn-primary">
                                <i class="material-icons">search</i> Buscar
                            </button>
                            
                            <a href="proveedores.php" class="btn btn-danger">
                                <i class="material-icons">clear</i> Limpiar
                            </a>
                        </div>
                    </form>
                </div>
                
                <div class="table-responsive">
                    <table class="proveedores-table">
                        <thead>
                            <tr>
                                <th>RUT</th>
                                <th>Nombre</th>
                                <th>Contacto</th>
                                <th>Teléfono</th>
                                <th>Email</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($proveedores)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center;">No se encontraron proveedores</td>
                            </tr>
                            <?php else: ?>
                                <?php foreach ($proveedores as $proveedor): ?>
                                <tr class="<?php echo $proveedor['activo'] ? '' : 'inactive-row'; ?>">
                                    <td><?php echo htmlspecialchars($proveedor['rut']); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['nombre']); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['contacto']); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['telefono']); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['email']); ?></td>
                                    <td>
                                        <?php if ($proveedor['activo']): ?>
                                            <span class="status-badge badge-estado-3">Activo</span>
                                        <?php else: ?>
                                            <span class="status-badge badge-estado-4">Inactivo</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="proveedores.php?action=edit&id=<?php echo $proveedor['id']; ?>" 
                                           title="Editar" class="action-icon edit-icon">
                                            <i class="material-icons">edit</i>
                                        </a>
                                        
                                        <?php if ($proveedor['activo']): ?>
                                        <a href="proveedores.php?action=toggle&id=<?php echo $proveedor['id']; ?>" 
                                           onclick="return confirm('¿Está seguro de desactivar este proveedor?')"
                                           title="Desactivar" class="action-icon toggle-off-icon">
                                            <i class="material-icons">toggle_off</i>
                                        </a>
                                        <?php else: ?>
                                        <a href="proveedores.php?action=toggle&id=<?php echo $proveedor['id']; ?>" 
                                           onclick="return confirm('¿Está seguro de activar este proveedor?')"
                                           title="Activar" class="action-icon toggle-icon">
                                            <i class="material-icons">toggle_on</i>
                                        </a>
                                        <?php endif; ?>
                                        
                                        <a href="proveedores.php?action=delete&id=<?php echo $proveedor['id']; ?>" 
                                           onclick="return confirm('¿Está seguro de eliminar este proveedor? Esta acción no se puede deshacer.')"
                                           title="Eliminar" class="action-icon delete-icon">
                                            <i class="material-icons">delete</i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php
        } catch (PDOException $e) {
            echo '<div class="alert alert-danger">Error al cargar proveedores: ' . $e->getMessage() . '</div>';
        }
        break;
}
?>

<script>
    // Función para formatear RUT mientras se escribe
    document.addEventListener('DOMContentLoaded', function() {
        const rutInput = document.getElementById('rut');
        if (rutInput) {
            rutInput.addEventListener('input', function(e) {
                let rut = e.target.value.replace(/[^0-9kK]/g, '');
                
                if (rut.length > 1) {
                    // Separar dígito verificador
                    const dv = rut.slice(-1);
                    const rutCuerpo = rut.slice(0, -1);
                    
                    // Formatear con puntos y guión
                    let rutFormateado = '';
                    for (let i = rutCuerpo.length - 1, j = 0; i >= 0; i--, j++) {
                        rutFormateado = rutCuerpo.charAt(i) + rutFormateado;
                        if ((j + 1) % 3 === 0 && i !== 0) {
                            rutFormateado = '.' + rutFormateado;
                        }
                    }
                    
                    // Agregar guión y dígito verificador
                    rutFormateado = rutFormateado + '-' + dv;
                    
                    e.target.value = rutFormateado;
                }
            });
        }
    });
</script>

<?php
// Incluir el footer
require_once 'includes/footer.php';
?>