<?php
/**
 * Página de perfil de usuario
 * Permite ver y actualizar la información personal
 */

// Definir el título de la página
$page_title = 'Mi Perfil';

// Incluir archivos de configuración y funciones
require_once 'includes/config.php';
require_once 'includes/functions.php';

// Verificar sesión
verificarSesion();
$user_id = $_SESSION['user_id'];

// Conexión a la base de datos
try {
    $conn = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass, $db_options);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Procesar formulario de actualización
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $nombre = trim($_POST['nombre']);
        $apellido = trim($_POST['apellido']);
        $email = trim($_POST['email']);
        $telefono = trim($_POST['telefono'] ?? '');
        $cambiar_password = isset($_POST['cambiar_password']) && $_POST['cambiar_password'] == '1';
        $password_actual = $_POST['password_actual'] ?? '';
        $password_nuevo = $_POST['password_nuevo'] ?? '';
        $password_confirmar = $_POST['password_confirmar'] ?? '';
        
        // Validar datos
        $errores = [];
        
        if (empty($nombre)) {
            $errores[] = "El nombre es obligatorio";
        }
        
        if (empty($apellido)) {
            $errores[] = "El apellido es obligatorio";
        }
        
        if (empty($email)) {
            $errores[] = "El email es obligatorio";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errores[] = "El email ingresado no es válido";
        }
        
        // Verificar que el email no exista (excepto para el usuario actual)
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = :email AND id <> :id");
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':id', $user_id);
        $stmt->execute();
        if ($stmt->rowCount() > 0) {
            $errores[] = "El email ya está registrado en el sistema";
        }
        
        // Si va a cambiar la contraseña
        if ($cambiar_password) {
            if (empty($password_actual)) {
                $errores[] = "La contraseña actual es obligatoria";
            }
            
            if (empty($password_nuevo)) {
                $errores[] = "La nueva contraseña es obligatoria";
            } elseif (strlen($password_nuevo) < 6) {
                $errores[] = "La contraseña debe tener al menos 6 caracteres";
            }
            
            if (empty($password_confirmar)) {
                $errores[] = "Debe confirmar la nueva contraseña";
            }
            
            if ($password_nuevo !== $password_confirmar) {
                $errores[] = "Las contraseñas nuevas no coinciden";
            }
            
            // Verificar contraseña actual
            $stmt = $conn->prepare("SELECT password FROM users WHERE id = :id");
            $stmt->bindParam(':id', $user_id);
            $stmt->execute();
            $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!password_verify($password_actual, $usuario['password'])) {
                $errores[] = "La contraseña actual es incorrecta";
            }
        }
        
        // Si hay errores, redirigir con mensaje de error
        if (!empty($errores)) {
            $_SESSION['flash_message'] = implode('<br>', $errores);
            $_SESSION['flash_type'] = 'danger';
            header('Location: perfil.php');
            exit;
        }
        
        try {
            // Actualizar datos básicos del usuario
            if ($cambiar_password) {
                // Generar hash de la contraseña nueva
                $password_hash = password_hash($password_nuevo, PASSWORD_DEFAULT);
                
                $stmt = $conn->prepare("
                    UPDATE users 
                    SET nombre = :nombre, apellido = :apellido, email = :email, 
                        password = :password, telefono = :telefono, updated_at = NOW()
                    WHERE id = :id
                ");
                
                $stmt->bindParam(':password', $password_hash);
            } else {
                $stmt = $conn->prepare("
                    UPDATE users 
                    SET nombre = :nombre, apellido = :apellido, email = :email, 
                        telefono = :telefono, updated_at = NOW()
                    WHERE id = :id
                ");
            }
            
            $stmt->bindParam(':nombre', $nombre);
            $stmt->bindParam(':apellido', $apellido);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':telefono', $telefono);
            $stmt->bindParam(':id', $user_id);
            $stmt->execute();
            
            // Actualizar datos de la sesión
            $_SESSION['user_nombre'] = $nombre;
            $_SESSION['user_apellido'] = $apellido;
            $_SESSION['user_email'] = $email;
            
            // Registrar actividad
            registrarActividad($conn, $user_id, 'editar', 'Actualización de perfil personal', $user_id, 'usuario');
            
            $_SESSION['flash_message'] = 'Perfil actualizado exitosamente';
            $_SESSION['flash_type'] = 'success';
            header('Location: perfil.php');
            exit;
        } catch (Exception $e) {
            $_SESSION['flash_message'] = 'Error al actualizar el perfil: ' . $e->getMessage();
            $_SESSION['flash_type'] = 'danger';
            header('Location: perfil.php');
            exit;
        }
    }
    
    // Obtener datos del usuario
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = :id");
    $stmt->bindParam(':id', $user_id);
    $stmt->execute();
    $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Obtener permisos del usuario - MODIFICADO para agrupar por categorías
    $stmt = $conn->prepare("
        SELECT p.* 
        FROM permisos p
        JOIN user_permisos up ON p.id = up.permiso_id
        WHERE up.user_id = :user_id
        ORDER BY 
            CASE 
                WHEN p.nombre LIKE '%rendicion%' THEN 1
                WHEN p.nombre LIKE '%orden_compra%' THEN 2
                WHEN p.nombre LIKE '%maquina%' THEN 3
                WHEN p.nombre LIKE '%repuesto%' THEN 4
                WHEN p.nombre LIKE '%devolucion%' THEN 5
                ELSE 6
            END, 
            p.id
    ");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    $permisos_usuario = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Agrupar permisos por categoría - NUEVO
    $permisos_agrupados = [
        'rendiciones' => [],
        'ordenes_compra' => [],
        'maquinas' => [],
        'repuestos' => [],
        'devoluciones' => [],
        'generales' => []
    ];
    
    foreach ($permisos_usuario as $permiso) {
        if (strpos($permiso['nombre'], 'rendicion') !== false) {
            $permisos_agrupados['rendiciones'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'orden_compra') !== false) {
            $permisos_agrupados['ordenes_compra'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'maquina') !== false) {
            $permisos_agrupados['maquinas'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'repuesto') !== false) {
            $permisos_agrupados['repuestos'][] = $permiso;
        } elseif (strpos($permiso['nombre'], 'devolucion') !== false) {
            $permisos_agrupados['devoluciones'][] = $permiso;
        } else {
            $permisos_agrupados['generales'][] = $permiso;
        }
    }
    
    // Obtener actividad reciente del usuario
    $stmt = $conn->prepare("
        SELECT *
        FROM historial_cambios
        WHERE user_id = :user_id
        ORDER BY fecha_hora DESC
        LIMIT 10
    ");
    $stmt->bindParam(':user_id', $user_id);
    $stmt->execute();
    $actividad = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Incluir el header
    require_once 'includes/header.php';
?>

<style>
    /* Estilos específicos para la página de perfil */
    .perfil-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .perfil-card {
        background-color: white;
        border-radius: 8px;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        padding: 20px;
    }

    .perfil-seccion-titulo {
        margin: 0 0 15px;
        font-size: 18px;
        border-bottom: 1px solid #eee;
        padding-bottom: 10px;
    }
    
    /* Estilos para la visualización de permisos - MODIFICADO */
    .permisos-container {
        background-color: #f8f9fa;
        padding: 15px;
        border-radius: 4px;
        margin-bottom: 20px;
    }
    
    .permiso-section {
        margin-bottom: 15px;
        border-left: 3px solid #1e88e5;
        padding-left: 15px;
    }
    
    .section-title {
        font-size: 16px;
        font-weight: 600;
        margin-bottom: 10px;
        color: #333;
    }
    
    .permiso-list {
        list-style-type: none;
        padding-left: 15px;
        margin: 0;
    }
    
    .permiso-list li {
        margin-bottom: 5px;
        position: relative;
    }
    
    .permiso-list li:before {
        content: "•";
        color: #1e88e5;
        position: absolute;
        left: -15px;
    }

    .perfil-actividad-lista {
        list-style-type: none;
        padding-left: 0;
        margin: 0;
        max-height: 300px;
        overflow-y: auto;
    }

    .perfil-actividad-item {
        margin-bottom: 10px;
        padding-bottom: 10px;
        border-bottom: 1px solid #eee;
    }

    .perfil-actividad-tipo {
        font-weight: 500;
    }

    .perfil-actividad-descripcion {
        display: block;
        font-size: 12px;
        margin-top: 3px;
    }

    .perfil-actividad-fecha {
        display: block;
        font-size: 12px;
        color: #666;
        margin-top: 3px;
    }

    @media (max-width: 768px) {
        .perfil-grid {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="card">
    <div class="panel-header">
        <h2 class="panel-title">Mi Perfil</h2>
    </div>
    
    <div class="perfil-grid">
        <!-- Información Personal -->
        <div class="perfil-card">
            <h3 class="perfil-seccion-titulo">Información Personal</h3>
            
            <form action="perfil.php" method="POST">
                <div class="form-group">
                    <label for="nombre" class="form-label">Nombre <span class="text-danger">*</span></label>
                    <input type="text" id="nombre" name="nombre" class="form-control" required maxlength="100" value="<?php echo htmlspecialchars($usuario['nombre']); ?>">
                </div>
                
                <div class="form-group">
                    <label for="apellido" class="form-label">Apellido <span class="text-danger">*</span></label>
                    <input type="text" id="apellido" name="apellido" class="form-control" required maxlength="100" value="<?php echo htmlspecialchars($usuario['apellido']); ?>">
                </div>
                
                <div class="form-group">
                    <label for="rut" class="form-label">RUT</label>
                    <input type="text" id="rut" class="form-control" value="<?php echo htmlspecialchars($usuario['rut']); ?>" readonly>
                    <small class="form-text text-muted">El RUT no se puede modificar</small>
                </div>
                
                <div class="form-group">
                    <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                    <input type="email" id="email" name="email" class="form-control" required maxlength="100" value="<?php echo htmlspecialchars($usuario['email']); ?>">
                </div>
                
                <div class="form-group">
                    <label for="telefono" class="form-label">Teléfono</label>
                    <input type="text" id="telefono" name="telefono" class="form-control" maxlength="20" value="<?php echo htmlspecialchars($usuario['telefono'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label for="cargo" class="form-label">Cargo</label>
                    <input type="text" id="cargo" class="form-control" value="<?php echo htmlspecialchars($usuario['cargo']); ?>" readonly>
                </div>
                
                <div class="form-group">
                    <label for="rol" class="form-label">Rol</label>
                    <input type="text" id="rol" class="form-control" value="<?php 
                        $roles = [
                            'super_admin' => 'Super Administrador',
                            'admin' => 'Administrador',
                            'usuario' => 'Usuario'
                        ];
                        echo $roles[$usuario['rol']] ?? $usuario['rol']; 
                    ?>" readonly>
                </div>
                
                <div class="form-group">
                    <div class="form-check">
                        <input type="checkbox" class="form-check-input" id="cambiar_password" name="cambiar_password" value="1">
                        <label class="form-check-label" for="cambiar_password">
                            Cambiar contraseña
                        </label>
                    </div>
                    
                    <div id="password_fields" style="display: none;">
                        <div class="form-group">
                            <label for="password_actual" class="form-label">Contraseña Actual <span class="text-danger">*</span></label>
                            <input type="password" id="password_actual" name="password_actual" class="form-control">
                        </div>
                        
                        <div class="form-group">
                            <label for="password_nuevo" class="form-label">Nueva Contraseña <span class="text-danger">*</span></label>
                            <input type="password" id="password_nuevo" name="password_nuevo" class="form-control" minlength="6">
                            <small class="form-text text-muted">Mínimo 6 caracteres</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="password_confirmar" class="form-label">Confirmar Contraseña <span class="text-danger">*</span></label>
                            <input type="password" id="password_confirmar" name="password_confirmar" class="form-control" minlength="6">
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons">save</i> Guardar Cambios
                    </button>
                </div>
            </form>
        </div>
        
        <!-- Información Adicional -->
        <div class="perfil-card">
            <h3 class="perfil-seccion-titulo">Mis Permisos</h3>
            
            <?php if (empty($permisos_usuario)): ?>
            <p>No tiene permisos asignados.</p>
            <?php else: ?>
            <div class="permisos-container">
                <?php if (!empty($permisos_agrupados['rendiciones'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Rendiciones de Gastos</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['rendiciones'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($permisos_agrupados['ordenes_compra'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Órdenes de Compra</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['ordenes_compra'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($permisos_agrupados['devoluciones'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Devoluciones</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['devoluciones'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($permisos_agrupados['maquinas'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Máquinas</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['maquinas'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($permisos_agrupados['repuestos'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Repuestos y Solicitudes</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['repuestos'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <?php if (!empty($permisos_agrupados['generales'])): ?>
                <div class="permiso-section">
                    <h4 class="section-title">Permisos Generales</h4>
                    <ul class="permiso-list">
                        <?php foreach ($permisos_agrupados['generales'] as $permiso): ?>
                        <li><?php echo htmlspecialchars($permiso['descripcion']); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <h3 class="perfil-seccion-titulo">Actividad Reciente</h3>
            
            <?php if (empty($actividad)): ?>
            <p>No hay actividad reciente registrada.</p>
            <?php else: ?>
            <div style="background-color: #f8f9fa; padding: 15px; border-radius: 4px; max-height: 300px; overflow-y: auto;">
                <ul class="perfil-actividad-lista">
                    <?php foreach ($actividad as $registro): ?>
                    <li class="perfil-actividad-item">
                        <span class="perfil-actividad-tipo"><?php echo ucfirst($registro['tipo']); ?></span>
                        <span class="perfil-actividad-descripcion"><?php echo htmlspecialchars($registro['descripcion']); ?></span>
                        <span class="perfil-actividad-fecha"><?php echo date('d/m/Y H:i', strtotime($registro['fecha_hora'])); ?></span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <h3 class="perfil-seccion-titulo">Información de Cuenta</h3>
            
            <div style="background-color: #f8f9fa; padding: 15px; border-radius: 4px;">
                <div style="margin-bottom: 10px;">
                    <p style="margin: 0; font-size: 14px; color: #666;">Fecha de Creación</p>
                    <p style="margin: 5px 0 0; font-weight: 500;"><?php echo date('d/m/Y H:i', strtotime($usuario['created_at'])); ?></p>
                </div>
                <div>
                    <p style="margin: 0; font-size: 14px; color: #666;">Última Actualización</p>
                    <p style="margin: 5px 0 0; font-weight: 500;"><?php echo date('d/m/Y H:i', strtotime($usuario['updated_at'])); ?></p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Toggle campos de contraseña
        const cambiarPasswordCheckbox = document.getElementById('cambiar_password');
        const passwordFields = document.getElementById('password_fields');
        
        cambiarPasswordCheckbox.addEventListener('change', function() {
            passwordFields.style.display = this.checked ? 'block' : 'none';
            
            // Limpiar campos si se desmarca
            if (!this.checked) {
                document.getElementById('password_actual').value = '';
                document.getElementById('password_nuevo').value = '';
                document.getElementById('password_confirmar').value = '';
            }
        });
    });
</script>

<?php
// Incluir el footer
require_once 'includes/footer.php';
} catch (PDOException $e) {
    echo '<div class="alert alert-danger">Error de conexión a la base de datos: ' . $e->getMessage() . '</div>';
}
?>